"use client";
import { useAuth } from '@/features/auth/useAuth';
import { useCallback, useEffect, useState, useRef } from 'react';
import { getSupabaseClient } from '@/config/supabase';
import type { CartItem } from '@/types/cart';
import type { Tables } from '@/types/supabase';
import toast from 'react-hot-toast';

export function useCart() {
  const { session, loading: authLoading } = useAuth();
  const [cart, setCart] = useState<CartItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const channelRef = useRef<any>(null);

  // Load cart data for this user
  const fetchCart = useCallback(async () => {
    if (!session) {
      setCart([]);
      setLoading(false);
      return;
    }
    setLoading(true);
    setError(null);
    const supabase = getSupabaseClient();

    try {
      // First get cart items with products in a single query
      const { data: cartData, error: cartError } = await supabase
        .from('cart')
        .select(`
          *,
          product:content_items (*)
        `)
        .eq('user_id', session.user.id);
        
      if (cartError) throw cartError;
      
      // Transform the data to match CartItem type
      const transformedCart = cartData?.map(item => ({
        ...item,
        product: item.product as Tables['content_items']
      })) || [];
      
      setCart(transformedCart as CartItem[]);
    } catch (err) {
      console.error('Error fetching cart:', err);
      setError('Failed to load cart.');
      setCart([]);
    }
    
    setLoading(false);
  }, [session]);

  // Set up real-time subscription
  useEffect(() => {
    if (authLoading) return;
    fetchCart();
    
    // Clean up previous subscription if it exists
    if (channelRef.current) {
      const supabase = getSupabaseClient();
      supabase.removeChannel(channelRef.current);
    }
    
    // Set up new subscription
    const supabase = getSupabaseClient();
    if (!session) return;
    
    const channel = supabase
      .channel('cart-changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'cart',
          filter: `user_id=eq.${session.user.id}`
        },
        async (payload) => {
          // Optimistically update the cart based on the event
          if (payload.eventType === 'INSERT') {
            const { data: product } = await supabase
              .from('content_items')
              .select('*')
              .eq('id', payload.new.product_id)
              .single();
              
            if (product) {
              const newItem: CartItem = {
                id: payload.new.id,
                user_id: payload.new.user_id,
                product_id: payload.new.product_id,
                quantity: payload.new.quantity,
                created_at: payload.new.created_at,
                product
              };
              setCart(prev => [...prev, newItem]);
            }
          } else if (payload.eventType === 'DELETE') {
            setCart(prev => prev.filter(item => item.id !== payload.old.id));
          } else if (payload.eventType === 'UPDATE') {
            setCart(prev => 
              prev.map(item => 
                item.id === payload.new.id 
                  ? { ...item, quantity: payload.new.quantity }
                  : item
              )
            );
          }
        }
      )
      .subscribe();

    channelRef.current = channel;

    return () => {
      supabase.removeChannel(channel);
    };
  }, [session, authLoading, fetchCart]);

  // Add item to cart or update quantity
  const addToCart = async (product: Tables['content_items'], quantity = 1) => {
    if (!session) {
      toast.error('You must be signed in to add to cart.');
      return;
    }
    setLoading(true);
    try {
      const supabase = getSupabaseClient();
      
      // Check if item already exists
      const { data: existing, error: findError } = await supabase
        .from('cart')
        .select('*')
        .eq('user_id', session.user.id)
        .eq('product_id', product.id)
        .single();

      if (findError && findError.code !== 'PGRST116') {
        throw findError;
      }

      // Optimistically update the UI
      if (existing) {
        const newQuantity = existing.quantity + quantity;
        setCart(prev => 
          prev.map(item => 
            item.id === existing.id 
              ? { ...item, quantity: newQuantity }
              : item
          )
        );
        
        // Update quantity (increment)
        const { error } = await supabase
          .from('cart')
          .update({ quantity: newQuantity })
          .eq('id', existing.id);
        if (error) throw error;
        toast.success('Cart updated!');
      } else {
        // Create temporary item for optimistic update
        const tempItem: CartItem = {
          id: 'temp-' + Date.now(),
          user_id: session.user.id,
          product_id: product.id,
          quantity,
          created_at: new Date().toISOString(),
          product
        };
        setCart(prev => [...prev, tempItem]);

        // Insert new row
        const { error, data } = await supabase
          .from('cart')
          .insert({
            user_id: session.user.id,
            product_id: product.id,
            quantity
          })
          .select()
          .single();
        
        if (error) throw error;
        
        // Update temporary item with real ID
        if (data) {
          setCart(prev => 
            prev.map(item => 
              item.id === tempItem.id
                ? { ...item, id: data.id }
                : item
            )
          );
        }
        
        toast.success('Added to cart!');
      }
    } catch (err: any) {
      // Revert optimistic update
      await fetchCart();
      setError('Unable to add to cart.');
      toast.error(err?.message || 'Unable to add to cart.');
    } finally {
      setLoading(false);
    }
  };

  // Update quantity with optimistic update
  const updateQuantity = async (cartId: string, quantity: number) => {
    setLoading(true);
    try {
      // Optimistically update the UI
      setCart(prev => 
        prev.map(item => 
          item.id === cartId 
            ? { ...item, quantity }
            : item
        )
      );

      const supabase = getSupabaseClient();
      const { error } = await supabase
        .from('cart')
        .update({ quantity })
        .eq('id', cartId);
      if (error) throw error;
      toast.success('Quantity updated!');
    } catch (err: any) {
      // Revert optimistic update on error
      await fetchCart();
      setError('Unable to update cart.');
      toast.error(err?.message || 'Unable to update cart.');
    } finally {
      setLoading(false);
    }
  };

  // Remove item with optimistic update
  const removeFromCart = async (cartId: string) => {
    setLoading(true);
    try {
      // Optimistically update the UI
      setCart(prev => prev.filter(item => item.id !== cartId));

      const supabase = getSupabaseClient();
      const { error } = await supabase
        .from('cart')
        .delete()
        .eq('id', cartId);
      if (error) throw error;
      toast.success('Removed from cart.');
    } catch (err: any) {
      // Revert optimistic update on error
      await fetchCart();
      setError('Unable to remove item.');
      toast.error(err?.message || 'Unable to remove item.');
    } finally {
      setLoading(false);
    }
  };

  // Clear cart with optimistic update
  const clearCart = async () => {
    if (!session) return;
    setLoading(true);
    try {
      // Optimistically update the UI
      setCart([]);

      const supabase = getSupabaseClient();
      const { error } = await supabase
        .from('cart')
        .delete()
        .eq('user_id', session.user.id);
      if (error) throw error;
      toast.success('Cart cleared.');
    } catch (err: any) {
      // Revert optimistic update on error
      await fetchCart();
      setError('Unable to clear cart.');
      toast.error(err?.message || 'Unable to clear cart.');
    } finally {
      setLoading(false);
    }
  };

  // Subtotal calculation
  const subtotal = cart.reduce((sum, item) => sum + (item.product?.price ?? 0) * item.quantity, 0);

  return {
    cart,
    loading,
    error,
    addToCart,
    updateQuantity,
    removeFromCart,
    clearCart,
    fetchCart,
    subtotal
  };
}
