import { create } from 'zustand';
import { persist, createJSONStorage, StateStorage } from 'zustand/middleware';
import { CartItem, Product } from '@/types';

interface CartStore {
  items: CartItem[];
  addItem: (product: Product) => void;
  removeItem: (productId: number) => void;
  updateQuantity: (productId: number, quantity: number) => void;
  clearCart: () => void;
  total: number;
  itemCount: number;
}

// Define the type for previous state versions
type CartState = {
  items: CartItem[];
  total: number;
  itemCount?: number; // Optional for backward compatibility
}

export const useCart = create<CartStore>()(
  persist(
    (set, get) => ({
      items: [],
      total: 0,
      itemCount: 0,

      addItem: (product: Product) => {
        try {
          const currentItems = get().items;
          const existingItem = currentItems.find(item => item.id === product.id);

          if (existingItem) {
            const updatedItems = currentItems.map(item =>
              item.id === product.id
                ? { ...item, quantity: item.quantity + 1 }
                : item
            );
            set((state) => ({
              items: updatedItems,
              total: state.total + product.price,
              itemCount: state.itemCount + 1
            }));
          } else {
            set((state) => ({
              items: [...currentItems, { ...product, quantity: 1 }],
              total: state.total + product.price,
              itemCount: state.itemCount + 1
            }));
          }
        } catch (error) {
          console.error('Error adding item to cart:', error);
          throw new Error('Failed to add item to cart');
        }
      },

      removeItem: (productId: number) => {
        try {
          const currentItems = get().items;
          const itemToRemove = currentItems.find(item => item.id === productId);
          
          if (itemToRemove) {
            set((state) => ({
              items: currentItems.filter(item => item.id !== productId),
              total: state.total - (itemToRemove.price * itemToRemove.quantity),
              itemCount: state.itemCount - itemToRemove.quantity
            }));
          }
        } catch (error) {
          console.error('Error removing item from cart:', error);
          throw new Error('Failed to remove item from cart');
        }
      },

      updateQuantity: (productId: number, quantity: number) => {
        try {
          const currentItems = get().items;
          const item = currentItems.find(item => item.id === productId);
          
          if (item) {
            if (quantity < 0) return;
            
            const quantityDiff = quantity - item.quantity;
            const updatedItems = currentItems.map(item =>
              item.id === productId
                ? { ...item, quantity }
                : item
            );
            
            set((state) => ({
              items: updatedItems,
              total: state.total + (item.price * quantityDiff),
              itemCount: state.itemCount + quantityDiff
            }));

            if (quantity === 0) {
              get().removeItem(productId);
            }
          }
        } catch (error) {
          console.error('Error updating cart quantity:', error);
          throw new Error('Failed to update cart quantity');
        }
      },

      clearCart: () => {
        try {
          set({
            items: [],
            total: 0,
            itemCount: 0
          });
        } catch (error) {
          console.error('Error clearing cart:', error);
          throw new Error('Failed to clear cart');
        }
      }
    }),
    {
      name: 'cart-storage',
      storage: createJSONStorage(() => localStorage),
      version: 1,
      migrate: (persistedState: any, version: number): CartState => {
        if (version === 0) {
          // If we have old state without itemCount, calculate it
          const state = persistedState as CartState;
          return {
            items: state.items || [],
            total: state.total || 0,
            itemCount: state.items?.reduce((count, item) => count + item.quantity, 0) || 0
          };
        }
        return persistedState as CartState;
      }
    }
  )
); 