
import time, random, requests
import pandas as pd

DEFAULT_TIMEOUT = 25

def _ua(source="vietmarket"):
    return {
        "User-Agent": f"{source}/1.0 (+https://example.local)",
        "Accept": "application/json, text/plain, */*",
        "Origin": "https://example.local",
        "Referer": "https://example.local/",
    }

def send_request(url, method="GET", headers=None, params=None, payload=None,
                 retries=2, backoff=(0.6, 1.2), timeout=DEFAULT_TIMEOUT):
    h = _ua()
    if headers:
        h.update(headers)
    for attempt in range(retries + 1):
        try:
            if method.upper() == "GET":
                r = requests.get(url, headers=h, params=params, timeout=timeout)
            else:
                r = requests.post(url, headers=h, params=params, json=payload, timeout=timeout)
            r.raise_for_status()
            if "application/json" in r.headers.get("Content-Type", ""):
                return r.json()
            return r.text
        except Exception:
            if attempt >= retries:
                raise
            time.sleep(random.uniform(*backoff))

def add_all_ta_features(
    df,
    open: str = "Open",
    high: str = "High",
    low: str = "Low",
    close: str = "Close",
    volume: str = "Volume",
    fillna: bool = True,
):
    """
    Thêm toàn bộ technical indicators từ thư viện `ta` vào DataFrame.
    Giữ nguyên signature giống `ta.add_all_ta_features` để sử dụng y hệt.

    Parameters
    ----------
    df : pandas.DataFrame
        DataFrame phải có các cột giá/khối lượng tương ứng.
    open,high,low,close,volume : str
        Tên cột trong df.
    fillna : bool
        Nếu True, sẽ điền các giá trị NaN theo mặc định của thư viện `ta`.

    Returns
    -------
    pandas.DataFrame
        DataFrame đầu vào + các cột TA features.
    """
    try:
        from ta import add_all_ta_features as _ta_add_all_ta_features
        from ta.utils import dropna as _ta_dropna
    except Exception as e:
        raise ImportError(
            "Thiếu thư viện 'ta'. Hãy cài: pip install ta"
        ) from e

    # Làm sạch NaN theo chuẩn của 'ta'
    _df = _ta_dropna(df.copy())

    # Gọi trực tiếp hàm gốc
    return _ta_add_all_ta_features(
        _df,
        open=open,
        high=high,
        low=low,
        close=close,
        volume=volume,
        fillna=fillna,
    )
