"""
Module containing data processing utilities for writing HDF5 files in openCFS format
"""

import numpy as np
import pytest

from pyCFS.data import io, v_def
from pyCFS.data.io.cfs_types import cfs_element_type, cfs_result_type, cfs_analysis_type


@pytest.fixture
def dummy_CFSMeshData_obj():
    """
    Create simple test mesh with 1 Volume and 2 Surface region / group
    """
    # Create Mesh
    mesh_data = io.CFSMeshData(verbosity=v_def.all)
    mesh_data.MeshInfo.Dimension = 3
    mesh_data.Coordinates = np.array(
        [
            [0, 0, 0],
            [0, 0, 1],
            [0, 1, 0],
            [0, 1, 1],
            [1, 0, 0],
            [1, 0, 1],
            [1, 1, 0],
            [1, 1, 1],
        ]
    )
    mesh_data.MeshInfo.NumNodes = 8
    mesh_data.Connectivity = np.array(
        [
            [1, 2, 4, 3, 5, 6, 8, 7],
            [1, 2, 3, 0, 0, 0, 0, 0],
            [2, 4, 3, 0, 0, 0, 0, 0],
            [5, 6, 8, 7, 0, 0, 0, 0],
        ]
    )
    mesh_data.Types = np.array(
        [
            cfs_element_type.HEXA8,
            cfs_element_type.TRIA3,
            cfs_element_type.TRIA3,
            cfs_element_type.QUAD4,
        ]
    ).reshape(-1, 1)
    mesh_data.MeshInfo.NumElems = 4
    mesh_data.MeshInfo.Num2DElems = 3
    mesh_data.MeshInfo.Num3DElems = 1
    mesh_data.MeshInfo.Num_HEXA8 = 1
    mesh_data.MeshInfo.Num_TRIA3 = 2
    mesh_data.MeshInfo.Num_QUAD4 = 1
    # Create Regions
    reg_info_vol = io.CFSRegData(name="Vol")
    reg_info_vol.Dimension = 3
    reg_info_vol.Nodes = np.array([1, 2, 3, 4, 5, 6, 7, 8])
    reg_info_vol.Elements = np.array([1])
    reg_info_surf = io.CFSRegData(name="Surf1")
    reg_info_surf.Dimension = 2
    reg_info_surf.Nodes = np.array([1, 2, 3, 4])
    reg_info_surf.Elements = np.array([2, 3])
    reg_info_surf_group = io.CFSRegData(
        name="Surf2",
        is_group=True,
        dimension=2,
        nodes=np.array([5, 6, 7, 8]),
        elements=np.array([4]),
    )
    reg_info = [reg_info_vol, reg_info_surf, reg_info_surf_group]

    mesh_data.Regions = reg_info

    return mesh_data


@pytest.fixture
def dummy_CFSResultContainer_obj():
    """
    Create simple test containing scalar as well as complex vector data on multiple steps. Compatible with dummy_mesh
    """
    # Create Result
    step_value_list = np.array([(i + 1) / 5 for i in range(5)])
    data_test = np.array([1.0, 2, 3, 4, 5, 6, 7, 8]).reshape((-1, 1))
    data_test = np.array([data_test * (i + (1 + 1j)) for i in range(5)])
    data3_test_stp = np.array([[1.0, 2, 3], [4, 5, 6]])
    data3_test = np.array([data3_test_stp] * 5)

    result_data = io.CFSResultContainer(analysis_type=cfs_analysis_type.HARMONIC, verbosity=v_def.all)
    result_data.add_data(
        data_test,
        step_values=step_value_list,
        quantity="quantity",
        region="Vol",
        restype=cfs_result_type.NODE,
        dim_names=["-"],
    )
    result_data.add_data(
        data3_test,
        step_values=step_value_list,
        quantity="quantity3",
        region="Surf1",
        restype=cfs_result_type.ELEMENT,
        dim_names=["x", "y", "z"],
        is_complex=False,
    )

    return result_data


@pytest.fixture
def dummy_CFSResultContainer_history_obj() -> io.CFSResultArray:
    """
    Create simple test containing history data. Compatible with dummy_mesh
    """
    # Create Result
    step_value_list = np.array([(i + 1) / 5 for i in range(5)])
    data_test = np.array([1.234 * (i + (1 + 1j)) for i in range(5)])

    result_array = io.CFSResultArray(
        input_array=data_test,
        quantity="quantity",
        region="Vol",
        step_values=step_value_list,
        dim_names=["-"],
        res_type=cfs_result_type.REGION_AVERAGE,
        is_complex=True,
        multi_step_id=1,
        analysis_type=cfs_analysis_type.HARMONIC,
    )

    return io.CFSResultContainer(data=[result_array])


@pytest.fixture
def dummy_CFSMeshData_linear_elements():
    """
    Create test mesh containing the following linear element types:
    - 2D: TRIA3, QUAD4
    - 3D: TET4, PYRA5, WEDGE6, HEXA8

    Returns
    -------
    CFSMeshData
        Mesh data object
    """

    coordinates = np.array(
        [
            [2.0, 1.0, 0.0],
            [2.0, 1.0, 1.0],
            [2.0, 0.0, 1.0],
            [2.0, 0.0, 0.0],
            [1.0, 0.0, 0.0],
            [1.0, 0.0, 1.0],
            [1.0, 1.0, 1.0],
            [1.0, 1.0, 0.0],
            [1.0, 0.0, 2.0],
            [0.0, 0.0, 2.0],
            [0.0, 0.0, 1.0],
            [0.0, 1.0, 2.0],
            [0.0, 1.0, 1.0],
            [1.0, 1.0, 2.0],
            [1.0, 1.0, 2.0],
            [1.0, 0.0, 2.0],
            [1.3054697513580322, 0.0, 1.488473653793335],
            [1.415229082107544, 0.0, 1.3067442178726196],
            [1.415229082107544, 1.0, 1.3067442178726196],
            [1.3054697513580322, 1.0, 1.488473653793335],
            [1.0, 1.0, 1.3333333730697632],
            [1.0, 1.0, 1.6666666269302368],
            [1.5, 1.0, 1.5],
            [1.5, 0.0, 1.5],
            [1.0, 0.0, 1.3333333730697632],
            [1.0, 0.0, 1.6666666269302368],
            [0.5, 1.0, 0.5],
            [0.5, 0.0, 0.5],
            [0.8500000238418579, 0.5, 0.5],
            [2.0, 1.0, 0.5],
            [2.0, 0.5, 0.0],
            [1.5, 1.0, 0.0],
            [2.0, 0.5, 1.0],
            [1.5, 1.0, 1.0],
            [2.0, 0.0, 0.5],
            [1.5, 0.0, 1.0],
            [1.5, 0.0, 0.0],
            [1.0, 0.0, 0.5],
            [1.0, 0.5, 0.0],
            [1.0, 0.5, 1.0],
            [1.0, 1.0, 0.5],
            [1.0, 0.0, 1.5],
            [0.5, 0.0, 1.5],
            [0.5, 0.0, 1.0],
            [0.5, 0.5, 1.0],
            [1.0, 0.5, 1.5],
            [0.5, 0.5, 1.5],
            [0.5, 1.0, 1.5],
            [0.5, 1.0, 1.0],
            [1.0, 1.0, 1.5],
            [0.5, 0.0, 2.0],
            [1.0, 0.5, 2.0],
            [0.0, 0.0, 1.5],
            [0.0, 0.5, 2.0],
            [0.0, 0.5, 1.5],
            [0.5, 0.5, 2.0],
            [0.0, 0.5, 1.0],
            [0.0, 1.0, 1.5],
            [0.5, 1.0, 2.0],
            [1.707614541053772, 1.0, 1.1533721685409546],
            [1.75, 1.0, 1.25],
            [1.707614541053772, 0.0, 1.1533721685409546],
            [1.75, 0.0, 1.25],
            [1.0, 0.0, 1.1666666269302368],
            [1.0, 1.0, 1.1666666269302368],
            [1.0, 0.5, 2.0],
            [1.0, 1.0, 1.8333333730697632],
            [1.25, 1.0, 1.75],
            [1.25, 0.0, 1.75],
            [1.0, 0.0, 1.8333333730697632],
            [1.360349416732788, 0.0, 1.3976088762283325],
            [1.3054697513580322, 0.5, 1.488473653793335],
            [1.4027348756790161, 0.0, 1.4942368268966675],
            [1.1527348756790161, 0.0, 1.5775701999664307],
            [1.415229082107544, 0.5, 1.3067442178726196],
            [1.207614541053772, 0.0, 1.3200387954711914],
            [1.360349416732788, 1.0, 1.3976088762283325],
            [1.207614541053772, 1.0, 1.3200387954711914],
            [1.1527348756790161, 1.0, 1.5775701999664307],
            [1.4027348756790161, 1.0, 1.4942368268966675],
            [1.0, 1.0, 1.5],
            [1.0, 0.5, 1.3333333730697632],
            [1.0, 0.5, 1.6666666269302368],
            [1.5, 0.5, 1.5],
            [1.0, 0.0, 1.5],
            [0.75, 0.0, 0.25],
            [0.925000011920929, 0.25, 0.25],
            [0.75, 0.5, 0.75],
            [0.75, 0.0, 0.75],
            [0.925000011920929, 0.25, 0.75],
            [0.75, 1.0, 0.75],
            [0.925000011920929, 0.75, 0.75],
            [0.75, 1.0, 0.25],
            [0.75, 0.5, 0.25],
            [0.925000011920929, 0.75, 0.25],
            [0.25, 0.5, 0.75],
            [0.25, 0.0, 0.75],
            [0.25, 1.0, 0.75],
            [0.5, 0.5, 0.5],
            [0.675000011920929, 0.75, 0.5],
            [0.675000011920929, 0.25, 0.5],
        ]
    )

    connectivity = np.array(
        [
            [7, 14, 10, 9, 50, 56, 47, 46, 52, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 7, 10, 9, 40, 47, 43, 42, 46, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 10, 14, 12, 47, 56, 50, 48, 54, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 7, 13, 10, 40, 49, 45, 43, 47, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 12, 13, 10, 48, 58, 49, 47, 54, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 13, 11, 10, 45, 57, 44, 43, 55, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [1, 4, 3, 2, 8, 5, 6, 7, 31, 35, 33, 30, 39, 38, 40, 41, 32, 37, 36, 34],
            [17, 18, 25, 26, 20, 19, 21, 22, 71, 76, 85, 74, 77, 78, 81, 79, 72, 75, 82, 83],
            [3, 6, 25, 18, 2, 7, 21, 19, 36, 64, 76, 62, 34, 65, 78, 60, 33, 40, 82, 75],
            [3, 18, 17, 24, 2, 19, 20, 23, 62, 71, 73, 63, 60, 77, 80, 61, 33, 75, 72, 84],
            [15, 22, 20, 23, 16, 26, 17, 24, 67, 79, 80, 68, 70, 74, 73, 69, 66, 83, 72, 84],
            [6, 13, 27, 11, 45, 98, 88, 44, 57, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 27, 28, 11, 88, 99, 89, 44, 96, 97, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 27, 13, 7, 88, 98, 45, 40, 91, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 27, 29, 28, 88, 100, 90, 89, 99, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [8, 29, 27, 28, 95, 100, 93, 94, 101, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 29, 27, 7, 90, 100, 88, 40, 92, 91, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 27, 8, 29, 91, 93, 41, 92, 100, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 28, 29, 5, 89, 101, 90, 38, 86, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [5, 8, 28, 29, 39, 94, 86, 87, 95, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [5, 6, 7, 8, 29, 38, 40, 41, 39, 87, 90, 92, 95, 0, 0, 0, 0, 0, 0, 0],
            [7, 13, 12, 49, 58, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 12, 14, 48, 59, 50, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [10, 14, 12, 56, 59, 54, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [9, 14, 10, 52, 56, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 10, 11, 43, 53, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 9, 10, 42, 51, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [11, 13, 6, 57, 45, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 13, 7, 45, 49, 40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [14, 9, 7, 52, 46, 50, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 9, 6, 46, 42, 40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [10, 13, 11, 55, 57, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [10, 12, 13, 54, 58, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [5, 4, 3, 6, 37, 35, 36, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [8, 7, 2, 1, 41, 34, 30, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 6, 3, 2, 40, 36, 33, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [1, 4, 5, 8, 31, 37, 39, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [2, 3, 4, 1, 33, 35, 31, 30, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [5, 6, 7, 8, 38, 40, 41, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [16, 24, 23, 15, 69, 84, 68, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [24, 3, 2, 23, 63, 33, 61, 84, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [15, 22, 26, 16, 67, 83, 70, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [22, 21, 25, 26, 81, 82, 85, 83, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [21, 7, 6, 25, 65, 40, 64, 82, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [19, 21, 22, 20, 78, 81, 79, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [19, 2, 7, 21, 60, 34, 65, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [20, 23, 2, 19, 80, 61, 60, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [22, 15, 23, 20, 67, 68, 80, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [26, 25, 18, 17, 85, 76, 71, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 3, 18, 25, 36, 62, 76, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [3, 24, 17, 18, 63, 73, 71, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [24, 16, 26, 17, 69, 70, 74, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [28, 11, 27, 97, 96, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [5, 28, 8, 86, 94, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [27, 11, 13, 96, 57, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [8, 28, 27, 94, 99, 93, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 8, 27, 41, 93, 91, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [7, 27, 13, 91, 98, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 11, 28, 44, 97, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            [6, 28, 5, 89, 86, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]
    )

    el_types = np.array(
        [
            10,
            10,
            10,
            10,
            10,
            10,
            12,
            12,
            12,
            12,
            12,
            10,
            10,
            10,
            10,
            10,
            10,
            10,
            10,
            10,
            15,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            7,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
            5,
        ]
    )

    regions = [
        io.CFSRegData(
            name="S_PYRAMID",
            nodes=np.array(
                [
                    5,
                    6,
                    7,
                    8,
                    11,
                    13,
                    27,
                    28,
                    38,
                    39,
                    41,
                    44,
                    49,
                    57,
                    86,
                    89,
                    91,
                    93,
                    94,
                    96,
                    97,
                    98,
                    99,
                ]
            ),
            elements=np.array(
                [
                    53,
                    54,
                    55,
                    56,
                    57,
                    58,
                    59,
                    60,
                ]
            ),
            dimension=2,
        ),
        io.CFSRegData(
            name="S_QUAD",
            nodes=np.array(
                [
                    1,
                    2,
                    3,
                    4,
                    5,
                    6,
                    7,
                    8,
                    30,
                    31,
                    32,
                    33,
                    34,
                    35,
                    36,
                    37,
                    38,
                    39,
                    40,
                    41,
                ]
            ),
            elements=np.array(
                [
                    34,
                    35,
                    36,
                    37,
                    38,
                    39,
                ]
            ),
            dimension=2,
        ),
        io.CFSRegData(
            name="S_TET",
            nodes=np.array(
                [
                    6,
                    7,
                    9,
                    10,
                    11,
                    12,
                    13,
                    14,
                    40,
                    42,
                    43,
                    44,
                    45,
                    46,
                    48,
                    49,
                    50,
                    51,
                    52,
                    53,
                    54,
                    55,
                    56,
                    57,
                    58,
                    59,
                ]
            ),
            elements=np.array(
                [
                    22,
                    23,
                    24,
                    25,
                    26,
                    27,
                    28,
                    29,
                    30,
                    31,
                    32,
                    33,
                ]
            ),
            dimension=2,
        ),
        io.CFSRegData(
            name="S_WEDGE",
            nodes=np.array(
                [
                    2,
                    3,
                    6,
                    7,
                    15,
                    16,
                    17,
                    18,
                    19,
                    20,
                    21,
                    22,
                    23,
                    24,
                    25,
                    26,
                    33,
                    34,
                    36,
                    40,
                    60,
                    61,
                    62,
                    63,
                    64,
                    65,
                    66,
                    67,
                    68,
                    69,
                    70,
                    71,
                    73,
                    74,
                    76,
                    77,
                    78,
                    79,
                    80,
                    81,
                    82,
                    83,
                    84,
                    85,
                ]
            ),
            elements=np.array(
                [
                    40,
                    41,
                    42,
                    43,
                    44,
                    45,
                    46,
                    47,
                    48,
                    49,
                    50,
                    51,
                    52,
                ]
            ),
            dimension=2,
        ),
        io.CFSRegData(
            name="V_PYRAMID",
            nodes=np.array(
                [
                    5,
                    6,
                    7,
                    8,
                    11,
                    13,
                    27,
                    28,
                    29,
                    38,
                    39,
                    40,
                    41,
                    44,
                    45,
                    49,
                    57,
                    86,
                    87,
                    88,
                    89,
                    90,
                    91,
                    92,
                    93,
                    94,
                    95,
                    96,
                    97,
                    98,
                    99,
                    100,
                    101,
                ]
            ),
            elements=np.array(
                [
                    12,
                    13,
                    14,
                    15,
                    16,
                    17,
                    18,
                    19,
                    20,
                    21,
                ]
            ),
            dimension=3,
        ),
        io.CFSRegData(
            name="V_QUAD",
            nodes=np.array(
                [
                    1,
                    2,
                    3,
                    4,
                    5,
                    6,
                    7,
                    8,
                    30,
                    31,
                    32,
                    33,
                    34,
                    35,
                    36,
                    37,
                    38,
                    39,
                    40,
                    41,
                ]
            ),
            elements=np.array([7]),
            dimension=3,
        ),
        io.CFSRegData(
            name="V_TET",
            nodes=np.array(
                [
                    6,
                    7,
                    9,
                    10,
                    11,
                    12,
                    13,
                    14,
                    40,
                    42,
                    43,
                    44,
                    45,
                    46,
                    47,
                    48,
                    49,
                    50,
                    51,
                    52,
                    53,
                    54,
                    55,
                    56,
                    57,
                    58,
                    59,
                ]
            ),
            elements=np.array([1, 2, 3, 4, 5, 6]),
            dimension=3,
        ),
        io.CFSRegData(
            name="V_WEDGE",
            nodes=np.array(
                [
                    2,
                    3,
                    6,
                    7,
                    15,
                    16,
                    17,
                    18,
                    19,
                    20,
                    21,
                    22,
                    23,
                    24,
                    25,
                    26,
                    33,
                    34,
                    36,
                    40,
                    60,
                    61,
                    62,
                    63,
                    64,
                    65,
                    66,
                    67,
                    68,
                    69,
                    70,
                    71,
                    72,
                    73,
                    74,
                    75,
                    76,
                    77,
                    78,
                    79,
                    80,
                    81,
                    82,
                    83,
                    84,
                    85,
                ]
            ),
            elements=np.array([8, 9, 10, 11]),
            dimension=3,
        ),
    ]

    return io.CFSMeshData(coordinates=coordinates, connectivity=connectivity, types=el_types, regions=regions)


@pytest.fixture
def dummy_CFSResultArray_obj():
    data = np.ones(shape=(3, 10, 3), dtype=float)
    return io.CFSResultArray(
        data,
        quantity="quantity",
        region="region",
        step_values=np.array([0.0, 1.0, 2.0]),
        res_type=cfs_result_type.NODE,
        dim_names=["x", "y", "z"],
        is_complex=False,
        analysis_type=cfs_analysis_type.TRANSIENT,
        multi_step_id=1,
    )
