from datetime import date
from pydantic import BaseModel, Field, model_validator
from typing import Annotated, Generic, Self, Type, TypeVar
from uuid import UUID
from nexo.enums.identity import (
    OptRhesus,
    RhesusMixin,
    OptBloodType,
    BloodTypeMixin,
    Gender,
    GenderMixin,
)
from nexo.enums.status import (
    DataStatus,
    SimpleDataStatusMixin,
)
from nexo.schemas.mixins.identity import (
    DataIdentifier,
    UUIDUserId,
    UUIDOrganizationId,
    DateOfBirth,
)
from nexo.schemas.mixins.timestamp import LifecycleTimestamp
from nexo.types.integer import OptInt
from nexo.types.string import OptStr, OptListOfStrs
from nexo.types.uuid import ListOfUUIDs
from ..enums.checkup import CheckupStatus as CheckupStatusEnum
from ..enums.examination import OptExaminationStatus
from ..enums.finding_parameter import Criteria as CriteriaEnum
from ..enums.finding import Logic as LogicEnum
from ..enums.parameter import (
    ParameterGroup,
    ValueType as ValueTypeEnum,
)
from ..enums.session import SessionType as SessionTypeEnum
from ..mixins.common import ParameterIds
from ..mixins.checkup import CheckupStatus, RadiologyRecordId
from ..mixins.client import Name as ClientName
from ..mixins.examination import ExaminationStatus, Value, Unit as ExaminationUnit
from ..mixins.finding_parameter import Criteria, Weight
from ..mixins.finding import (
    Name as FindingName,
    Aliases as FindingAliases,
    Recommendation,
    Logic,
)
from ..mixins.parameter import (
    _validate_value_type_and_options,
    Group,
    IsMandatory,
    Name as ParameterName,
    Aliases,
    ValueType,
    Options,
    IsNullable,
    Unit as ParameterUnit,
)
from ..mixins.patient import IdCard, FullName, PlaceOfBirth
from ..mixins.rule import RuleData
from ..mixins.session import SessionType, Name as SessionName
from .document import DocumentName, DocumentURL


class ClientSchema(
    ClientName[str],
    UUIDOrganizationId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


OptClientSchema = ClientSchema | None
OptClientSchemaT = TypeVar("OptClientSchemaT", bound=OptClientSchema)


class ClientSchemaMixin(BaseModel, Generic[OptClientSchemaT]):
    client: Annotated[OptClientSchemaT, Field(..., description="Client")]


class PatientSchema(
    RhesusMixin[OptRhesus],
    BloodTypeMixin[OptBloodType],
    GenderMixin[Gender],
    DateOfBirth[date],
    PlaceOfBirth[OptStr],
    FullName[str],
    IdCard[str],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):

    pass


class PatientSchemaMixin(BaseModel):
    patient: Annotated[PatientSchema, Field(..., description="Patient")]


class StandardParameterSchema(
    ParameterUnit[OptStr],
    IsNullable[bool],
    Options[OptListOfStrs],
    ValueType[ValueTypeEnum],
    Aliases[OptListOfStrs],
    ParameterName[str],
    Group[ParameterGroup],
    IsMandatory[bool],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    @model_validator(mode="after")
    def validate_value_type_and_options(self) -> Self:
        _validate_value_type_and_options(self.value_type, self.options)
        return self


OptStandardParameterSchema = StandardParameterSchema | None
ListOfStandardParameterSchemas = list[StandardParameterSchema]


class StandardParameterSchemaMixin(BaseModel):
    parameter: Annotated[StandardParameterSchema, Field(..., description="Parameter")]


class StandardRuleSchema(
    RuleData,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


OptStandardRuleSchema = StandardRuleSchema | None
ListOfStandardRuleSchemas = list[StandardRuleSchema]


class StandardRuleSchemasMixin(BaseModel):
    rules: Annotated[
        ListOfStandardRuleSchemas,
        Field(ListOfStandardRuleSchemas(), description="Rules"),
    ] = ListOfStandardRuleSchemas()


class FullParameterSchema(
    StandardRuleSchemasMixin,
    StandardParameterSchema,
):
    pass


AnyParameterSchemaType = Type[StandardParameterSchema] | Type[FullParameterSchema]
AnyParameterSchema = StandardParameterSchema | FullParameterSchema
AnyParameterSchemaT = TypeVar("AnyParameterSchemaT", bound=AnyParameterSchema)
OptAnyParameterSchema = AnyParameterSchema | None
OptAnyParameterSchemaT = TypeVar("OptAnyParameterSchemaT", bound=OptAnyParameterSchema)


class ParameterSchemaMixin(BaseModel, Generic[OptAnyParameterSchemaT]):
    parameter: Annotated[OptAnyParameterSchemaT, Field(..., description="Parameter")]


class FullRuleSchema(
    RuleData,
    StandardParameterSchemaMixin,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


AnyRuleSchemaType = Type[StandardRuleSchema] | Type[FullRuleSchema]
AnyRuleSchema = StandardRuleSchema | FullRuleSchema
AnyRuleSchemaT = TypeVar("AnyRuleSchemaT", bound=AnyRuleSchema)
OptAnyRuleSchema = AnyRuleSchema | None
OptAnyRuleSchemaT = TypeVar("OptAnyRuleSchemaT", bound=OptAnyRuleSchema)


class RuleSchemaMixin(BaseModel, Generic[OptAnyRuleSchemaT]):
    rule: Annotated[OptAnyRuleSchemaT, Field(..., description="Rule")]


class StandardFindingSchema(
    Logic[LogicEnum],
    Recommendation[str],
    FindingAliases[OptListOfStrs],
    FindingName[str],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


ListOfStandardFindingSchemas = list[StandardFindingSchema]


class FindingParameterSchema(
    Weight[OptInt],
    Criteria[CriteriaEnum],
    StandardParameterSchemaMixin,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


ListOfFindingParameterSchemas = list[FindingParameterSchema]


class FindingParameterSchemasMixin(BaseModel):
    parameters: Annotated[
        ListOfFindingParameterSchemas,
        Field(ListOfFindingParameterSchemas(), description="Finding Parameters"),
    ] = ListOfFindingParameterSchemas()


class FullFindingSchema(
    FindingParameterSchemasMixin,
    StandardFindingSchema,
):
    pass


ListOfFullFindingSchemas = list[FullFindingSchema]


AnyFindingSchemaType = Type[StandardFindingSchema] | Type[FullFindingSchema]
AnyFindingSchema = StandardFindingSchema | FullFindingSchema
AnyFindingSchemaT = TypeVar("AnyFindingSchemaT", bound=AnyFindingSchema)
OptAnyFindingSchema = AnyFindingSchema | None
OptAnyFindingSchemaT = TypeVar("OptAnyFindingSchemaT", bound=OptAnyFindingSchema)


class LabExtractedExaminationData(BaseModel):
    parameter: Annotated[str, Field(..., description="Parameter's name")]
    value: Annotated[str, Field(..., description="Parameter's value")]
    unit: Annotated[str, Field(..., description="Parameter's unit")]


ListOfLabExtractedExaminationData = list[LabExtractedExaminationData]


class RawExaminationSchema(
    ExaminationUnit,
    Value,
    ParameterSchemaMixin[StandardParameterSchema],
):
    pass


OptRawExaminationSchema = RawExaminationSchema | None
ListOfRawExaminationSchemas = list[RawExaminationSchema]


class BaseExaminationSchema(
    RuleSchemaMixin[OptStandardRuleSchema],
    ExaminationStatus[OptExaminationStatus],
    RawExaminationSchema,
):
    pass


class ExaminationSchema(
    BaseExaminationSchema,
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


ListOfExaminationSchemas = list[ExaminationSchema]


class ExaminationSchemasMixin(BaseModel):
    examinations: Annotated[
        ListOfExaminationSchemas,
        Field(ListOfExaminationSchemas(), description="Examinations"),
    ] = ListOfExaminationSchemas()


class StandardCheckupSchema(
    ExaminationSchemasMixin,
    RadiologyRecordId,
    PatientSchemaMixin,
    DocumentURL,
    DocumentName,
    CheckupStatus[CheckupStatusEnum],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


ListOfStandardCheckupSchemas = list[StandardCheckupSchema]


class StandardCheckupSchemasMixin(BaseModel):
    checkups: Annotated[
        ListOfStandardCheckupSchemas,
        Field(ListOfStandardCheckupSchemas(), description="Checkups"),
    ] = ListOfStandardCheckupSchemas()


class StandardSessionSchema(
    ParameterIds[ListOfUUIDs],
    SessionName[str],
    ClientSchemaMixin[OptClientSchema],
    SessionType[SessionTypeEnum],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    @model_validator(mode="after")
    def validate_type_client(self) -> Self:
        if self.type is SessionTypeEnum.GROUP:
            if self.client is None:
                raise ValueError("Client can not be None for group MCU")
        elif self.type is SessionTypeEnum.INDIVIDUAL:
            if self.client is not None:
                raise ValueError("Client must be None for individual MCU")
        return self


class StandardSessionSchemaMixin(BaseModel):
    session: Annotated[StandardSessionSchema, Field(..., description="Session")]


class FullCheckupSchema(StandardSessionSchemaMixin, StandardCheckupSchema):
    pass


AnyCheckupSchemaType = Type[StandardCheckupSchema] | Type[FullCheckupSchema]
AnyCheckupSchema = StandardCheckupSchema | FullCheckupSchema


class FullSessionSchema(StandardCheckupSchemasMixin, StandardSessionSchema):
    pass


AnySessionSchemaType = Type[StandardSessionSchema] | Type[FullSessionSchema]
AnySessionSchema = StandardSessionSchema | FullSessionSchema
