from nonebot import logger
from nonebot.adapters.onebot.v11 import (
    Bot,
    Message,
    MessageEvent,
    MessageSegment,
    PrivateMessageEvent,
)
from nonebot.matcher import Matcher
from nonebot.params import CommandArg

from amrita.plugins.chat.config import config_manager
from amrita.utils.send import send_forward_msg

from ..utils.llm_tools.mcp_client import ClientManager


async def mcp_status(
    bot: Bot, matcher: Matcher, event: MessageEvent, arg: Message = CommandArg()
):
    arg_text = arg.extract_plain_text().strip()
    tools_count = len(ClientManager.name_to_clients)
    mcp_server_counts = len(ClientManager.clients)
    tools_mapping_count = len(ClientManager.tools_remapping)
    std_txt = f"MCP状态统计\nMCP Servers: {mcp_server_counts}\nMCP Tools: {tools_count}\nMCP Tools(Mapped): {tools_mapping_count}"
    if arg_text in ("-d", "--detail", "--details"):
        if not isinstance(event, PrivateMessageEvent):
            await matcher.finish("-d只允许在私聊执行来避免安全问题")
        detailed_info = [
            MessageSegment.text(std_txt),
            *[
                MessageSegment.text(
                    f"Server@{client.server_script!s} Tools: \n".join(
                        [
                            f" - {tool.function.name}:{tool.function.description}\n"
                            for tool in client.openai_tools
                        ]
                    )
                )
                for client in ClientManager.clients
            ],
        ]

        await send_forward_msg(
            bot, event, "Amrita-MCP", str(event.self_id), detailed_info
        )

    await matcher.finish(std_txt)


async def add_mcp_server(
    matcher: Matcher, bot: Bot, event: MessageEvent, args: Message = CommandArg()
):
    if not config_manager.config.llm_config.tools.agent_mcp_client_enable:
        return
    config = config_manager.ins_config
    mcp_server = args.extract_plain_text().strip()
    if not mcp_server:
        await matcher.finish("请输入MCP Server脚本路径")
    if mcp_server in config.llm_config.tools.agent_mcp_server_scripts:
        await matcher.finish("MCP Server脚本已存在")
    try:
        await ClientManager().initialize_this(mcp_server)
        config.llm_config.tools.agent_mcp_server_scripts.append(mcp_server)
        await config_manager.save_config()
        await matcher.finish("添加成功")
    except Exception as e:
        await matcher.send(f"添加失败: {e}")
        logger.opt(exception=e, colors=True).exception(e)


async def del_mcp_server(matcher: Matcher, args: Message = CommandArg()):
    if not config_manager.config.llm_config.tools.agent_mcp_client_enable:
        return
    config = config_manager.ins_config
    mcp_server = args.extract_plain_text().strip()
    if not mcp_server:
        await matcher.finish("请输入要删除的MCP Server")
    if mcp_server not in config.llm_config.tools.agent_mcp_server_scripts:
        await matcher.finish("MCP Server不存在")
    try:
        await ClientManager().unregister_client(mcp_server)
        config.llm_config.tools.agent_mcp_server_scripts.remove(mcp_server)
        await config_manager.save_config()
        await matcher.finish("删除成功")
    except Exception as e:
        logger.opt(exception=e, colors=True).exception(e)
        await matcher.finish("删除失败")


async def reload(matcher: Matcher):
    if not config_manager.config.llm_config.tools.agent_mcp_client_enable:
        return
    try:
        await ClientManager().initialize_all()
        await matcher.send("重载成功")
    except Exception as e:
        logger.opt(exception=e, colors=True).exception(e)
        await matcher.send("重载失败")
