import pytest
import requests

from easy_acumatica.client import AcumaticaClient
from easy_acumatica.sub_services.work_calendars import WorkCalendarsService
from easy_acumatica.models.work_calendar_builder import WorkCalendarBuilder
from easy_acumatica.models.query_builder import QueryOptions
from easy_acumatica.models.filter_builder import F

API_VERSION = "24.200.001"
BASE = "https://fake"
ENTITY_NAME = "WorkCalendar"

class DummyResponse:
    def __init__(self, status_code: int, json_body=None, headers=None):
        self.status_code = status_code
        self._json = json_body or {}
        self.headers = headers or {}

    def raise_for_status(self):
        if self.status_code >= 400:
            raise requests.HTTPError(f"HTTP {self.status_code}")

    def json(self):
        return self._json

@pytest.fixture
def client(monkeypatch):
    monkeypatch.setattr(AcumaticaClient, "login", lambda self: 204)
    monkeypatch.setattr(AcumaticaClient, "logout", lambda self: 204)
    client_instance = AcumaticaClient(base_url=BASE, username="u", password="p", tenant="t", branch="b")
    # Manually attach the service since it's new
    client_instance.work_calendars = WorkCalendarsService(client_instance)
    return client_instance

@pytest.fixture
def service(client):
    return client.work_calendars

# --- Tests for get_work_calendar ---

def test_get_work_calendar_success(monkeypatch, service):
    """Tests successful retrieval of work calendars."""
    expected_response = [{"WorkCalendarID": {"value": "CAL001"}}]

    def fake_request(method, url, **kwargs):
        assert method.lower() == "get"
        assert url == f"{BASE}/entity/Default/{API_VERSION}/{ENTITY_NAME}"
        return DummyResponse(200, expected_response)

    monkeypatch.setattr(service._client, "_request", fake_request)
    result = service.get_work_calendar(API_VERSION)
    assert result == expected_response

def test_get_work_calendar_with_options(monkeypatch, service):
    """Tests retrieving work calendars with QueryOptions."""
    expected_response = [{"WorkCalendarID": {"value": "SPECIFIC"}}]
    opts = QueryOptions(filter=F.WorkCalendarID == "SPECIFIC")

    def fake_request(method, url, **kwargs):
        assert method.lower() == "get"
        assert url == f"{BASE}/entity/Default/{API_VERSION}/{ENTITY_NAME}"
        assert kwargs.get("params") == opts.to_params()
        return DummyResponse(200, expected_response)

    monkeypatch.setattr(service._client, "_request", fake_request)
    result = service.get_work_calendar(API_VERSION, options=opts)
    assert result == expected_response

def test_get_work_calendar_error(monkeypatch, service):
    """Tests that an API error is propagated correctly during a GET request."""
    def fake_request(method, url, **kwargs):
        raise RuntimeError("Acumatica API error 500: Server Error")

    monkeypatch.setattr(service._client, "_request", fake_request)

    with pytest.raises(RuntimeError, match="Acumatica API error 500"):
        service.get_work_calendar(API_VERSION)

# --- Tests for create_work_calendar ---

def test_create_work_calendar_success(monkeypatch, service):
    """Tests successful creation of a work calendar."""
    builder = (
        WorkCalendarBuilder()
        .work_calendar_id("NEWCAL")
        .description("New Calendar")
    )
    expected_response = builder.to_body()

    def fake_request(method, url, **kwargs):
        assert method.lower() == "put"
        assert url == f"{BASE}/entity/Default/{API_VERSION}/{ENTITY_NAME}"
        assert kwargs.get("json") == builder.to_body()
        return DummyResponse(201, expected_response)

    monkeypatch.setattr(service._client, "_request", fake_request)
    result = service.create_work_calendar(API_VERSION, builder)
    assert result == expected_response

def test_create_work_calendar_error(monkeypatch, service):
    """Tests that an API error is propagated correctly during a create request."""
    builder = WorkCalendarBuilder().work_calendar_id("INVALID")

    def fake_request(method, url, **kwargs):
        raise RuntimeError("Acumatica API error 422: Unprocessable Entity")

    monkeypatch.setattr(service._client, "_request", fake_request)

    with pytest.raises(RuntimeError, match="Acumatica API error 422"):
        service.create_work_calendar(API_VERSION, builder)