############################################################
#
#       #   #  #   #   #    #
#      ##  ##  #  ##  #    #
#     # # # #  # # # #    #  #
#    #  ##  #  ##  ##    ######
#   #   #   #  #   #       #
#
# Python-based Tool for interaction with the 10micron mounts
# GUI with PySide
#
# written in python3, (c) 2019-2025 by mworion
# Licence APL2.0
#
###########################################################
# standard libraries
import logging
import xml.etree.ElementTree as ETree

# external packages
from PySide6.QtCore import QObject
from PySide6.QtNetwork import QTcpSocket

# local import
from mw4.indibase import indiXML
from mw4.indibase.indiDevice import Device
from mw4.indibase.indiSignals import INDISignals


class Client(QObject):
    """
    Client implements an INDI Base Client for INDI servers. it relies on PySide6
    and it's signalling scheme. there might be not all capabilities implemented
    right now. all the data, properties and attributes are stored in a device
    dict. The reading and parsing of the XML data is done in a streaming way,
    so for xml the xml.parse.feed() mechanism is used.
    """

    __all__ = ["Client"]
    log = logging.getLogger("MW4")

    GENERAL_INTERFACE = 0
    TELESCOPE_INTERFACE = 1 << 0
    CCD_INTERFACE = 1 << 1
    GUIDER_INTERFACE = 1 << 2
    FOCUSER_INTERFACE = 1 << 3
    FILTER_INTERFACE = 1 << 4
    DOME_INTERFACE = 1 << 5
    GPS_INTERFACE = 1 << 6
    WEATHER_INTERFACE = 1 << 7
    AO_INTERFACE = 1 << 8
    DUSTCAP_INTERFACE = 1 << 9
    LIGHTBOX_INTERFACE = 1 << 10
    DETECTOR_INTERFACE = 1 << 11
    AUX_INTERFACE = 1 << 15

    DEFAULT_PORT = 7624
    CONNECTION_TIMEOUT = 1000

    def __init__(self, host=None):
        super().__init__()

        self.host = host
        self.signals = INDISignals()
        self.connected = False
        self.blobMode = "Never"
        self.devices = {}
        self.curDepth = 0
        self.parser = None
        self.socket = QTcpSocket()
        self.socket.readyRead.connect(self.handleReadyRead)
        self.socket.errorOccurred.connect(self.handleError)
        self.socket.disconnected.connect(self.handleDisconnected)
        self.socket.connected.connect(self.handleConnected)
        self.clearParser()

    @property
    def host(self):
        return self._host

    @host.setter
    def host(self, value):
        value = self.checkFormat(value)
        self._host = value

    def checkFormat(self, value):
        """
        :param value:
        :return:
        """
        if not value:
            return None
        if not isinstance(value, tuple | str):
            self.log.info(f"wrong host value: {value}")
            return None
        if isinstance(value, str):
            value = (value, self.DEFAULT_PORT)
        if not value[0]:
            return None

        return value

    def clearParser(self):
        """
        :return: success for test purpose
        """
        self.parser = ETree.XMLPullParser(["start", "end"])
        self.parser.feed("<root>")
        for _, _ in self.parser.read_events():
            pass

        return True

    def setServer(self, host=None, port=7624):
        """
        Part of BASE CLIENT API of EKOS
        setServer sets the server address of the indi server

        :param host: host name as string
        :param port: port as int
        :return: success for test purpose
        """
        self.host = (host, port)
        self.host = self.checkFormat((host, port))
        if self.host is None:
            return False

        self.connected = False
        return True

    def watchDevice(self, deviceName=""):
        """
        Part of BASE CLIENT API of EKOS
        adds a device to the watchlist. if the device name is empty, all traffic
        for all devices will be watched and therefore received

        :param deviceName: name string of INDI device
        :return: success for test purpose
        """
        if deviceName:
            cmd = indiXML.clientGetProperties(
                indi_attr={"version": "1.7", "device": deviceName}
            )
        else:
            cmd = indiXML.clientGetProperties(indi_attr={"version": "1.7"})

        suc = self._sendCmd(cmd)
        return suc

    def connectServer(self):
        """
        Part of BASE CLIENT API of EKOS
        connect starts the link to the indi server.

        :return: success
        """
        if self._host is None:
            return False
        if self.connected:
            return False

        self.connected = False
        if self.socket.state() != QTcpSocket.SocketState.UnconnectedState:
            self.socket.abort()

        self.socket.connectToHost(*self._host)
        return True

    def clearDevices(self, deviceName=""):
        """
        clearDevices deletes all the actual knows devices and sens out the
        appropriate qt signals

        :param deviceName: name string of INDI device
        :return: success for test purpose
        """
        for device in self.devices:
            if device != deviceName and deviceName:
                continue

            self.signals.deviceDisconnected.emit(device)
            self.signals.removeDevice.emit(device)
            self.log.info(f"Remove device [{device}]")

        self.devices = {}
        return True

    def disconnectServer(self, deviceName=""):
        """
        Part of BASE CLIENT API of EKOS
        disconnect drops the connection to the indi server

        :param deviceName: name string of INDI device
        :return: success
        """
        self.connected = False
        self.clearParser()
        self.clearDevices(deviceName)
        self.signals.serverDisconnected.emit(self.devices)
        self.socket.abort()
        return True

    def isServerConnected(self):
        """
        Part of BASE CLIENT API of EKOS

        :return: true if server connected
        """
        return self.connected

    def connectDevice(self, deviceName=""):
        """
        Part of BASE CLIENT API of EKOS

        :param deviceName: name string of INDI device
        :return: success
        """
        if not self.connected:
            return False
        if not deviceName:
            return False
        if deviceName not in self.devices:
            return False

        con = self.devices[deviceName].getSwitch("CONNECTION")
        if con["CONNECT"] == "On":
            self.log.info(f"Device [{deviceName}] was connected at startup")
            return False

        else:
            self.log.info(f"Device [{deviceName}] unconnected - connect it now")

        suc = self.sendNewSwitch(
            deviceName=deviceName,
            propertyName="CONNECTION",
            elements={"CONNECT": "On", "DISCONNECT": "Off"},
        )
        return suc

    def disconnectDevice(self, deviceName=""):
        """
        Part of BASE CLIENT API of EKOS

        :param deviceName: name string of INDI device
        :return: success
        """
        if not self.connected:
            return False
        if not deviceName:
            return False
        if deviceName not in self.devices:
            return False

        con = self.devices[deviceName].getSwitch("CONNECTION")
        if con["DISCONNECT"] == "On":
            self.log.info(f"{deviceName} already disconnected")
            return False

        suc = self.sendNewSwitch(
            deviceName=deviceName,
            propertyName="CONNECTION",
            elements={"CONNECT": "Off", "DISCONNECT": "On"},
        )
        return suc

    def getDevice(self, deviceName=""):
        """
        Part of BASE CLIENT API of EKOS
        getDevice collects all the data of the given device

        :param deviceName: name of device
        :return: dict with data of that give device
        """
        value = self.devices.get(deviceName, None)
        return value

    def getDevices(self, driverInterface=0xFFFF):
        """
        Part of BASE CLIENT API of EKOS
        getDevices generates a list of devices, which are from type of the given
        driver interface type.

        :param driverInterface: binary value of driver interface type
        :return: list of knows devices of this type
        """
        deviceList = []
        for deviceName in self.devices:
            typeCheck = self._getDriverInterface(deviceName) & driverInterface
            if typeCheck:
                deviceList.append(deviceName)
        return deviceList

    def setBlobMode(self, blobHandling="Never", deviceName="", propertyName=""):
        """
        Part of BASE CLIENT API of EKOS

        :param blobHandling:
        :param deviceName: name string of INDI device
        :param propertyName: name string of device property
        :return: true if server connected
        """
        if not deviceName:
            return False
        if deviceName not in self.devices:
            return False

        cmd = indiXML.enableBLOB(
            blobHandling, indi_attr={"name": propertyName, "device": deviceName}
        )
        self.blobMode = blobHandling
        suc = self._sendCmd(cmd)
        return suc

    def getBlobMode(self, deviceName="", propertyName=""):
        """
        Part of BASE CLIENT API of EKOS

        :param deviceName: name string of INDI device
        :param propertyName: name string of device property
        :return: None, because not implemented
        """
        pass

    def getHost(self):
        """
        Part of BASE CLIENT API of EKOS

        :return: host name as str
        """
        if self._host is None:
            return ""

        return self._host[0]

    def getPort(self):
        """
        Part of BASE CLIENT API of EKOS

        :return: port number as int
        """
        if self._host is None:
            return 0

        return self._host[1]

    def sendNewText(self, deviceName="", propertyName="", elements="", text=""):
        """
        Part of BASE CLIENT API of EKOS

        :param deviceName: name string of INDI device
        :param propertyName: name string of device property
        :param elements: element name or dict of element name / values
        :param text: string in case of having only one element in elements
        :return: success for test
        """
        if deviceName not in self.devices:
            return False
        # if not hasattr(self.devices[deviceName], propertyName):
        #     return False
        if not isinstance(elements, dict):
            elements = {elements: text}

        elementList = []
        for element in elements:
            text = elements[element]
            elementList.append(indiXML.oneText(text, indi_attr={"name": element}))
        cmd = indiXML.newTextVector(
            elementList, indi_attr={"name": propertyName, "device": deviceName}
        )
        suc = self._sendCmd(cmd)
        return suc

    def sendNewNumber(self, deviceName="", propertyName="", elements="", number=0):
        """
        Part of BASE CLIENT API of EKOS

        :param deviceName: name string of INDI device
        :param propertyName: name string of device property
        :param elements: element name or dict of element name / values
        :param number: value in case of having only one element in elements
        :return: success for test
        """
        if deviceName not in self.devices:
            return False
        # if not hasattr(self.devices[deviceName], propertyName):
        #     return False
        if not isinstance(elements, dict):
            elements = {elements: number}

        elementList = []
        for element in elements:
            number = float(elements[element])
            elementList.append(indiXML.oneNumber(number, indi_attr={"name": element}))
        cmd = indiXML.newNumberVector(
            elementList, indi_attr={"name": propertyName, "device": deviceName}
        )
        suc = self._sendCmd(cmd)
        return suc

    def sendNewSwitch(self, deviceName="", propertyName="", elements=""):
        """
        Part of BASE CLIENT API of EKOS

        :param deviceName: name string of INDI device
        :param propertyName: name string of device property
        :param elements: element name or dict of element name / values
        :return: success for test
        """
        if deviceName not in self.devices:
            return False
        # if not hasattr(self.devices[deviceName], propertyName):
        #     return False
        if not isinstance(elements, dict):
            elements = {elements: "On"}

        elementList = []
        for element in elements:
            switch = elements[element]
            elementList.append(indiXML.oneSwitch(switch, indi_attr={"name": element}))
        cmd = indiXML.newSwitchVector(
            elementList, indi_attr={"name": propertyName, "device": deviceName}
        )
        suc = self._sendCmd(cmd)
        return suc

    @staticmethod
    def startBlob(deviceName="", propertyName="", timestamp=""):
        """
        Part of BASE CLIENT API of EKOS
        :return:
        """
        return True

    @staticmethod
    def sendOneBlob(blobName="", blobSize=0, blobFormat="", blobBuffer=None):
        """
        Part of BASE CLIENT API of EKOS
        :return:
        """
        return True

    @staticmethod
    def finishBlob():
        """
        Part of BASE CLIENT API of EKOS
        :return:
        """
        return True

    @staticmethod
    def setVerbose(status):
        """
        Part of BASE CLIENT API of EKOS
        :return:
        """
        return True

    @staticmethod
    def isVerbose():
        """
        Part of BASE CLIENT API of EKOS
        :return: status of verbose
        """
        return False

    def setConnectionTimeout(self, seconds=2, microseconds=0):
        """
        Part of BASE CLIENT API of EKOS
        :return: success for test purpose
        """
        self.CONNECTION_TIMEOUT = seconds + microseconds / 1000000
        return True

    def _sendCmd(self, indiCommand):
        """
        :param indiCommand: XML command to send
        :return: success of sending
        """
        if self.connected:
            cmd = indiCommand.toXML()
            self.log.trace(f"SendCmd: [{cmd.decode().lstrip('<').rstrip('/>')}]")
            number = self.socket.write(cmd + b"\n")
            self.socket.flush()
            return number > 0
        else:
            return False

    def _getDriverInterface(self, deviceName):
        """
        _getDriverInterface look the type of the device's driver interface up
        and gives it back as binary value.

        :param deviceName: device name
        :return: binary value of type of device drivers interface
        """
        device = self.devices[deviceName]
        if not hasattr(device, "DRIVER_INFO"):
            return -1

        val = getattr(device, "DRIVER_INFO")
        if val:
            val = val["elementList"].get("DRIVER_INTERFACE", "")
            if val:
                interface = val["value"]
                return int(interface)

            else:
                return -1
        else:
            return -1

    def _fillAttributes(self, deviceName=None, chunk=None, elementList=None, defVector=None):
        """
        :param deviceName: device name
        :param chunk:   xml element from INDI
        :param elementList:
        :param defVector:
        :return: True for test purpose
        """
        for elt in chunk.elt_list:
            name = elt.attr.get("name", "")
            if not name:
                return False
            if name in elementList:
                elementList[name].clear()
            else:
                elementList[name] = {}
            elementList[name]["elementType"] = elt.etype

            # as a new blob vector does not contain an initial value, we have to
            # separate this
            if not isinstance(elt, indiXML.DefBLOB):
                elementList[name]["value"] = elt.getValue()

            # now all other attributes of element are stored
            for attr in elt.attr:
                elementList[name][attr] = elt.attr[attr]

            # send connected signals
            if name == "CONNECT" and elt.getValue() == "On" and chunk.attr["state"] == "Ok":
                self.signals.deviceConnected.emit(deviceName)
                self.log.info(f"Device [{deviceName}] connected")

            if name == "DISCONNECT" and elt.getValue() == "On":
                self.signals.deviceDisconnected.emit(deviceName)
                self.log.info(f"Device [{deviceName}] disconnected")

        return True

    @staticmethod
    def _setupPropertyStructure(chunk=None, device=None):
        """
        :param chunk:   xml element from INDI
        :param device:  device class
        :return:
        """
        iProperty = chunk.attr.get("name", "")
        if not hasattr(device, iProperty):
            setattr(device, iProperty, {})

        # shortening for readability
        deviceProperty = getattr(device, iProperty)

        deviceProperty["propertyType"] = chunk.etype
        for vecAttr in chunk.attr:
            deviceProperty[vecAttr] = chunk.attr.get(vecAttr)

        # adding subspace for atomic elements (text, switch, etc)
        deviceProperty["elementList"] = {}
        elementList = deviceProperty["elementList"]

        return iProperty, elementList

    def _getDeviceReference(self, chunk=None):
        """
        _getDeviceReference extracts the device name from INDI chunk and looks
        device presence in INDi base class up. if not present, a new device will
        be generated

        :param chunk:   xml element from INDI
        :return: device and device name
        """
        deviceName = chunk.attr.get("device", "")
        if deviceName not in self.devices:
            self.devices[deviceName] = Device(deviceName)
            self.signals.newDevice.emit(deviceName)
            self.log.info(f"New device [{deviceName}]")

        device = self.devices[deviceName]
        return device, deviceName

    def _delProperty(self, chunk=None, device=None, deviceName=None):
        """
        _delProperty removes property from device class

        :param chunk:   xml element from INDI
        :param device:  device class
        :param deviceName: device name
        :return: success
        """
        if deviceName not in self.devices:
            return False
        if "name" not in chunk.attr:
            return False

        iProperty = chunk.attr["name"]

        self.log.trace(f"DEL: Device:{device.name}, Property: {iProperty}")

        if hasattr(device, iProperty):
            delattr(device, iProperty)
            self.signals.removeProperty.emit(deviceName, iProperty)
            self.log.info(f"Device [{deviceName}] del property [{iProperty}]")

        return True

    def _setProperty(self, chunk=None, device=None, deviceName=None):
        """
        _sefProperty generate and write all data to device class for SefVector
        chunks

        :param chunk:   xml element from INDI
        :param device:  device class
        :param deviceName: device name
        :return: success
        """
        iProperty, elementList = self._setupPropertyStructure(chunk=chunk, device=device)
        self._fillAttributes(
            deviceName=deviceName, chunk=chunk, elementList=elementList, defVector=False
        )

        self.log.trace(
            f"SET: Device:{device.name}, Property: {iProperty}, Elements: {elementList}"
        )

        if isinstance(chunk, indiXML.SetBLOBVector):
            self.signals.newBLOB.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.SetSwitchVector):
            self.signals.newSwitch.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.SetNumberVector):
            self.signals.newNumber.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.SetTextVector):
            self.signals.newText.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.SetLightVector):
            self.signals.newLight.emit(deviceName, iProperty)

        return True

    def _defProperty(self, chunk=None, device=None, deviceName=None):
        """
        _defProperty generate and write all data to device class for DefVector
        chunks

        :param chunk:   xml element from INDI
        :param device:  device class
        :param deviceName: device name
        :return: success
        """
        iProperty, elementList = self._setupPropertyStructure(chunk=chunk, device=device)
        self._fillAttributes(
            deviceName=deviceName, chunk=chunk, elementList=elementList, defVector=True
        )
        self.log.trace(
            f"DEF: Device:{device.name}, Property: {iProperty}, Elements: {elementList}"
        )
        self.signals.newProperty.emit(deviceName, iProperty)
        if isinstance(chunk, indiXML.DefBLOBVector):
            self.signals.defBLOB.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.DefSwitchVector):
            self.signals.defSwitch.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.DefNumberVector):
            self.signals.defNumber.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.DefTextVector):
            self.signals.defText.emit(deviceName, iProperty)
        elif isinstance(chunk, indiXML.DefLightVector):
            self.signals.defLight.emit(deviceName, iProperty)
        return True

    def _getProperty(self, chunk=None, device=None, deviceName=None):
        """
        :param chunk:   xml element from INDI
        :param device:  device class
        :param deviceName: device name
        :return: success
        """
        self.log.trace(f"GET: Device:{device.name}")
        return True

    def _message(self, chunk=None, deviceName=None):
        """
        :param chunk:   xml element from INDI
        :param deviceName: device name
        :return: success
        """
        message = chunk.attr.get("message", "-")
        self.signals.newMessage.emit(deviceName, message)
        return True

    def _parseCmd(self, chunk):
        """
        _parseCmd parses the incoming indi XL data and builds up a dictionary
        of devices in device class which holds all the data transferred through
        INDI protocol.

        :param chunk: raw indi XML element
        :return: success if it could be parsed
        """
        if not self.connected:
            return False

        if "device" not in chunk.attr:
            self.log.warning(f"No device in chunk: [{chunk}]")
            return False

        device, deviceName = self._getDeviceReference(chunk=chunk)

        # all message have no device names, they could be general
        if isinstance(chunk, indiXML.Message):
            self._message(chunk=chunk, deviceName=deviceName)
            return True

        if "name" not in chunk.attr:
            self.log.warning(f"No property in chunk: [{chunk}]")
            return False

        if isinstance(chunk, indiXML.DelProperty):
            self._delProperty(chunk=chunk, device=device, deviceName=deviceName)
            return True

        if isinstance(
            chunk,
            indiXML.SetBLOBVector
            | indiXML.SetSwitchVector
            | indiXML.SetTextVector
            | indiXML.SetLightVector
            | indiXML.SetNumberVector,
        ):
            self._setProperty(chunk=chunk, device=device, deviceName=deviceName)
            return True

        if isinstance(
            chunk,
            indiXML.DefBLOBVector
            | indiXML.DefSwitchVector
            | indiXML.DefTextVector
            | indiXML.DefLightVector
            | indiXML.DefNumberVector,
        ):
            self._defProperty(chunk=chunk, device=device, deviceName=deviceName)
            return True

        if isinstance(chunk, indiXML.GetProperties):
            self._getProperty(chunk=chunk, device=device, deviceName=deviceName)
            return True

        if isinstance(
            chunk,
            indiXML.NewBLOBVector
            | indiXML.NewSwitchVector
            | indiXML.NewTextVector
            | indiXML.NewNumberVector,
        ):
            # todo: what to do with the "New" vector ?
            return True

        if isinstance(
            chunk,
            indiXML.OneBLOB
            | indiXML.OneSwitch
            | indiXML.OneText
            | indiXML.OneNumber
            | indiXML.OneLight,
        ):
            # todo: what to do with the "One" vector ?
            return True

        self.log.error(f"Unknown vectors: [{chunk}]")
        return False

    def handleReadyRead(self):
        """
        _handleReadyRead gets the date in buffer signal and starts to read data
        from the network. as long as data is streaming, it feeds to the xml
        parser. with this construct you don't have to put the whole data set into
        the parser at once, but doing the work step be step.

        :return: nothing
        """
        buf = self.socket.readAll()
        self.parser.feed(buf)
        try:
            for event, elem in self.parser.read_events():
                if event == "start":
                    self.curDepth += 1

                elif event == "end":
                    self.curDepth -= 1

                else:
                    self.log.critical(f"Problem parsing event: [{event}]")
                    continue

                if self.curDepth > 0:
                    continue

                if self.curDepth < 0:
                    self.log.critical(f"Problem parsing event: [{event}]")
                    continue
                elemParsed = indiXML.parseETree(elem)
                elem.clear()
                self._parseCmd(elemParsed)

        except Exception as e:
            self.log.error(f"{e}: {buf}")
            return False

        return True

    def handleConnected(self):
        """
        :return:
        """
        self.connected = True
        self.signals.serverConnected.emit()
        return True

    def handleDisconnected(self):
        """
        :return: nothing
        """
        self.connected = False
        self.signals.serverDisconnected.emit(self.devices)
        self.log.info("INDI client disconnected")
        return True

    def handleError(self, socketError):
        """
        :param socketError: the error from socket library
        :return: nothing
        """
        Refuse = QTcpSocket.SocketError.ConnectionRefusedError
        Unknown = QTcpSocket.SocketError.UnknownSocketError
        RemoteClosed = QTcpSocket.SocketError.RemoteHostClosedError
        if socketError not in [Refuse, Unknown, RemoteClosed]:
            self.log.error(f"INDI error: [{socketError}]")
        else:
            self.log.debug(f"INDI error: [{socketError}]")
        self.disconnectServer()
        return True
