import json
from datetime import datetime, timezone
from http import HTTPStatus
from uuid import UUID

import pytest
import responses

from fiddler_evals.entities.experiment import Experiment, ExperimentStatus
from fiddler_evals.pydantic_models.experiment import (
    ExperimentItemResult,
    NewExperimentItem,
)
from fiddler_evals.pydantic_models.score import Score, ScoreStatus
from fiddler_evals.tests.constants import (
    APPLICATION_COMPACT,
    APPLICATION_ID,
    APPLICATION_NAME,
    DATASET_COMPACT,
    DATASET_ID,
    DATASET_NAME,
    EXPERIMENT_ID,
    EXPERIMENT_NAME,
    HEADERS,
    ORG_ID,
    ORG_NAME,
    PROJECT_COMPACT,
    PROJECT_ID,
    PROJECT_NAME,
    URL,
    USER_COMPACT,
    USER_EMAIL,
    USER_FULL_NAME,
    USER_ID,
)

# Test constants
EXPERIMENT_DESCRIPTION = "Test experiment"
EXPERIMENT_METADATA = {"model": "gpt-4o"}

API_RESPONSE_200 = {
    "data": {
        "id": EXPERIMENT_ID,
        "name": EXPERIMENT_NAME,
        "description": EXPERIMENT_DESCRIPTION,
        "metadata": EXPERIMENT_METADATA,
        "status": "COMPLETED",
        "error_reason": None,
        "error_message": None,
        "traceback": None,
        "organization": {
            "id": ORG_ID,
            "name": ORG_NAME,
        },
        "application": {
            "id": APPLICATION_ID,
            "name": APPLICATION_NAME,
        },
        "dataset": {
            "id": DATASET_ID,
            "name": DATASET_NAME,
        },
        "project": {
            "id": PROJECT_ID,
            "name": PROJECT_NAME,
            "asset_type": "GEN_AI_APP",
        },
        "created_at": "2023-11-22 16:50:57.705784",
        "updated_at": "2023-11-22 16:50:57.705784",
        "created_by": {
            "id": USER_ID,
            "full_name": USER_FULL_NAME,
            "email": USER_EMAIL,
        },
        "updated_by": {
            "id": USER_ID,
            "full_name": USER_FULL_NAME,
            "email": USER_EMAIL,
        },
    },
    "api_version": "3.0",
    "kind": "NORMAL",
}

API_RESPONSE_409 = {
    "error": {
        "code": 409,
        "message": "Experiment already exists",
        "errors": [
            {
                "reason": "Conflict",
                "message": "Experiment already exists",
                "help": "",
            }
        ],
    }
}

API_RESPONSE_404 = {
    "error": {
        "code": 404,
        "message": "Experiment not found for the given identifier",
        "errors": [
            {
                "reason": "ObjectNotFound",
                "message": "Experiment not found for the given identifier",
                "help": "",
            }
        ],
    }
}

API_RESPONSE_FROM_NAME = {
    "data": {
        "page_size": 100,
        "total": 1,
        "item_count": 1,
        "page_count": 1,
        "page_index": 1,
        "offset": 0,
        "items": [API_RESPONSE_200["data"]],
    },
    "api_version": "3.0",
    "kind": "PAGINATED",
}

LIST_API_RESPONSE = {
    "data": {
        "page_size": 100,
        "total": 2,
        "item_count": 2,
        "page_count": 1,
        "page_index": 1,
        "offset": 0,
        "items": [
            API_RESPONSE_200["data"],
            {
                "id": "6531bfd9-2ca2-4a7b-bb5a-136c8da09ca6",
                "name": "test_experiment_2",
                "description": "Test experiment 2",
                "metadata": {"source": "from api"},
                "status": "COMPLETED",
                "error_reason": None,
                "error_message": None,
                "traceback": None,
                "organization": {
                    "id": ORG_ID,
                    "name": ORG_NAME,
                },
                "application": {
                    "id": APPLICATION_ID,
                    "name": APPLICATION_NAME,
                },
                "dataset": {
                    "id": DATASET_ID,
                    "name": DATASET_NAME,
                },
                "project": {
                    "id": PROJECT_ID,
                    "name": PROJECT_NAME,
                    "asset_type": "GEN_AI_APP",
                },
                "created_at": "2023-11-22 16:50:57.705784",
                "updated_at": "2023-11-22 16:50:57.705784",
                "created_by": {
                    "id": USER_ID,
                    "full_name": USER_FULL_NAME,
                    "email": USER_EMAIL,
                },
                "updated_by": {
                    "id": USER_ID,
                    "full_name": USER_FULL_NAME,
                    "email": USER_EMAIL,
                },
            },
        ],
    },
    "api_version": "3.0",
    "kind": "PAGINATED",
}

# Create experiment instance for testing
experiment = Experiment(
    id=UUID(EXPERIMENT_ID),
    name=EXPERIMENT_NAME,
    status=ExperimentStatus.PENDING,
    error_reason=None,
    error_message=None,
    traceback=None,
    created_at=datetime.now(),
    updated_at=datetime.now(),
    created_by=USER_COMPACT,
    updated_by=USER_COMPACT,
    project=PROJECT_COMPACT,
    application=APPLICATION_COMPACT,
    dataset=DATASET_COMPACT,
)


@responses.activate
def test_add_results_success() -> None:
    """When adding results to an experiment
    Then the results should be successfully added
    And the API should be called with correct payload."""
    # Mock the add_results API call with error
    responses.post(
        url=f"{URL}/v3/evals/experiments/{EXPERIMENT_ID}/results",
        json={"error": "Internal server error"},
        headers=HEADERS,
    )

    # Create test data
    dataset_item_id = UUID("12345678-1234-1234-1234-123456789012")
    start_time = datetime.now(tz=timezone.utc)
    end_time = datetime.now(tz=timezone.utc)

    # Create experiment item
    experiment_item = NewExperimentItem(
        dataset_item_id=dataset_item_id,
        outputs={"prediction": "fraud", "confidence": 0.95},
        start_time=start_time,
        end_time=end_time,
        status="COMPLETED",
    )

    # Create scores
    scores = [
        Score(
            name="accuracy",
            evaluator_name="AccuracyEvaluator",
            value=1.0,
            label="Correct",
            status=ScoreStatus.SUCCESS,
            reasoning="Prediction matches ground truth",
        ),
        Score(
            name="confidence",
            evaluator_name="ConfidenceEvaluator",
            value=0.95,
            label="High",
            status=ScoreStatus.SUCCESS,
            reasoning="High confidence in prediction",
        ),
    ]

    # Create result
    result = ExperimentItemResult(experiment_item=experiment_item, scores=scores)

    # Add results to experiment
    experiment.add_results([result])

    # Verify the add_results API call was made
    assert len(responses.calls) == 1  # add_results
    add_results_call = responses.calls[0]

    # Verify request payload
    request_data = json.loads(add_results_call.request.body)
    assert "results" in request_data
    assert len(request_data["results"]) == 1

    result_data = request_data["results"][0]
    assert "experiment_item" in result_data
    assert "scores" in result_data

    # Verify experiment item data
    item_data = result_data["experiment_item"]
    assert item_data["dataset_item_id"] == str(dataset_item_id)
    assert item_data["outputs"] == {"prediction": "fraud", "confidence": 0.95}
    assert item_data["status"] == "COMPLETED"

    # Verify scores data
    scores_data = result_data["scores"]
    assert len(scores_data) == 2
    assert scores_data[0]["name"] == "accuracy"
    assert scores_data[0]["value"] == 1.0
    assert scores_data[1]["name"] == "confidence"
    assert scores_data[1]["value"] == 0.95


@responses.activate
def test_add_results_empty_items() -> None:
    """When adding empty results to an experiment
    Then it should raise ValueError
    And no API call should be made."""

    # Try to add empty results
    with pytest.raises(ValueError, match="Items cannot be empty"):
        experiment.add_results([])

    # Verify only the create call was made, no add_results call
    assert len(responses.calls) == 0


@responses.activate
def test_add_results_api_error() -> None:
    """When add_results API call fails
    Then it should propagate the API error
    And the error should be handled by the decorator."""

    # Mock the add_results API call with error
    responses.post(
        url=f"{URL}/v3/experiments/{EXPERIMENT_ID}/results",
        json={"error": "Internal server error"},
        status=HTTPStatus.INTERNAL_SERVER_ERROR,
        headers=HEADERS,
    )

    # Create test data
    dataset_item_id = UUID("12345678-1234-1234-1234-123456789012")
    start_time = datetime.now(tz=timezone.utc)
    end_time = datetime.now(tz=timezone.utc)

    # Create experiment item
    experiment_item = NewExperimentItem(
        dataset_item_id=dataset_item_id,
        outputs={"prediction": "fraud", "confidence": 0.95},
        start_time=start_time,
        end_time=end_time,
        status="COMPLETED",
    )

    # Create result
    result = ExperimentItemResult(experiment_item=experiment_item, scores=[])

    # Try to add results - should raise an exception
    with pytest.raises(Exception):  # The @handle_api_error decorator will handle this
        experiment.add_results([result])

    # Verify both API calls were made
    assert len(responses.calls) == 1
