from __future__ import annotations

from datetime import datetime, timezone
from uuid import UUID, uuid4

from pydantic import ConfigDict, Field

from fiddler_evals.pydantic_models.base import BaseModel
from fiddler_evals.pydantic_models.compact import (
    ApplicationCompact,
    DatasetCompact,
    OrganizationCompact,
    ProjectCompact,
    UserCompact,
)
from fiddler_evals.pydantic_models.error import Error
from fiddler_evals.pydantic_models.score import Score


class ExperimentResponse(BaseModel):
    """Experiment response from Fiddler API"""

    id: UUID
    name: str
    description: str | None
    metadata: dict
    status: str
    duration_ms: int | None
    error_reason: str | None
    error_message: str | None
    traceback: str | None
    created_at: datetime
    updated_at: datetime
    created_by: UserCompact
    updated_by: UserCompact
    organization: OrganizationCompact
    project: ProjectCompact
    application: ApplicationCompact
    dataset: DatasetCompact


class NewExperimentItem(BaseModel):
    """Model to create a new experiment item"""

    model_config = ConfigDict(use_enum_values=True)

    id: UUID | None = Field(default_factory=uuid4)
    timestamp: datetime = Field(default_factory=lambda: datetime.now(tz=timezone.utc))
    dataset_item_id: UUID
    outputs: dict
    status: str
    duration_ms: int | None = None
    error_reason: str | None = None
    error_message: str | None = None

    def set_error(self, error: Error) -> None:
        self.error_reason = error.reason
        self.error_message = error.message


class ExperimentItem(BaseModel):
    """Experiment item from Fiddler API"""

    id: UUID
    timestamp: datetime
    dataset_item_id: UUID
    outputs: dict
    status: str
    duration_ms: int | None = None
    error_reason: str | None = None
    error_message: str | None = None
    error_traceback: str | None = None
    created_at: datetime
    updated_at: datetime


class ExperimentItemResult(BaseModel):
    """Result of evaluating a single dataset item in an experiment.

    Contains the experiment item data and all scores generated by evaluators
    for that specific dataset item.

    Attributes:
        experiment_item: The experiment item containing outputs, timing, and status.
        scores: List of scores generated by evaluators for this item.
    """

    experiment_item: NewExperimentItem
    scores: list[Score]
