"""
This module provides functions for generating statistics for ISEA3H DGGS cells.
"""
import platform
import math
import pandas as pd
import numpy as np
import argparse
import geopandas as gpd
import matplotlib.pyplot as plt
from mpl_toolkits.axes_grid1 import make_axes_locatable
from matplotlib.colors import TwoSlopeNorm
from vgrid.utils.constants import AUTHALIC_AREA, VMIN_HEX, VMAX_HEX, VCENTER_HEX
from vgrid.generator.isea3hgrid import isea3hgrid
from vgrid.utils.geometry import check_crossing_geom

if platform.system() == "Windows":
    from vgrid.dggs.eaggr.enums.shape_string_format import ShapeStringFormat
    from vgrid.dggs.eaggr.eaggr import Eaggr
    from vgrid.dggs.eaggr.shapes.dggs_cell import DggsCell
    from vgrid.dggs.eaggr.enums.model import Model
    isea3h_dggs = Eaggr(Model.ISEA3H)


def isea3h_metrics(res, unit: str = "m"):
    """
    Calculate metrics for ISEA3H DGGS cells at a given resolution.
    
    Args:
        res: Resolution level (0-40)
        unit: 'm' or 'km' for length; area will be 'm^2' or 'km^2'
    
    Returns:
        tuple: (num_cells, edge_length_in_unit, cell_area_in_unit_squared)
    """
    # normalize and validate unit
    unit = unit.strip().lower()
    if unit not in {"m", "km"}:
        raise ValueError("unit must be one of {'m','km'}")
    
    base_cells = 20 # Icosahedron faces
    num_cells = base_cells * (7**res)
    avg_cell_area_km2 = AUTHALIC_AREA / num_cells
    avg_edge_len_km = math.sqrt((2 * avg_cell_area_km2) / (3 * math.sqrt(3)))

    if res == 0: # icosahedron faces
        avg_edge_len_km = math.sqrt((4 * avg_cell_area_km2) / math.sqrt(3))

    # Convert to requested unit
    if unit == "m":
        avg_edge_len = avg_edge_len_km * 1000
        avg_cell_area = avg_cell_area_km2 * (10**6)
    else:  # unit == "m"
        avg_edge_len = avg_edge_len_km
        avg_cell_area = avg_cell_area_km2

    return num_cells, avg_edge_len, avg_cell_area 


def isea3hstats(unit: str = "m"):
    """
    Generate statistics for ISEA3H DGGS cells.
    
    Args:
        unit: 'm' or 'km' for length; area will be 'm^2' or 'km^2'
    
    Returns:
        pandas.DataFrame: DataFrame containing ISEA3H DGGS statistics with columns:
            - Resolution: Resolution level (0-40)
            - Number_of_Cells: Number of cells at each resolution
            - Avg_Edge_Length_{unit}: Average edge length in the given unit
            - Avg_Cell_Area_{unit}2: Average cell area in the squared unit
    """
    # normalize and validate unit
    unit = unit.strip().lower()
    if unit not in {"m", "km"}:
        raise ValueError("unit must be one of {'m','km'}")

    min_res = 0
    max_res = 40
    
    # Initialize lists to store data
    resolutions = []
    num_cells_list = []
    avg_edge_lens = []
    avg_cell_areas = []
    
    for res in range(min_res, max_res + 1):
        num_cells, avg_edge_len, avg_cell_area = isea3h_metrics(res, unit=unit)
        resolutions.append(res)
        num_cells_list.append(num_cells)
        avg_edge_lens.append(avg_edge_len)
        avg_cell_areas.append(avg_cell_area)
    
    # Create DataFrame
    # Build column labels with unit awareness
    avg_edge_len = f"avg_edge_len_{unit}"
    unit_area_label = {"m": "m2", "km": "km2"}[unit]
    avg_cell_area = f"avg_cell_area_{unit_area_label}"

    df = pd.DataFrame({
        'resolution': resolutions,
        'number_of_cells': num_cells_list,
        avg_edge_len: avg_edge_lens,
        avg_cell_area: avg_cell_areas
    })
    
    return df


def isea3hstats_cli(unit: str = "m"):
    """
    Command-line interface for generating ISEA3H DGGS statistics.

    CLI options:
      -unit, --unit {m,km}
    """
    if platform.system() == "Windows":
        parser = argparse.ArgumentParser(add_help=False)
        parser.add_argument('-unit', '--unit', dest='unit', choices=['m', 'km'], default=None)
        args, _ = parser.parse_known_args()

        unit = args.unit if args.unit is not None else unit

        # Get the DataFrame
        df = isea3hstats(unit=unit)
        
        # Display the DataFrame
        print(df.to_string(index=False))


def isea3hinspect(res):
    """
    Generate comprehensive inspection data for ISEA3H DGGS cells at a given resolution.
    
    This function creates a detailed analysis of ISEA3H cells including area variations,
    compactness measures, and dateline crossing detection.
    
    Args:
        res: ISEA3H resolution level (0-15)
    
    Returns:
        geopandas.GeoDataFrame: DataFrame containing ISEA3H cell inspection data with columns:
            - isea3h: ISEA3H cell ID
            - resolution: Resolution level
            - geometry: Cell geometry
            - cell_area: Cell area in square meters
            - cell_perimeter: Cell perimeter in meters
            - crossed: Whether cell crosses the dateline
            - norm_area: Normalized area (cell_area / mean_area)
            - ipq: Isoperimetric Quotient compactness
            - zsc: Zonal Standardized Compactness
    """
    if platform.system() != "Windows":
        raise RuntimeError("ISEA3H inspection is only supported on Windows due to EAGGR dependency")
    
    isea3h_gpd = isea3hgrid(res, output_format="gpd")
    isea3h_gpd['crossed'] = isea3h_gpd['geometry'].apply(check_crossing_geom)
    mean_area = isea3h_gpd['cell_area'].mean()
    # Calculate normalized area
    isea3h_gpd['norm_area'] = isea3h_gpd['cell_area'] / mean_area  
    # Calculate IPQ compactness using the standard formula: CI = 4πA/P²
    isea3h_gpd['ipq'] = 4 * np.pi * isea3h_gpd['cell_area'] / (isea3h_gpd['cell_perimeter'] ** 2)    
    # Calculate zonal standardized compactness
    isea3h_gpd['zsc'] = np.sqrt(4*np.pi*isea3h_gpd['cell_area'] - np.power(isea3h_gpd['cell_area'],2)/np.power(6378137,2))/isea3h_gpd['cell_perimeter']
    return isea3h_gpd

def isea3h_norm_area(isea3h_gpd):
    """
    Plot normalized area map for ISEA3H cells.
    
    This function creates a visualization showing how ISEA3H cell areas vary relative
    to the mean area across the globe, highlighting areas of distortion.
    
    Args:
        isea3h_gpd: GeoDataFrame from isea3hinspect function
    """
    fig, ax = plt.subplots(figsize=(10,5))
    divider = make_axes_locatable(ax)
    cax = divider.append_axes("bottom", size="5%", pad=0.1)
    vmin, vmax, vcenter = isea3h_gpd['norm_area'].min(), isea3h_gpd['norm_area'].max(), 1
    norm = TwoSlopeNorm(vmin=vmin, vcenter=vcenter, vmax=vmax)
    isea3h_gpd = isea3h_gpd[~isea3h_gpd['crossed']] # remove cells that cross the dateline
    isea3h_gpd.to_crs('proj=moll').plot(column='norm_area', ax=ax, norm=norm, legend=True,cax=cax, cmap='RdYlBu_r', legend_kwds={'label': "cell area/mean cell area",'orientation': "horizontal"})
    world_countries = gpd.read_file('https://raw.githubusercontent.com/opengeoshub/vopendata/refs/heads/main/shape/world_countries.geojson')
    world_countries.boundary.to_crs('proj=moll').plot(color=None, edgecolor='black',linewidth = 0.2,ax=ax)
    ax.axis('off')
    cb_ax = fig.axes[1] 
    cb_ax.tick_params(labelsize=14)
    cb_ax.set_xlabel(xlabel= "ISEA3H Normalized Area",fontsize=14)
    ax.margins(0)
    ax.tick_params(left=False, labelleft=False, bottom=False, labelbottom=False)
    plt.tight_layout()

def isea3h_compactness(isea3h_gpd):
    """
    Plot IPQ compactness map for ISEA3H cells.
    
    This function creates a visualization showing the Isoperimetric Quotient (IPQ)
    compactness of ISEA3H cells across the globe. IPQ measures how close each cell
    is to being circular, with values closer to 0.907 indicating more regular hexagons.
    
    Args:
        isea3h_gpd: GeoDataFrame from isea3hinspect function
    """
    fig, ax = plt.subplots(figsize=(10,5))
    divider = make_axes_locatable(ax)
    cax = divider.append_axes("bottom", size="5%", pad=0.1)
    # vmin, vmax, vcenter = isea3h_gpd['ipq'].min(), isea3h_gpd['ipq'].max(), np.mean([isea3h_gpd['ipq'].min(), isea3h_gpd['ipq'].max()])
    norm = TwoSlopeNorm(vmin=VMIN_HEX, vcenter=VCENTER_HEX, vmax=VMAX_HEX)
    isea3h_gpd = isea3h_gpd[~isea3h_gpd['crossed']] # remove cells that cross the dateline
    isea3h_gpd.to_crs('proj=moll').plot(column='ipq', ax=ax, norm=norm, legend=True,cax=cax, cmap='viridis', legend_kwds={'orientation': "horizontal" }) 
    world_countries = gpd.read_file('https://raw.githubusercontent.com/opengeoshub/vopendata/refs/heads/main/shape/world_countries.geojson')
    world_countries.boundary.to_crs('proj=moll').plot(color=None, edgecolor='black',linewidth = 0.2,ax=ax)
    ax.axis('off')
    cb_ax = fig.axes[1] 
    cb_ax.tick_params(labelsize=14)
    cb_ax.set_xlabel(xlabel= "ISEA3H IPQ Compactness",fontsize=14)
    ax.margins(0)
    ax.tick_params(left=False, labelleft=False, bottom=False, labelbottom=False)
    plt.tight_layout()

def isea3hinspect_cli():
    """
    Command-line interface for ISEA3H cell inspection.
    
    CLI options:
      -r, --resolution: ISEA3H resolution level (0-15)
    """
    if platform.system() != "Windows":
        print("ISEA3H inspection is only supported on Windows due to EAGGR dependency")
        return
        
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument('-r', '--resolution', dest='resolution', type=int, default=None)
    args, _ = parser.parse_known_args()
    res = args.resolution if args.resolution is not None else 0
    print(isea3hinspect(res))


if __name__ == "__main__":
    isea3hstats_cli()
