#!/usr/bin/env python3
import os
import json
import time
import string
import random
import secrets
import hashlib
import requests

from datetime import datetime, timedelta

class Glickeys:

    server = 'https://api.github.com/gists'

    def __init__(self, token: str = None):
        self.token = token
        self.session = requests.Session()
        self.session.headers.update({
            'Accept': 'application/json',
            'Authorization': f'Bearer {self.token}',
            'Content-Type': 'application/json',
            'User-Agent': 'Mozilla/5.0'
        })

    def all_file(self):
        response = self.session.get(self.server)
        response.raise_for_status()
        files = []
        for gist in response.json():
            owner = gist['owner']['login']
            gist_id = gist['id']
            raw_url = f"https://gist.githubusercontent.com/{owner}/{gist_id}/raw"
            first_file_key = list(gist['files'].keys())[0]
            files.append({
                'id': gist_id,
                'owner': owner,
                'public': gist['public'],
                'filename': gist['files'][first_file_key]['filename'],
                'description': gist['description'],
                'created': gist['created_at'],
                'updated': gist['updated_at'],
                'raw_url': raw_url
            })
        return files

    def file_name(self, id: str):
        response = self.session.get(f"{self.server}/{id}")
        response.raise_for_status()
        gist_data = response.json()
        return list(gist_data['files'].keys())[0]

    def create_file(self, public=False, content=None, filename=None, description=''):
        if content is None:
            content = []
        if filename is None:
            filename = secrets.token_hex(16)
        if isinstance(content, (list, dict)):
            content = json.dumps(content, indent=4)
        data = {
            'description': description,
            'public': public,
            'files': {filename: {'content': content}}
        }
        response = self.session.post(self.server, json=data)
        response.raise_for_status()
        gist_response = response.json()
        owner = gist_response['owner']['login']
        gist_id = gist_response['id']
        raw_url = f"https://gist.githubusercontent.com/{owner}/{gist_id}/raw"
        return {
            'id': gist_id,
            'owner': owner,
            'public': gist_response['public'],
            'filename': list(gist_response['files'].keys())[0],
            'description': gist_response['description'],
            'created': gist_response['created_at'],
            'updated': gist_response['updated_at'],
            'raw_url': raw_url
        }

    def read_file(self, id: str, filename: str):
        response = self.session.get(f"{self.server}/{id}")
        response.raise_for_status()
        gist_data = response.json()
        file_content = gist_data['files'][filename]['content']
        try:
            return json.loads(file_content)
        except json.JSONDecodeError:
            return file_content

    def update_data(self, id: str, data: dict, filename: str):
        payload = {'files': {filename: {'content': json.dumps(data, indent=4)}}}
        response = self.session.patch(f"{self.server}/{id}", json=payload)
        response.raise_for_status()
        updated = response.json()
        return updated['files'][filename]['content']

    def create_key(self, id: str, user: str, machine: str, premium=False, expired=0, filename=None):
        key = '-'.join(''.join(random.choices(string.ascii_uppercase, k=4)) for _ in range(4))
        created = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        expiry = (datetime.now() + timedelta(days=expired)).strftime('%Y-%m-%d %H:%M:%S')
        newdata = {
            'key': key,
            'user': user,
            'premium': premium,
            'created': created,
            'expired': expiry,
            'machine': machine
        }
        if filename is None:
            filename = self.file_name(id)
        content = self.read_file(id, filename)
        if not isinstance(content, list):
            content = []
        content.append(newdata)
        return self.update_data(id, content, filename)

    def delete_gist(self, gist_id: str):
        response = self.session.delete(f"{self.server}/{gist_id}")
        response.raise_for_status()
        return f"Gist {gist_id} deleted."

    def delete_file(self, gist_id: str, filename: str):
        response = self.session.get(f"{self.server}/{gist_id}")
        response.raise_for_status()
        gist_data = response.json()
        if filename not in gist_data['files']:
            raise Exception("Filename not found in the specified gist.")
        payload = {'files': {filename: None}}
        patch_response = self.session.patch(f"{self.server}/{gist_id}", json=payload)
        patch_response.raise_for_status()
        return f"File {filename} in Gist {gist_id} deleted."

    def change_privacy(self, gist_id: str):
        response = self.session.get(f"{self.server}/{gist_id}")
        response.raise_for_status()
        gist_data = response.json()
        current_public_status = gist_data['public']
        new_public_status = not current_public_status
        payload = {
            'description': gist_data['description'],
            'public': new_public_status,
            'files': {fname: {'content': fdata['content']} for fname, fdata in gist_data['files'].items()}
        }
        patch_response = self.session.patch(f"{self.server}/{gist_id}", json=payload)
        patch_response.raise_for_status()
        status = "public" if new_public_status else "private"
        return f"Gist {gist_id} set to {status}."

class Glickey:

    link = 'https://timeapi.io/api/Time/current/zone?timeZone=Asia/Jakarta'
    path = ['/bin', '/etc', '/lib', '/root', '/sbin', '/usr', '/var']

    def __init__(self, gist: str = None):
        self.gist = gist

    def machine(self):
        machine = []
        for line in self.path:
            try:
                stat = os.stat(line)
                machine.append(str(stat.st_ino))
            except Exception:
                pass
        return hashlib.sha256(''.join(machine).encode()).hexdigest()

    def verify_key(self, key: str):
        response = requests.get(f'{self.gist}?nocache={int(time.time())}', headers={'Accept': 'application/json', 'User-Agent': 'Mozilla/5.0', 'Cache-Control': 'no-cache'}, allow_redirects=True)
        response.raise_for_status()
        license_data = response.json()
        if isinstance(license_data, dict) and 'files' in license_data:
            file_content = next(iter(license_data['files'].values()))['content']
            try:
                license_data = json.loads(file_content)
            except json.JSONDecodeError:
                raise Exception("Invalid license file format")
        machine_id = self.machine()
        time_response = requests.get(self.link, headers={'Accept': 'application/json', 'User-Agent': 'Mozilla/5.0'})
        time_response.raise_for_status()
        time_data = time_response.json()
        now = datetime(time_data['year'], time_data['month'], time_data['day'], time_data['hour'], time_data['minute'], time_data['seconds'])
        for data in license_data:
            if data.get('key') == key and data.get('machine') == machine_id:
                expiry = datetime.strptime(data.get('expired'), '%Y-%m-%d %H:%M:%S')
                return {'status': now < expiry, 'license': data, 'message': 'License key is valid' if now < expiry else 'License key has expired'}
        return {'status': False, 'license': key, 'message': 'License key is not valid'}