#ifndef AMGCL_BACKEND_BLOCK_CRS_HPP
#define AMGCL_BACKEND_BLOCK_CRS_HPP

/*
The MIT License

Copyright (c) 2012-2015 Denis Demidov <dennis.demidov@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/**
 * \file   amgcl/backend/block_crs.hpp
 * \author Denis Demidov <dennis.demidov@gmail.com>
 * \brief  Sparse matrix in block-CRS format.
 */

#include <boost/range/algorithm.hpp>
#include <boost/range/numeric.hpp>

#include <amgcl/util.hpp>
#include <amgcl/backend/interface.hpp>
#include <amgcl/backend/builtin.hpp>
#include <amgcl/solver/skyline_lu.hpp>

namespace amgcl {
namespace backend {

/// Sparse matrix in Block CRS format.
/**
 * \param V Value type.
 * \param C Column number type.
 * \param P Index type.
 */
template < typename V, typename C, typename P >
struct bcrs {
    typedef V val_type;
    typedef C col_type;
    typedef P ptr_type;

    size_t block_size;
    size_t nrows, ncols;
    size_t brows, bcols;

    std::vector<ptr_type> ptr;
    std::vector<col_type> col;
    std::vector<val_type> val;

    /// Converts matrix in CRS format to Block CRS format.
    /**
     * \param A          Input matrix.
     * \param block_size Block size.
     *
     * \note Input matrix dimensions are *not* required to be divisible by
     * block_size.
     */
    template < class Matrix >
    bcrs(const Matrix &A, size_t block_size)
        : block_size(block_size), nrows( rows(A) ), ncols( cols(A) ),
          brows((nrows + block_size - 1) / block_size),
          bcols((ncols + block_size - 1) / block_size),
          ptr(brows + 1, 0)
    {
#pragma omp parallel
        {
            std::vector<ptrdiff_t> marker(bcols, -1);

#ifdef _OPENMP
            int nt  = omp_get_num_threads();
            int tid = omp_get_thread_num();

            size_t chunk_size  = (brows + nt - 1) / nt;
            size_t chunk_start = tid * chunk_size;
            size_t chunk_end   = std::min(brows, chunk_start + chunk_size);
#else
            size_t chunk_start = 0;
            size_t chunk_end   = brows;
#endif

            // Count number of nonzeros in block matrix.
            typedef typename backend::row_iterator<Matrix>::type row_iterator;
            for(size_t ib = chunk_start, ia = ib * block_size; ib < chunk_end; ++ib) {
                for(size_t k = 0; k < block_size && ia < nrows; ++k, ++ia) {
                    for(row_iterator a = backend::row_begin(A, ia); a; ++a) {
                        col_type cb = a.col() / block_size;

                        if (marker[cb] != static_cast<col_type>(ib)) {
                            marker[cb]  = static_cast<col_type>(ib);
                            ++ptr[ib + 1];
                        }
                    }
                }
            }

            boost::fill(marker, -1);

#pragma omp barrier
#pragma omp single
            {
                boost::partial_sum(ptr, ptr.begin());
                col.resize(ptr.back());
                val.resize(ptr.back() * block_size * block_size, 0);
            }

            // Fill the block matrix.
            for(size_t ib = chunk_start, ia = ib * block_size; ib < chunk_end; ++ib) {
                ptr_type row_beg = ptr[ib];
                ptr_type row_end = row_beg;

                for(size_t k = 0; k < block_size && ia < nrows; ++k, ++ia) {
                    for(row_iterator a = backend::row_begin(A, ia); a; ++a) {
                        col_type cb = a.col() / block_size;
                        col_type cc = a.col() % block_size;
                        val_type va = a.value();

                        if (marker[cb] < row_beg) {
                            marker[cb] = row_end;
                            col[row_end] = cb;
                            val[block_size * (block_size * row_end + k) + cc] = va;
                            ++row_end;
                        } else {
                            val[block_size * (block_size * marker[cb] + k) + cc] = va;
                        }
                    }
                }
            }
        }
    }
};

/// block_crs backend definition.
/**
 * \param real Value type.
 * \ingroup backends
 */
template <typename real>
struct block_crs {
    typedef real      value_type;
    typedef ptrdiff_t index_type;

    typedef bcrs<real, index_type, index_type> matrix;
    typedef typename builtin<real>::vector     vector;
    typedef solver::skyline_lu<value_type>     direct_solver;

    struct provides_row_iterator : boost::false_type {};

    /// Backend parameters.
    struct params {
        /// Block size to use with the created matrices.
        size_t block_size;

        params(size_t block_size = 4) : block_size(block_size) {}
        params(const boost::property_tree::ptree &p)
            : AMGCL_PARAMS_IMPORT_VALUE(p, block_size)
        {}
        void get(boost::property_tree::ptree &p, const std::string &path) const {
            AMGCL_PARAMS_EXPORT_VALUE(p, path, block_size);
        }
    };

    static std::string name() { return "block_crs"; }

    /// Copy matrix from builtin backend.
    static boost::shared_ptr<matrix>
    copy_matrix(boost::shared_ptr< typename backend::builtin<real>::matrix > A,
            const params &prm)
    {
        return boost::make_shared<matrix>(*A, prm.block_size);
    }

    /// Copy vector from builtin backend.
    static boost::shared_ptr<vector>
    copy_vector(const vector &x, const params&)
    {
        return boost::make_shared<vector>(x);
    }

    /// Copy vector from builtin backend.
    static boost::shared_ptr<vector>
    copy_vector(boost::shared_ptr< vector > x, const params&)
    {
        return x;
    }

    /// Create vector of the specified size.
    static boost::shared_ptr<vector>
    create_vector(size_t size, const params&)
    {
        return boost::make_shared<vector>(size);
    }

    static boost::shared_ptr<direct_solver>
    create_solver(
            boost::shared_ptr< typename backend::builtin<real>::matrix > A,
            const params&)
    {
        return boost::make_shared<direct_solver>(*A);
    }
};

//---------------------------------------------------------------------------
// Specialization of backend interface
//---------------------------------------------------------------------------
template < typename V, typename C, typename P >
struct value_type< bcrs<V, C, P> > {
    typedef V type;
};

template < typename V, typename C, typename P >
struct rows_impl< bcrs<V, C, P> > {
    static size_t get(const bcrs<V, C, P> &A) {
        return A.nrows;
    }
};

template < typename V, typename C, typename P >
struct cols_impl< bcrs<V, C, P> > {
    static size_t get(const bcrs<V, C, P> &A) {
        return A.ncols;
    }
};

template < typename V, typename C, typename P >
struct nonzeros_impl< bcrs<V, C, P> > {
    static size_t get(const bcrs<V, C, P> &A) {
        return A.ptr.back() * A.block_size * A.block_size;
    }
};

template < typename V, typename C, typename P, class Vec1, class Vec2 >
struct spmv_impl< bcrs<V, C, P>, Vec1, Vec2 >
{
    typedef bcrs<V, C, P>  matrix;

    static void apply(V alpha, const matrix &A, const Vec1 &x,
            V beta, Vec2 &y)
    {
        const size_t nb  = A.brows;
        const size_t na  = A.nrows;
        const size_t ma  = A.ncols;
        const size_t b1 = A.block_size;
        const size_t b2 = b1 * b1;

        if (beta) {
            if (beta != 1) {
#pragma omp parallel for
                for(ptrdiff_t i = 0; i < static_cast<ptrdiff_t>(na); ++i) {
                    y[i] *= beta;
                }
            }
        } else {
            backend::clear(y);
        }

#pragma omp parallel for
        for(ptrdiff_t ib = 0; ib < static_cast<ptrdiff_t>(nb); ++ib) {
            for(P jb = A.ptr[ib], eb = A.ptr[ib + 1]; jb < eb; ++jb) {
                size_t x0 = A.col[jb] * b1;
                size_t y0 = ib * b1;
                block_prod(b1, std::min(b1, ma - x0), std::min(b1, na - y0),
                        alpha, &A.val[jb * b2], &x[x0], &y[y0]
                        );
            }
        }
    }

    static void block_prod(size_t dim, size_t nx, size_t ny,
            V alpha, const V *A, const V *x, V *y)
    {
        for(size_t i = 0; i < ny; ++i, ++y) {
            const V * xx = x;
            V sum = 0;
            for(size_t j = 0; j < dim; ++j, ++A, ++xx)
                if (j < nx) sum += (*A) * (*xx);
            *y += alpha * sum;
        }
    }
};

template < typename V, typename C, typename P, class Vec1, class Vec2, class Vec3 >
struct residual_impl< bcrs<V, C, P>, Vec1, Vec2, Vec3 >
{
    typedef bcrs<V, C, P>  matrix;

    static void apply(const Vec1 &rhs, const matrix &A, const Vec2 &x, Vec3 &r)
    {
        backend::copy(rhs, r);
        backend::spmv(-1, A, x, 1, r);
    }
};

} // namespace backend
} // namespace amgcl

#endif
