"""
Student Update Helper - Python Implementation

This module provides a comprehensive Python implementation
that helps instructors assist students with repository updates, template syncing,
git operations, and conflict resolution.

Key Features:
- Check student repository status against template/classroom
- Help individual students with updates
- Batch process multiple students
- Automatic conflict resolution preserving student work
- Generate update instructions for students
- Support for both classroom and direct template modes

Author: Classroom Pilot Team
"""

import os
import shutil
import subprocess
import tempfile
import time
from dataclasses import dataclass
from enum import Enum
from pathlib import Path
from typing import List, Optional
from urllib.parse import urlparse

import typer
from rich.console import Console
from rich.panel import Panel
from rich.progress import Progress
from rich.table import Table

from ..config.global_config import get_global_config, load_global_config
from ..utils.git import GitManager
from ..utils.logger import get_logger

# Module-level logger for backward compatibility
logger = get_logger("assignments.student_helper")


class UpdateMode(Enum):
    """Available update modes."""
    SINGLE = "single"
    ONE_STUDENT = "one-student"
    BATCH = "batch"
    STATUS = "status"
    CHECK_CLASSROOM = "check-classroom"
    INSTRUCTIONS = "instructions"


class OperationResult(Enum):
    """Result of update operation."""
    SUCCESS = "success"
    UP_TO_DATE = "up_to_date"
    ACCESS_ERROR = "access_error"
    CONFLICT_ERROR = "conflict_error"
    GENERAL_ERROR = "general_error"


@dataclass
class StudentStatus:
    """Status information for a student repository."""
    student_name: str
    repo_url: str
    accessible: bool
    needs_update: bool
    student_commit: Optional[str] = None
    template_commit: Optional[str] = None
    classroom_commit: Optional[str] = None
    error_message: Optional[str] = None


@dataclass
class UpdateResult:
    """Result of an update operation."""
    student_name: str
    repo_url: str
    result: OperationResult
    message: str
    backup_branch: Optional[str] = None
    work_dir: Optional[Path] = None


@dataclass
class BatchSummary:
    """Summary of batch processing operation."""
    total_processed: int
    successful: int
    up_to_date: int
    errors: int
    results: List[UpdateResult]


class StudentUpdateHelper:
    """
    Helper for instructor assistance with student repository updates.

    Provides functionality to check status, apply updates, resolve conflicts,
    and assist students with template synchronization.
    """

    def __init__(self, config_file: Optional[Path] = None, auto_confirm: bool = False):
        """Initialize the student helper with configuration."""
        self.logger = get_logger(__name__)
        self.console = Console()
        self.auto_confirm = auto_confirm

        # Load global configuration
        if config_file:
            try:
                load_global_config(str(config_file))
            except FileNotFoundError:
                self.logger.warning(
                    f"Configuration file not found: {config_file}")

        self.global_config = get_global_config()
        self.config_file = config_file or Path.cwd() / "assignment.conf"

        # Initialize git manager
        self.git = GitManager()

        # Configuration
        self.template_remote = "origin"
        self.classroom_remote = "classroom"
        self.branch = "main"
        self.temp_dir = Path(tempfile.gettempdir()) / "student-helper"

        # Ensure temp directory exists
        self.temp_dir.mkdir(exist_ok=True)

    def validate_configuration(self) -> bool:
        """Validate that required configuration is present."""
        if not self.global_config:
            self.logger.error("Global configuration not loaded")
            return False

        required_fields = ['github_organization',
                           'template_repo_url', 'assignment_name']
        for field in required_fields:
            if not getattr(self.global_config, field, None):
                self.logger.error(
                    f"Required configuration field missing: {field}")
                return False

        return True

    def extract_student_name(self, repo_url: str) -> str:
        """Extract student name from repository URL."""
        if not self.global_config or not self.global_config.assignment_name:
            # Fallback parsing
            path = urlparse(repo_url).path.strip('/')
            parts = path.split('/')
            if len(parts) >= 2:
                repo_name = parts[-1].replace('.git', '')
                if '-' in repo_name:
                    return repo_name.split('-', 1)[1]
            return "unknown"

        assignment_name = self.global_config.assignment_name
        path = urlparse(repo_url).path.strip('/')
        parts = path.split('/')

        if len(parts) >= 2:
            repo_name = parts[-1].replace('.git', '')
            prefix = f"{assignment_name}-"
            if repo_name.startswith(prefix):
                return repo_name[len(prefix):]

        return "unknown"

    def validate_repo_url(self, repo_url: str) -> bool:
        """Validate repository URL format."""
        if not self.global_config:
            self.logger.warning(
                "Cannot validate URL - no configuration loaded")
            return True  # Allow validation to pass for basic functionality

        org = self.global_config.github_organization
        assignment = self.global_config.assignment_name

        expected_pattern = f"https://github.com/{org}/.*{assignment}-.*"

        import re
        if not re.match(expected_pattern.replace('.*', '.*'), repo_url):
            self.logger.error("Invalid repository URL format")
            self.logger.error(
                f"Expected: https://github.com/{org}/{assignment}-[student-name]")
            return False

        return True

    def confirm_action(self, prompt: str) -> bool:
        """Confirm action with user, respecting auto-confirm mode."""
        if self.auto_confirm:
            self.console.print(f"{prompt} [auto-confirmed: Y]", style="yellow")
            return True

        return typer.confirm(prompt)

    def check_repo_access(self, repo_url: str) -> bool:
        """Check if repository is accessible."""
        try:
            result = subprocess.run(
                ['git', 'ls-remote', repo_url],
                capture_output=True,
                text=True,
                timeout=30
            )
            return result.returncode == 0
        except (subprocess.TimeoutExpired, subprocess.SubprocessError):
            return False

    def get_remote_commit(self, repo_url: str, ref: str = "refs/heads/main") -> Optional[str]:
        """Get the latest commit from a remote repository."""
        try:
            result = subprocess.run(
                ['git', 'ls-remote', repo_url, ref],
                capture_output=True,
                text=True,
                timeout=30
            )
            if result.returncode == 0 and result.stdout.strip():
                return result.stdout.split()[0]
        except (subprocess.TimeoutExpired, subprocess.SubprocessError):
            pass
        return None

    def check_student_status(self, repo_url: str) -> StudentStatus:
        """Check the status of a student repository."""
        student_name = self.extract_student_name(repo_url)

        self.logger.info(f"Checking status for student: {student_name}")

        # Check repository access
        if not self.check_repo_access(repo_url):
            return StudentStatus(
                student_name=student_name,
                repo_url=repo_url,
                accessible=False,
                needs_update=False,
                error_message="Cannot access student repository"
            )

        # Get commit information
        student_commit = self.get_remote_commit(repo_url)
        template_commit = None
        classroom_commit = None

        # Get template commit
        if self.global_config and self.global_config.template_repo_url:
            template_commit = self.get_remote_commit(
                self.global_config.template_repo_url)

        # Get classroom commit if available
        if self.global_config and self.global_config.classroom_repo_url:
            classroom_commit = self.get_remote_commit(
                self.global_config.classroom_repo_url)

        # Determine if update is needed
        needs_update = True
        if classroom_commit and student_commit == classroom_commit:
            needs_update = False
        elif template_commit and student_commit == template_commit:
            needs_update = False

        return StudentStatus(
            student_name=student_name,
            repo_url=repo_url,
            accessible=True,
            needs_update=needs_update,
            student_commit=student_commit,
            template_commit=template_commit,
            classroom_commit=classroom_commit
        )

    def display_student_status(self, status: StudentStatus) -> None:
        """Display student status in a nice format."""
        table = Table(title=f"Status for Student: {status.student_name}")
        table.add_column("Property", style="bold blue")
        table.add_column("Value", style="green")

        table.add_row("Repository", status.repo_url)
        table.add_row("Accessible", "✅ Yes" if status.accessible else "❌ No")

        if status.accessible:
            table.add_row("Needs Update",
                          "⚠️ Yes" if status.needs_update else "✅ No")
            if status.student_commit:
                table.add_row("Student Commit", status.student_commit[:8])
            if status.template_commit:
                table.add_row("Template Commit", status.template_commit[:8])
            if status.classroom_commit:
                table.add_row("Classroom Commit", status.classroom_commit[:8])
        else:
            table.add_row("Error", status.error_message or "Unknown error")

        self.console.print(table)

    def check_classroom_ready(self) -> bool:
        """Check if classroom repository is ready for updates."""
        if not self.global_config or not self.global_config.classroom_repo_url:
            self.logger.warning("No classroom repository URL configured")
            return False

        classroom_url = self.global_config.classroom_repo_url

        self.console.print(
            Panel("📋 Checking Classroom Repository Status", style="blue"))

        # Check accessibility
        if not self.check_repo_access(classroom_url):
            self.console.print(
                "❌ Cannot access classroom repository", style="red")
            self.console.print(f"URL: {classroom_url}", style="dim")
            return False

        self.console.print(
            "✅ Classroom repository is accessible", style="green")

        # Compare commits
        classroom_commit = self.get_remote_commit(classroom_url)
        template_commit = None

        if self.global_config.template_repo_url:
            template_commit = self.get_remote_commit(
                self.global_config.template_repo_url)

        table = Table()
        table.add_column("Repository", style="bold")
        table.add_column("Commit", style="cyan")

        if classroom_commit:
            table.add_row("Classroom", classroom_commit[:8])
        if template_commit:
            table.add_row("Template", template_commit[:8])

        self.console.print(table)

        if classroom_commit and template_commit:
            if classroom_commit == template_commit:
                self.console.print(
                    "✅ Classroom repository is up to date", style="green")
                return True
            else:
                self.console.print(
                    "⚠️ Classroom repository may need updates", style="yellow")
                self.console.print(
                    "Consider running template push operations", style="dim")
                return True

        return True

    def help_single_student(self, repo_url: str, use_template_direct: bool = False) -> UpdateResult:
        """Help a single student with repository updates."""
        student_name = self.extract_student_name(repo_url)
        work_dir = self.temp_dir / student_name

        self.console.print(
            Panel(f"🔧 Helping Student: {student_name}", style="blue"))

        # Validate URL
        if not self.validate_repo_url(repo_url):
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.GENERAL_ERROR,
                message="Invalid repository URL format"
            )

        # Check status first
        status = self.check_student_status(repo_url)
        if not status.accessible:
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.ACCESS_ERROR,
                message=status.error_message or "Cannot access repository"
            )

        if not status.needs_update:
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.UP_TO_DATE,
                message="Student is already up to date"
            )

        # Confirm action
        if not self.confirm_action("Clone and update the student's repository?"):
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.GENERAL_ERROR,
                message="Operation cancelled by user"
            )

        try:
            # Setup work directory
            if work_dir.exists():
                shutil.rmtree(work_dir)
            work_dir.mkdir(parents=True)

            # Clone student repository
            self.logger.info("Cloning student repository...")
            clone_result = subprocess.run(
                ['git', 'clone', repo_url, str(work_dir)],
                capture_output=True,
                text=True
            )

            if clone_result.returncode != 0:
                return UpdateResult(
                    student_name=student_name,
                    repo_url=repo_url,
                    result=OperationResult.GENERAL_ERROR,
                    message=f"Failed to clone repository: {clone_result.stderr}"
                )

            # Change to work directory
            original_cwd = os.getcwd()
            os.chdir(work_dir)

            try:
                # Add upstream remote
                upstream_url = (
                    self.global_config.template_repo_url if use_template_direct
                    else self.global_config.classroom_repo_url
                )

                if not upstream_url:
                    upstream_url = self.global_config.template_repo_url

                if not upstream_url:
                    return UpdateResult(
                        student_name=student_name,
                        repo_url=repo_url,
                        result=OperationResult.GENERAL_ERROR,
                        message="No template or classroom repository URL configured"
                    )

                subprocess.run(
                    ['git', 'remote', 'add', 'upstream', upstream_url], check=True)
                subprocess.run(['git', 'fetch', 'upstream'], check=True)

                # Create backup branch
                backup_branch = f"backup-before-update-{int(time.time())}"
                subprocess.run(
                    ['git', 'checkout', '-b', backup_branch], check=True)
                subprocess.run(['git', 'checkout', self.branch], check=True)

                # Try to merge updates
                merge_result = subprocess.run(
                    ['git', 'merge', f'upstream/{self.branch}',
                        '--no-edit', '--allow-unrelated-histories'],
                    capture_output=True,
                    text=True
                )

                if merge_result.returncode == 0:
                    # Successful merge
                    subprocess.run(
                        ['git', 'push', 'origin', self.branch], check=True)
                    subprocess.run(
                        ['git', 'push', 'origin', backup_branch], check=True)

                    return UpdateResult(
                        student_name=student_name,
                        repo_url=repo_url,
                        result=OperationResult.SUCCESS,
                        message="Updates applied successfully",
                        backup_branch=backup_branch,
                        work_dir=work_dir
                    )
                else:
                    # Handle merge conflicts
                    return self._handle_merge_conflicts(
                        student_name, repo_url, backup_branch, work_dir
                    )

            finally:
                os.chdir(original_cwd)

        except subprocess.CalledProcessError as e:
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.GENERAL_ERROR,
                message=f"Git operation failed: {e}"
            )
        except Exception as e:
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.GENERAL_ERROR,
                message=f"Unexpected error: {e}"
            )

    def _handle_merge_conflicts(
        self,
        student_name: str,
        repo_url: str,
        backup_branch: str,
        work_dir: Path
    ) -> UpdateResult:
        """Handle merge conflicts with automatic resolution."""
        self.logger.warning(
            "Merge conflicts detected, attempting automatic resolution...")

        try:
            # Abort the failed merge
            subprocess.run(['git', 'merge', '--abort'], capture_output=True)

            # Try merge with strategy favoring upstream changes
            merge_result = subprocess.run(
                ['git', 'merge', f'upstream/{self.branch}', '--no-edit',
                    '--allow-unrelated-histories', '-X', 'theirs'],
                capture_output=True,
                text=True
            )

            if merge_result.returncode == 0:
                # Restore student's protected files and folders
                self._restore_student_files(backup_branch)

                # Check if there are changes to commit
                status_result = subprocess.run(
                    ['git', 'status', '--porcelain'],
                    capture_output=True,
                    text=True
                )

                if status_result.stdout.strip():
                    subprocess.run(['git', 'add', '.'], check=True)
                    subprocess.run(
                        ['git', 'commit', '-m', 'Preserve student work after template update'], check=True)

                # Push changes
                subprocess.run(
                    ['git', 'push', 'origin', self.branch], check=True)
                subprocess.run(
                    ['git', 'push', 'origin', backup_branch], check=True)

                return UpdateResult(
                    student_name=student_name,
                    repo_url=repo_url,
                    result=OperationResult.SUCCESS,
                    message="Updates applied with automatic conflict resolution",
                    backup_branch=backup_branch,
                    work_dir=work_dir
                )
            else:
                return UpdateResult(
                    student_name=student_name,
                    repo_url=repo_url,
                    result=OperationResult.CONFLICT_ERROR,
                    message="Automatic conflict resolution failed - manual intervention required",
                    backup_branch=backup_branch,
                    work_dir=work_dir
                )

        except subprocess.CalledProcessError as e:
            return UpdateResult(
                student_name=student_name,
                repo_url=repo_url,
                result=OperationResult.CONFLICT_ERROR,
                message=f"Conflict resolution failed: {e}",
                backup_branch=backup_branch,
                work_dir=work_dir
            )

    def _restore_student_files(self, backup_branch: str) -> None:
        """
        Restore student's protected files and folders from backup branch.

        Supports:
        - Individual files (e.g., "assignment.ipynb")
        - Glob patterns (e.g., "*.py", "data/*.csv")  
        - Folders (e.g., "student_work/", "outputs/")

        Args:
            backup_branch: Name of the backup branch containing student work
        """
        import glob

        student_files = self.global_config.get_student_files()

        for file_pattern in student_files:
            file_pattern = file_pattern.strip()
            if not file_pattern:
                continue

            try:
                # Check if it's a folder (ends with /)
                if file_pattern.endswith('/'):
                    # Restore entire folder
                    result = subprocess.run(
                        ['git', 'checkout', backup_branch, '--', file_pattern],
                        capture_output=True,
                        text=True
                    )
                    if result.returncode == 0:
                        self.logger.debug(f"Restored folder: {file_pattern}")
                    else:
                        self.logger.warning(
                            f"Could not restore folder {file_pattern}: {result.stderr}")

                # Check if it contains wildcards (glob pattern)
                elif '*' in file_pattern or '?' in file_pattern or '[' in file_pattern:
                    # Expand glob pattern and restore matching files
                    matching_files = glob.glob(file_pattern)
                    if matching_files:
                        for matched_file in matching_files:
                            result = subprocess.run(
                                ['git', 'checkout', backup_branch,
                                    '--', matched_file],
                                capture_output=True,
                                text=True
                            )
                            if result.returncode == 0:
                                self.logger.debug(
                                    f"Restored file: {matched_file}")
                            else:
                                self.logger.warning(
                                    f"Could not restore file {matched_file}: {result.stderr}")
                    else:
                        self.logger.debug(
                            f"No files matched pattern: {file_pattern}")

                else:
                    # Restore specific file
                    result = subprocess.run(
                        ['git', 'checkout', backup_branch, '--', file_pattern],
                        capture_output=True,
                        text=True
                    )
                    if result.returncode == 0:
                        self.logger.debug(f"Restored file: {file_pattern}")
                    else:
                        self.logger.warning(
                            f"Could not restore file {file_pattern}: {result.stderr}")

            except Exception as e:
                self.logger.warning(f"Error restoring {file_pattern}: {e}")

    def batch_help_students(self, repo_file: Path) -> BatchSummary:
        """Process multiple students from a file."""
        if not repo_file.exists():
            raise FileNotFoundError(f"Repository file not found: {repo_file}")

        self.console.print(Panel("🔄 Batch Processing Students", style="blue"))

        # Read repository URLs
        repo_urls = []
        with open(repo_file, 'r') as f:
            for line in f:
                line = line.strip()
                if line and not line.startswith('#') and line.startswith('https://'):
                    repo_urls.append(line)

        if not repo_urls:
            raise ValueError("No valid repository URLs found in file")

        self.logger.info(f"Found {len(repo_urls)} repositories to process")

        if not self.confirm_action(f"Process all {len(repo_urls)} repositories?"):
            raise ValueError("Batch processing cancelled by user")

        # Process each repository
        results = []
        success_count = 0
        up_to_date_count = 0
        error_count = 0

        with Progress() as progress:
            task = progress.add_task(
                "Processing students...", total=len(repo_urls))

            for i, repo_url in enumerate(repo_urls, 1):
                progress.update(
                    task, description=f"Processing {i}/{len(repo_urls)}")

                # Check status first
                status = self.check_student_status(repo_url)

                if not status.accessible:
                    result = UpdateResult(
                        student_name=status.student_name,
                        repo_url=repo_url,
                        result=OperationResult.ACCESS_ERROR,
                        message="Repository not accessible"
                    )
                    error_count += 1
                elif not status.needs_update:
                    result = UpdateResult(
                        student_name=status.student_name,
                        repo_url=repo_url,
                        result=OperationResult.UP_TO_DATE,
                        message="Already up to date"
                    )
                    up_to_date_count += 1
                else:
                    # Try to help the student
                    result = self.help_single_student(repo_url)
                    if result.result == OperationResult.SUCCESS:
                        success_count += 1
                    else:
                        error_count += 1

                results.append(result)
                progress.advance(task)

        return BatchSummary(
            total_processed=len(repo_urls),
            successful=success_count,
            up_to_date=up_to_date_count,
            errors=error_count,
            results=results
        )

    def display_batch_summary(self, summary: BatchSummary) -> None:
        """Display batch processing summary."""
        table = Table(title="Batch Processing Summary")
        table.add_column("Metric", style="bold")
        table.add_column("Count", style="cyan")

        table.add_row("Total Processed", str(summary.total_processed))
        table.add_row("Successfully Updated", str(summary.successful))
        table.add_row("Already Up to Date", str(summary.up_to_date))
        table.add_row("Errors", str(summary.errors))

        self.console.print(table)

        # Show individual results if there were errors
        if summary.errors > 0:
            error_table = Table(title="Error Details")
            error_table.add_column("Student", style="bold")
            error_table.add_column("Result", style="red")
            error_table.add_column("Message", style="dim")

            for result in summary.results:
                if result.result in [OperationResult.ACCESS_ERROR, OperationResult.CONFLICT_ERROR, OperationResult.GENERAL_ERROR]:
                    error_table.add_row(
                        result.student_name,
                        result.result.value,
                        result.message
                    )

            self.console.print(error_table)

    def generate_student_instructions(self, repo_url: str) -> str:
        """Generate update instructions for a student."""
        student_name = self.extract_student_name(repo_url)

        classroom_url = getattr(
            self.global_config, 'classroom_repo_url', 'CLASSROOM_REPO_URL')

        instructions = f"""
Update Instructions for {student_name}

Dear {student_name},

There are updates available for the assignment template. Please follow these steps to update your repository:

OPTION 1 - Automated Update (Recommended):
1. Open your terminal in your assignment directory
2. Run: classroom-pilot assignments update
3. Follow the prompts

OPTION 2 - Manual Process:
1. Save and commit your current work:
   git add .
   git commit -m "Save work before template update"

2. Add the template as a remote (one-time setup):
   git remote add upstream {classroom_url}

3. Get the updates:
   git fetch upstream
   git merge upstream/main

4. If there are conflicts, resolve them and commit:
   git add .
   git commit -m "Resolve merge conflicts"

OPTION 3 - Detailed Guide:
Follow the complete guide in: docs/UPDATE-GUIDE.md

If you encounter any issues, please:
- Check the troubleshooting section in docs/UPDATE-GUIDE.md
- Ask for help during office hours
- Contact the instructor

Best regards,
Instructional Team
"""
        return instructions.strip()

    def cleanup(self) -> None:
        """Clean up temporary files."""
        if self.temp_dir.exists():
            self.logger.info("Cleaning up temporary files...")
            shutil.rmtree(self.temp_dir, ignore_errors=True)

    def execute_update_workflow(self, auto_confirm: bool = False, verbose: bool = False) -> tuple[bool, str]:
        """
        Execute the update workflow for student repositories.

        This is the main entry point for updating student repositories. It validates
        the configuration, checks if the classroom repository is ready, and returns
        status information.

        Args:
            auto_confirm (bool): If True, automatically confirm all prompts.
            verbose (bool): Enable verbose logging output.

        Returns:
            tuple[bool, str]: Success status and descriptive message.

        Example:
            >>> helper = StudentUpdateHelper(config_path="assignment.conf")
            >>> success, message = helper.execute_update_workflow(auto_confirm=True)
            >>> if success:
            ...     print(f"Update workflow complete: {message}")
        """
        try:
            # Update auto_confirm if provided
            if auto_confirm:
                self.auto_confirm = auto_confirm

            # Validate configuration
            if not self.validate_configuration():
                return False, "Configuration validation failed"

            # Check if classroom is ready
            if not self.check_classroom_ready():
                return False, "Classroom repository not ready for updates"

            # Return success with message
            return True, "Update workflow validated successfully"

        except Exception as e:
            self.logger.error(f"Update workflow failed: {e}")
            return False, str(e)
