"""
Configuration File Generator for the GitHub Classroom Setup Wizard.

This module handles the generation of assignment configuration files
with all necessary sections and formatting.
"""

from datetime import datetime
from pathlib import Path
from typing import Dict

from ..utils.ui_components import print_header, print_success


class ConfigGenerator:
    """ 
    ConfigGenerator is responsible for generating configuration files for GitHub Classroom assignments.

    This class provides methods to assemble and write a configuration file containing assignment information,
    secret management, workflow configuration, and advanced options. It supports injecting values for assignment
    URLs, repository details, secrets, and workflow steps, and can validate and format secrets as needed.

    Args:
        config_file (Path): The path where the generated configuration file will be written.

    Methods:
        create_config_file(config_values, token_files, token_validation):
            Generates and writes the configuration file using provided assignment and secret values.

        _generate_header():
            Returns the header section for the configuration file, including a timestamp.

        _generate_assignment_section(config_values):
            Returns the assignment information section, populated with URLs and assignment metadata.

        _generate_secrets_section(config_values, token_files, token_validation):
            Returns the secrets management section, including secret definitions and validation flags.

        _generate_workflow_section(config_values):
            Returns the workflow configuration section, specifying which steps to execute.

        _generate_advanced_section():
            Returns the advanced configuration section, including repository filtering and logging options.
    """

    def __init__(self, config_file: Path):
        """
        Initializes the configuration generator with the specified configuration file.

        Args:
            config_file (Path): The path to the configuration file.
        """
        self.config_file = config_file

    def create_config_file(
        self,
        config_values: Dict[str, str],
        token_files: Dict[str, str],
        token_validation: Dict[str, bool]
    ) -> None:
        """
        Creates a configuration file for the assignment using the provided configuration values, token files, and token validation results.

        Args:
            config_values (Dict[str, str]): Dictionary containing configuration values for the assignment.
            token_files (Dict[str, str]): Dictionary mapping token names to their corresponding file paths.
            token_validation (Dict[str, bool]): Dictionary indicating the validation status of each token.

        Returns:
            None

        Side Effects:
            Writes the generated configuration content to the file specified by self.config_file.
            Prints status messages to the console.
        """
        print_header("Creating Assignment Configuration")

        config_content = self._generate_header()
        config_content += self._generate_assignment_section(config_values)
        config_content += self._generate_secrets_section(
            config_values, token_files, token_validation)
        config_content += self._generate_workflow_section(config_values)
        config_content += self._generate_advanced_section()

        # Write configuration file
        with open(self.config_file, 'w') as f:
            f.write(config_content)

        print_success(f"Configuration file created: {self.config_file}")

    def _generate_header(self) -> str:
        """
        Generates a header string for a GitHub Classroom assignment configuration file.

        The header includes metadata such as the generation timestamp and a brief description
        of the file's purpose. This helps users identify the file's origin and intent.

        Returns:
            str: The formatted header string for the configuration file.
        """
        return f"""# GitHub Classroom Assignment Configuration
# Generated by setup-assignment.py on {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
# This file contains all the necessary information to manage a GitHub Classroom assignment

"""

    def _generate_assignment_section(self, config_values: Dict[str, str]) -> str:
        """
        Generate the assignment information section for a configuration file.

        This method constructs a formatted string containing assignment-related configuration
        variables, such as the GitHub Classroom assignment URL, repository URL, template
        repository URL, organization name, assignment name, and the main assignment file.
        It uses the provided `config_values` dictionary to populate these fields, inserting
        comments and default values where appropriate.

        Args:
            config_values (Dict[str, str]): A dictionary containing configuration values
                for assignment information. Expected keys include:
                - 'CLASSROOM_URL': The GitHub Classroom assignment URL.
                - 'CLASSROOM_REPO_URL': (Optional) The repository URL for the assignment.
                - 'TEMPLATE_REPO_URL': The template repository URL.
                - 'GITHUB_ORGANIZATION': The GitHub organization name.
                - 'ASSIGNMENT_NAME': (Optional) The assignment name.
                - 'MAIN_ASSIGNMENT_FILE': The main assignment file name.

        Returns:
            str: A formatted string representing the assignment information section,
                ready to be written to a configuration file.
        """
        section = """# =============================================================================
# ASSIGNMENT INFORMATION
# =============================================================================

# GitHub Classroom assignment URL (used to extract assignment name and discover student repos)
# This is the URL you see when managing the assignment in GitHub Classroom
# Example: https://classroom.github.com/classrooms/12345/assignments/assignment-name
CLASSROOM_URL="{}"

# GitHub Classroom repository URL (OPTIONAL - only needed for template synchronization)
# This is the actual repository URL created by GitHub Classroom for the assignment
# Different from CLASSROOM_URL above - this is the git repository URL
# Format: https://github.com/[ORG]/[classroom-semester-assignment-name]
# To find this: look for a repo in your organization with a name like "classroom-fall25-assignment-name"
""".format(config_values.get('CLASSROOM_URL', ''))

        if config_values.get('CLASSROOM_REPO_URL'):
            section += f'CLASSROOM_REPO_URL="{config_values["CLASSROOM_REPO_URL"]}"\n'
        else:
            section += '# CLASSROOM_REPO_URL=""  # Optional - add if using template synchronization\n'

        section += f"""
# Template repository URL (source of truth for updates)
TEMPLATE_REPO_URL="{config_values.get('TEMPLATE_REPO_URL', '')}"

# GitHub organization name (usually extracted from URLs but can be overridden)
GITHUB_ORGANIZATION="{config_values.get('GITHUB_ORGANIZATION', '')}"

# Assignment name (auto-extracted from classroom URL if not specified)
"""

        if config_values.get('ASSIGNMENT_NAME'):
            section += f'ASSIGNMENT_NAME="{config_values["ASSIGNMENT_NAME"]}"\n'
        else:
            section += '# ASSIGNMENT_NAME=""  # Auto-extracted from template URL if not specified\n'

        section += f"""
# Student Files Protection (NEW in v3.1+)
# Protect these files/folders during template updates (comma-separated)
# Supports: specific files, glob patterns, and folder paths
# Examples:
#   Single file: "assignment.ipynb"
#   Multiple files: "assignment.ipynb,solutions.py,data.csv"
#   Folders: "student_work/,outputs/"
#   Mixed: "assignment.ipynb,student_work/,*.txt"
STUDENT_FILES="{config_values.get('STUDENT_FILES', config_values.get('MAIN_ASSIGNMENT_FILE', 'assignment.ipynb'))}"

# Legacy support (deprecated - use STUDENT_FILES instead)
# ASSIGNMENT_FILE="{config_values.get('MAIN_ASSIGNMENT_FILE', '')}"

"""

        return section

    def _generate_secrets_section(
        self,
        config_values: Dict[str, str],
        token_files: Dict[str, str],
        token_validation: Dict[str, bool]
    ) -> str:
        """
        Generate the secrets management section for a configuration file.

        This method constructs a formatted string that documents and defines the secrets
        to be added to student repositories, based on the provided configuration values,
        token files, and token validation settings. It supports two modes:
        - When secrets management is enabled (`USE_SECRETS` is 'true'), it lists the secrets
            (including instructor test tokens and any additional secrets) with their descriptions,
            file paths, maximum age, and validation requirements.
        - When secrets management is disabled, it provides commented instructions and examples
            for configuring secrets, including guidance for assignments where tests are included
            in the template repository.

        Args:
                config_values (Dict[str, str]): Configuration values, including flags and secret descriptions.
                token_files (Dict[str, str]): Mapping of secret names to their corresponding token file paths.
                token_validation (Dict[str, bool]): Mapping of secret names to their validation requirements.

        Returns:
                str: The formatted secrets management section as a string.

        """
        section = """# =============================================================================
# SECRET MANAGEMENT
# =============================================================================

"""

        # Add secrets configuration
        if config_values.get('USE_SECRETS') == 'true':
            section += """# Secrets to add to student repositories
# NEW Format (v3.1+): SECRET_NAME:description:validate_format
# Uses centralized token management - no separate token files needed!
# validate_format: true for GitHub tokens (ghp_), false for other secrets
# 
# Use this when you have a separate private instructor repository with tests
# that students need access to via GitHub secrets.
SECRETS_CONFIG="
"""

            # Add instructor tests token using new simplified format
            validation = token_validation.get('INSTRUCTOR_TESTS_TOKEN', True)
            section += f"INSTRUCTOR_TESTS_TOKEN:Token for accessing instructor test repository:{str(validation).lower()}\n"

            # Add additional secrets
            for secret_name, token_file in token_files.items():
                if secret_name != 'INSTRUCTOR_TESTS_TOKEN':
                    validation = token_validation.get(secret_name, True)
                    description = config_values.get(
                        f'{secret_name}_DESCRIPTION', f'{secret_name} for assignment functionality')
                    section += f"{secret_name}:{description}:{str(validation).lower()}\n"

            section += '"\n'
        else:
            section += """# Secrets to add to student repositories
# NEW Format (v3.1+): SECRET_NAME:description:validate_format
# Uses centralized token management - no separate token files needed!
# validate_format: true for GitHub tokens (ghp_), false for other secrets
# 
# Use this when you have a separate private instructor repository with tests
# that students need access to via GitHub secrets.
# 
# If your tests are included in the same template repository, you can:
# 1. Set STEP_MANAGE_SECRETS=false in the WORKFLOW CONFIGURATION section, OR
# 2. Leave SECRETS_CONFIG empty (comment out or set to empty string)
# 
# SECRETS_CONFIG="
# INSTRUCTOR_TESTS_TOKEN:Token for accessing instructor test repository:true
# "

# For assignments where tests are in the template repository, use:
SECRETS_CONFIG=""

# Legacy format still supported (will be automatically converted):
# OLD Format: SECRET_NAME:description:token_file_path:max_age_days:validate_format
# SECRETS_CONFIG="
# INSTRUCTOR_TESTS_TOKEN:Token for accessing instructor test repository:instructor_token.txt:90:true
# "
"""

        section += "\n"
        return section

    def _generate_workflow_section(self, config_values: Dict[str, str]) -> str:
        """
        Generate the workflow configuration section as a formatted string.

        Args:
            config_values (Dict[str, str]): A dictionary containing configuration values, 
                such as 'USE_SECRETS', to customize the workflow section.

        Returns:
            str: A string representing the workflow configuration section, with values 
                interpolated from the provided config_values.

        """
        return f"""# =============================================================================
# WORKFLOW CONFIGURATION
# =============================================================================

# Workflow steps to execute (true/false)
STEP_SYNC_TEMPLATE=true
STEP_DISCOVER_REPOS=true
STEP_MANAGE_SECRETS={config_values.get('USE_SECRETS', 'false')}   # Set to false if tests are in template repo (no separate instructor repo)
STEP_ASSIST_STUDENTS=false

# Output directory for generated files
OUTPUT_DIR="tools/generated"

"""

    def _generate_advanced_section(self) -> str:
        """
        Generate the advanced configuration section as a formatted string.

        Returns:
            str: A multi-line string containing advanced configuration options,
                including repository filtering, dry run mode, logging level,
                and confirmation prompt settings.

        """
        return """# =============================================================================
# ADVANCED CONFIGURATION
# =============================================================================

# Repository filtering
EXCLUDE_INSTRUCTOR_REPOS=true
INCLUDE_TEMPLATE_REPO=false

# Dry run mode (for testing)
DEFAULT_DRY_RUN=false

# Logging level (DEBUG, INFO, WARNING, ERROR)
LOG_LEVEL=INFO

# Confirmation prompts
SKIP_CONFIRMATIONS=false
"""
