"""jiff types"""

import datetime as pydt
import typing as t
from typing import Protocol

import typing_extensions as te

from ry._types import (
    DateTimeTypedDict,
    DateTypedDict,
    TimeSpanTypedDict,
    TimeTypedDict,
    ToPy,
)
from ry.ryo3 import Duration

_T = t.TypeVar("_T")
# =============================================================================
# JIFF
# =============================================================================
JIFF_UNIT: te.TypeAlias = t.Literal[
    "year",
    "month",
    "week",
    "day",
    "hour",
    "minute",
    "second",
    "millisecond",
    "microsecond",
    "nanosecond",
]

JIFF_ROUND_MODE: te.TypeAlias = t.Literal[
    "ceil",
    "floor",
    "expand",
    "trunc",
    "half_ceil",
    "half_floor",
    "half_expand",
    "half_trunc",
    "half_even",
]

WEEKDAY_STR: te.TypeAlias = t.Literal[
    "monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday"
]

WEEKDAY_INT: te.TypeAlias = t.Literal[
    1,  # Monday
    2,  # Tuesday
    3,  # Wednesday
    4,  # Thursday
    5,  # Friday
    6,  # Saturday
    7,  # Sunday
]

WEEKDAY: te.TypeAlias = WEEKDAY_STR | WEEKDAY_INT

class ToPyDate(Protocol):
    def to_pydate(self) -> pydt.date: ...

class ToPyTime(Protocol):
    def to_pytime(self) -> pydt.time: ...

class ToPyDateTime(Protocol):
    def to_pydatetime(self) -> pydt.datetime: ...

class ToPyTimeDelta(Protocol):
    def to_pytimedelta(self) -> pydt.timedelta: ...

class ToPyTzInfo(Protocol):
    def to_pytzinfo(self) -> pydt.tzinfo: ...

@t.final
class Date(ToPy[pydt.date], ToPyDate):
    MIN: Date
    MAX: Date
    ZERO: Date

    def __init__(self, year: int, month: int, day: int) -> None: ...

    # =========================================================================
    # STRING
    # =========================================================================
    def string(self) -> str: ...
    def isoformat(self) -> str: ...

    # =========================================================================
    # PYTHON_CONVERSIONS
    # =========================================================================
    def to_py(self) -> pydt.date: ...
    def to_pydate(self) -> pydt.date: ...
    @classmethod
    def from_pydate(cls: type[Date], date: pydt.date) -> Date: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def year(self) -> int: ...
    @property
    def month(self) -> int: ...
    @property
    def day(self) -> int: ...
    @property
    def weekday(self) -> int: ...

    # =========================================================================
    # CLASSMETHODS
    # =========================================================================
    @classmethod
    def from_iso_week_date(
        cls: type[Date], year: int, week: int, weekday: int
    ) -> Date: ...
    @classmethod
    def today(cls: type[Date]) -> Date: ...
    @classmethod
    def parse(cls: type[Date], s: str) -> Date: ...
    @classmethod
    def from_str(cls: type[Date], s: str) -> Date: ...
    # =========================================================================
    # STRPTIME/STRFTIME
    # =========================================================================
    @classmethod
    def strptime(cls: type[Date], format: str, string: str) -> Date: ...
    def strftime(self, format: str) -> str: ...

    # =========================================================================
    # OPERATORS
    # =========================================================================
    def __add__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...
    @t.overload
    def __sub__(self, other: Date) -> TimeSpan: ...
    @t.overload
    def __sub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...
    @t.overload
    def __isub__(self, other: Date) -> TimeSpan: ...
    @t.overload
    def __isub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def at(self, hour: int, minute: int, second: int, nanosecond: int) -> DateTime: ...
    def asdict(self) -> DateTypedDict: ...
    def astuple(self) -> tuple[int, int, int]: ...
    def checked_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...
    def day_of_year(self) -> int: ...
    def day_of_year_no_leap(self) -> int | None: ...
    def days_in_month(self) -> int: ...
    def days_in_year(self) -> int: ...
    def duration_since(self, other: Date) -> Date: ...
    def duration_until(self, other: Date) -> Date: ...
    def era_year(self) -> tuple[int, t.Literal["BCE", "CE"]]: ...
    def first_of_month(self) -> Date: ...
    def first_of_year(self) -> Date: ...
    def iso_week_date(self) -> ISOWeekDate: ...
    def in_leap_year(self) -> bool: ...
    def in_tz(self, tz: str) -> ZonedDateTime: ...
    def intz(self, tz: str) -> ZonedDateTime: ...
    def last_of_month(self) -> Date: ...
    def last_of_year(self) -> Date: ...
    def nth_weekday(self, nth: int, weekday: WEEKDAY) -> Date: ...
    def nth_weekday_of_month(self, nth: int, weekday: WEEKDAY) -> Date: ...
    def saturating_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...
    def saturating_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...
    def series(self, span: TimeSpan) -> JiffSeries[Date]: ...
    def to_datetime(self, t: Time) -> DateTime: ...
    def to_zoned(self, tz: TimeZone) -> ZonedDateTime: ...
    def tomorrow(self) -> Date: ...
    def yesterday(self) -> Date: ...

    # =========================================================================
    # SINCE/UNTIL
    # =========================================================================
    def _since(self, other: DateDifference) -> TimeSpan: ...
    def _until(self, other: DateDifference) -> TimeSpan: ...
    def since(
        self,
        other: Date | DateTime | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...
    def until(
        self,
        other: Date | DateTime | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...

    # =========================================================================
    # INSTANCE METHODS W/ OVERLOADS
    # =========================================================================
    @t.overload
    def checked_sub(self, other: Date) -> TimeSpan: ...
    @t.overload
    def checked_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Date: ...

@t.final
class DateDifference:
    def __init__(
        self,
        date: Date,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> None: ...
    def smallest(self, unit: JIFF_UNIT) -> DateDifference: ...
    def largest(self, unit: JIFF_UNIT) -> DateDifference: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> DateDifference: ...
    def increment(self, increment: int) -> DateDifference: ...

@t.final
class Time(ToPy[pydt.time], ToPyTime):
    MIN: Time
    MAX: Time

    def __init__(
        self,
        hour: int = 0,
        minute: int = 0,
        second: int = 0,
        nanosecond: int = 0,
    ) -> None: ...

    # =========================================================================
    # STRING
    # =========================================================================
    def string(self) -> str: ...
    def isoformat(self) -> str: ...

    # =========================================================================
    # OPERATORS/DUNDERS
    # =========================================================================
    def __add__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    @t.overload
    def __sub__(self, other: Time) -> TimeSpan: ...
    @t.overload
    def __sub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    @t.overload
    def __isub__(self, other: Time) -> TimeSpan: ...
    @t.overload
    def __isub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...

    # =========================================================================
    # STRPTIME/STRFTIME/PARSE
    # =========================================================================
    @classmethod
    def strptime(cls: type[Time], format: str, string: str) -> Time: ...
    def strftime(self, format: str) -> str: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    def to_py(self) -> pydt.time: ...
    def to_pytime(self) -> pydt.time: ...
    @classmethod
    def from_pytime(cls: type[Time], t: pydt.time) -> Time: ...

    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def midnight(cls: type[Time]) -> Time: ...
    @classmethod
    def now(cls: type[Time]) -> Time: ...
    @classmethod
    def parse(cls: type[Time], s: str) -> Time: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def hour(self) -> int: ...
    @property
    def minute(self) -> int: ...
    @property
    def second(self) -> int: ...
    @property
    def millisecond(self) -> int: ...
    @property
    def microsecond(self) -> int: ...
    @property
    def nanosecond(self) -> int: ...
    @property
    def subsec_nanosecond(self) -> None: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def astuple(self) -> tuple[int, int, int, int]: ...
    def asdict(self) -> TimeTypedDict: ...
    def series(self, span: TimeSpan) -> JiffSeries[Time]: ...
    @t.overload
    def checked_sub(self, other: Time) -> TimeSpan: ...
    @t.overload
    def checked_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    def saturating_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    def saturating_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    def wrapping_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    def wrapping_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Time: ...
    def on(self, year: int, month: int, day: int) -> DateTime: ...
    def duration_until(self, other: Time) -> SignedDuration: ...
    def duration_since(self, other: Time) -> SignedDuration: ...
    def round(
        self,
        smallest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> Time: ...
    def to_datetime(self, d: Date) -> DateTime: ...

    # =========================================================================
    # SINCE/UNTIL
    # =========================================================================
    def _since(self, other: TimeDifference) -> TimeSpan: ...
    def _until(self, other: TimeDifference) -> TimeSpan: ...
    def since(
        self,
        other: Time | DateTime | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...
    def until(
        self,
        other: Time | DateTime | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...

@t.final
class TimeDifference:
    def __init__(
        self,
        date: Time,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> None: ...
    def smallest(self, unit: JIFF_UNIT) -> TimeDifference: ...
    def largest(self, unit: JIFF_UNIT) -> TimeDifference: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> TimeDifference: ...
    def increment(self, increment: int) -> TimeDifference: ...

@t.final
class DateTime(ToPy[pydt.datetime], ToPyDate, ToPyTime, ToPyDateTime):
    MIN: DateTime
    MAX: DateTime
    ZERO: DateTime

    def __init__(
        self,
        year: int,
        month: int,
        day: int,
        hour: int = 0,
        minute: int = 0,
        second: int = 0,
        nanosecond: int = 0,
    ) -> None: ...
    def string(self) -> str: ...
    def isoformat(self) -> str: ...

    # =========================================================================
    # STRPTIME/STRFTIME/PARSE
    # =========================================================================
    def strftime(self, format: str) -> str: ...
    @classmethod
    def strptime(cls: type[DateTime], format: str, string: str) -> DateTime: ...
    @classmethod
    def parse(cls: type[DateTime], s: str) -> DateTime: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    @classmethod
    def from_pydatetime(cls: type[DateTime], dt: pydt.datetime) -> DateTime: ...
    def to_py(self) -> pydt.datetime: ...
    def to_pydate(self) -> pydt.date: ...
    def to_pydatetime(self) -> pydt.datetime: ...
    def to_pytime(self) -> pydt.time: ...
    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def now(cls: type[DateTime]) -> DateTime: ...
    @classmethod
    def from_parts(cls: type[DateTime], date: Date, time: Time) -> DateTime: ...

    # =========================================================================
    # OPERATORS
    # =========================================================================
    def __add__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...
    @t.overload
    def __sub__(self, other: DateTime) -> TimeSpan: ...
    @t.overload
    def __sub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...
    @t.overload
    def __isub__(self, other: DateTime) -> TimeSpan: ...
    @t.overload
    def __isub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def asdict(self) -> DateTimeTypedDict: ...
    def checked_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...
    def date(self) -> Date: ...
    def day_of_year(self) -> int: ...
    def day_of_year_no_leap(self) -> int | None: ...
    def days_in_month(self) -> int: ...
    def days_in_year(self) -> int: ...
    def duration_since(self, other: DateTime) -> SignedDuration: ...
    def duration_until(self, other: DateTime) -> SignedDuration: ...
    def end_of_day(self) -> DateTime: ...
    def era_year(self) -> tuple[int, t.Literal["BCE", "CE"]]: ...
    def first_of_month(self) -> DateTime: ...
    def first_of_year(self) -> DateTime: ...
    def in_leap_year(self) -> bool: ...
    def in_tz(self, tz: str) -> ZonedDateTime: ...
    def intz(self, tz: str) -> ZonedDateTime: ...
    def iso_week_date(self) -> ISOWeekDate: ...
    def last_of_month(self) -> DateTime: ...
    def last_of_year(self) -> DateTime: ...
    def nth_weekday(self, nth: int, weekday: WEEKDAY) -> DateTime: ...
    def nth_weekday_of_month(self, nth: int, weekday: WEEKDAY) -> DateTime: ...
    def round(
        self,
        smallest: JIFF_UNIT | None = None,
        *,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> DateTime: ...
    def _round(self, options: DateTimeRound) -> DateTime: ...
    def saturating_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...
    def series(self, span: TimeSpan) -> JiffSeries[DateTime]: ...
    def start_of_day(self) -> DateTime: ...
    def time(self) -> Time: ...
    def to_zoned(self, tz: TimeZone) -> ZonedDateTime: ...
    def tomorrow(self) -> DateTime: ...
    def yesterday(self) -> DateTime: ...
    # =========================================================================
    # INSTANCE METHODS W/ OVERLOADS
    # =========================================================================
    @t.overload
    def checked_sub(self, other: DateTime) -> TimeSpan: ...
    @t.overload
    def checked_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...
    @t.overload
    def saturating_sub(self, other: DateTime) -> TimeSpan: ...
    @t.overload
    def saturating_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> DateTime: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def year(self) -> int: ...
    @property
    def month(self) -> int: ...
    @property
    def day(self) -> int: ...
    @property
    def hour(self) -> int: ...
    @property
    def minute(self) -> int: ...
    @property
    def second(self) -> int: ...
    @property
    def millisecond(self) -> int: ...
    @property
    def microsecond(self) -> int: ...
    @property
    def nanosecond(self) -> int: ...
    @property
    def subsec_nanosecond(self) -> int: ...
    @property
    def weekday(self) -> int: ...

    # =========================================================================
    # SINCE/UNTIL
    # =========================================================================
    def _since(self, other: DateTimeDifference) -> TimeSpan: ...
    def _until(self, other: DateTimeDifference) -> TimeSpan: ...
    def since(
        self,
        other: Date | Time | DateTime | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...
    def until(
        self,
        other: Date | Time | DateTime | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...

@t.final
class DateTimeDifference:
    def __init__(
        self,
        date: DateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> None: ...
    def smallest(self, unit: JIFF_UNIT) -> DateTimeDifference: ...
    def largest(self, unit: JIFF_UNIT) -> DateTimeDifference: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> DateTimeDifference: ...
    def increment(self, increment: int) -> DateTimeDifference: ...

@t.final
class TimeZone(ToPy[pydt.tzinfo], ToPyTzInfo):
    def __init__(self, name: str) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __call__(self) -> te.Self: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================

    def to_py(self) -> pydt.tzinfo: ...
    def to_pytzinfo(self) -> pydt.tzinfo: ...
    @classmethod
    def from_pytzinfo(cls: type[TimeZone], tz: pydt.tzinfo) -> TimeZone: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def name(self) -> str: ...
    @property
    def is_unknown(self) -> bool: ...

    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def fixed(cls: type[TimeZone], offset: Offset) -> TimeZone: ...
    @classmethod
    def get(cls: type[TimeZone], name: str) -> TimeZone: ...
    @classmethod
    def posix(cls: type[TimeZone], name: str) -> TimeZone: ...
    @classmethod
    def system(cls: type[TimeZone]) -> TimeZone: ...
    @classmethod
    def try_system(cls: type[TimeZone]) -> TimeZone: ...
    @classmethod
    def tzif(cls: type[TimeZone], name: str, data: bytes) -> TimeZone: ...
    @classmethod
    def utc(cls: type[TimeZone]) -> TimeZone: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def iana_name(self) -> str | None: ...
    def to_datetime(self, dt: Timestamp) -> DateTime: ...
    def to_offset(self, timestamp: Timestamp) -> Offset: ...
    def to_timestamp(self, dt: DateTime) -> Timestamp: ...
    def to_zoned(self, other: DateTime) -> ZonedDateTime: ...

    # =========================================================================
    # NOT IMPLEMENTED
    # =========================================================================
    def to_ambiguous_timestamp(self) -> t.NoReturn: ...
    def to_ambiguous_zoned(self) -> t.NoReturn: ...

@t.final
class SignedDuration(ToPy[pydt.timedelta], ToPyTimeDelta):
    MIN: SignedDuration
    MAX: SignedDuration
    ZERO: SignedDuration

    def __init__(self, secs: int = 0, nanos: int = 0) -> None: ...

    # =========================================================================
    # OPERATORS/DUNDERS
    # =========================================================================
    def __hash__(self) -> int: ...
    def __mul__(self, other: int) -> SignedDuration: ...
    def __rmul__(self, other: int) -> SignedDuration: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: object) -> bool: ...
    def __le__(self, other: object) -> bool: ...
    def __gt__(self, other: object) -> bool: ...
    def __ge__(self, other: object) -> bool: ...
    def __neg__(self) -> SignedDuration: ...
    def __add__(self, other: SignedDuration) -> SignedDuration: ...
    def __abs__(self) -> SignedDuration: ...
    def __float__(self) -> float: ...
    def __int__(self) -> int: ...
    def __bool__(self) -> bool: ...
    def __div__(self, other: int) -> SignedDuration: ...
    def abs(self) -> SignedDuration: ...
    def unsigned_abs(self) -> Duration: ...
    def __richcmp__(self, other: SignedDuration | pydt.timedelta, op: int) -> bool: ...

    # =========================================================================
    # STRING
    # =========================================================================
    def string(self, human: bool = False) -> str: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    @classmethod
    def from_pytimedelta(
        cls: type[SignedDuration], td: pydt.timedelta
    ) -> SignedDuration: ...
    def to_py(self) -> pydt.timedelta: ...
    def to_pytimedelta(self) -> pydt.timedelta: ...

    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def parse(cls: type[SignedDuration], s: str) -> SignedDuration: ...
    @classmethod
    def from_hours(cls: type[SignedDuration], n: int) -> SignedDuration: ...
    @classmethod
    def from_micros(cls: type[SignedDuration], n: int) -> SignedDuration: ...
    @classmethod
    def from_millis(cls: type[SignedDuration], n: int) -> SignedDuration: ...
    @classmethod
    def from_mins(cls: type[SignedDuration], n: int) -> SignedDuration: ...
    @classmethod
    def from_nanos(cls: type[SignedDuration], n: int) -> SignedDuration: ...
    @classmethod
    def from_secs(cls: type[SignedDuration], n: int) -> SignedDuration: ...
    @classmethod
    def from_secs_f32(cls: type[SignedDuration], n: float) -> SignedDuration: ...
    @classmethod
    def from_secs_f64(cls: type[SignedDuration], n: float) -> SignedDuration: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def is_negative(self) -> bool: ...
    @property
    def is_zero(self) -> bool: ...
    @property
    def secs(self) -> int: ...
    @property
    def nanos(self) -> int: ...
    @property
    def days(self) -> int: ...
    @property
    def seconds(self) -> int: ...
    @property
    def microseconds(self) -> int: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def as_hours(self) -> int: ...
    def as_micros(self) -> int: ...
    def as_millis(self) -> int: ...
    def as_millis_f32(self) -> float: ...
    def as_millis_f64(self) -> float: ...
    def as_mins(self) -> int: ...
    def as_nanos(self) -> int: ...
    def as_secs(self) -> int: ...
    def as_secs_f32(self) -> float: ...
    def as_secs_f64(self) -> float: ...
    def checked_add(self, other: SignedDuration) -> SignedDuration | None: ...
    def checked_div(self, other: int) -> SignedDuration | None: ...
    def checked_mul(self, other: int) -> SignedDuration | None: ...
    def checked_neg(self) -> SignedDuration | None: ...
    def checked_sub(self, other: SignedDuration) -> SignedDuration | None: ...
    def div_duration_f32(self, other: SignedDuration) -> float: ...
    def div_duration_f64(self, other: SignedDuration) -> float: ...
    def div_f32(self, other: int) -> float: ...
    def div_f64(self, other: int) -> float: ...
    def is_positive(self) -> bool: ...
    def mul_f32(self, other: int) -> SignedDuration: ...
    def mul_f64(self, other: int) -> SignedDuration: ...
    def saturating_add(self, other: SignedDuration) -> SignedDuration: ...
    def saturating_mul(self, other: int) -> SignedDuration: ...
    def saturating_sub(self, other: SignedDuration) -> SignedDuration: ...
    def signum(self) -> t.Literal[-1, 0, 1]: ...
    def subsec_micros(self) -> int: ...
    def subsec_millis(self) -> int: ...
    def subsec_nanos(self) -> int: ...
    def to_timespan(self) -> TimeSpan: ...

# put in quotes to avoid ruff F821 - undefined name
_TimeSpanArithmeticSingle: te.TypeAlias = TimeSpan | Duration | SignedDuration
_TimeSpanArithmeticTuple: te.TypeAlias = tuple[
    _TimeSpanArithmeticSingle, ZonedDateTime | Date | DateTime
]
TimeSpanArithmetic: te.TypeAlias = _TimeSpanArithmeticSingle | _TimeSpanArithmeticTuple

@t.final
class TimeSpan(ToPy[pydt.timedelta], ToPyTimeDelta):
    def __init__(
        self,
        years: int = 0,
        months: int = 0,
        weeks: int = 0,
        days: int = 0,
        hours: int = 0,
        minutes: int = 0,
        seconds: int = 0,
        milliseconds: int = 0,
        microseconds: int = 0,
        nanoseconds: int = 0,
    ) -> None: ...

    # =========================================================================
    # STRING
    # =========================================================================
    def string(self, human: bool = False) -> str: ...
    def repr_full(self) -> str: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    @classmethod
    def from_pytimedelta(cls, td: pydt.timedelta) -> TimeSpan: ...
    def to_pytimedelta(self) -> pydt.timedelta: ...
    def to_py(self) -> pydt.timedelta: ...

    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def parse(cls, s: str) -> TimeSpan: ...
    @classmethod
    def parse_common_iso(cls, s: str) -> TimeSpan: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def is_positive(self) -> bool: ...
    @property
    def is_negative(self) -> bool: ...
    @property
    def is_zero(self) -> bool: ...
    @property
    def years(self) -> int: ...
    @property
    def months(self) -> int: ...
    @property
    def weeks(self) -> int: ...
    @property
    def days(self) -> int: ...
    @property
    def hours(self) -> int: ...
    @property
    def minutes(self) -> int: ...
    @property
    def seconds(self) -> int: ...
    @property
    def milliseconds(self) -> int: ...
    @property
    def microseconds(self) -> int: ...
    @property
    def nanoseconds(self) -> int: ...

    # =========================================================================
    # OPERATORS
    # =========================================================================
    def __add__(
        self,
        val: TimeSpanArithmetic,
    ) -> te.Self: ...
    def __sub__(
        self,
        val: TimeSpanArithmetic,
    ) -> te.Self: ...
    def __mul__(self, other: int) -> te.Self: ...
    def __neg__(self) -> te.Self: ...
    def __abs__(self) -> te.Self: ...
    def __invert__(self) -> te.Self: ...
    def __eq__(self, other: object) -> bool: ...
    def __ge__(self, other: TimeSpan) -> bool: ...
    def __gt__(self, other: TimeSpan) -> bool: ...
    def __le__(self, other: TimeSpan) -> bool: ...
    def __lt__(self, other: TimeSpan) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __rmul__(self, other: TimeSpan) -> bool: ...
    def __hash__(self) -> int: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================

    def abs(self) -> te.Self: ...
    def asdict(self) -> TimeSpanTypedDict: ...
    def checked_add(self, val: TimeSpanArithmetic) -> te.Self: ...
    def checked_mul(self, other: int) -> te.Self: ...
    def checked_sub(self, val: TimeSpanArithmetic) -> te.Self: ...
    def compare(
        self,
        other: TimeSpan,
        relative: ZonedDateTime | DateTime | Date | None = None,
        days_are_24_hours: bool = False,
    ) -> int: ...
    def negate(self) -> te.Self: ...
    def replace(
        self,
        years: int | None = None,
        months: int | None = None,
        weeks: int | None = None,
        days: int | None = None,
        hours: int | None = None,
        minutes: int | None = None,
        seconds: int | None = None,
        milliseconds: int | None = None,
        microseconds: int | None = None,
        nanoseconds: int | None = None,
    ) -> te.Self: ...
    def round(
        self,
        smallest: JIFF_UNIT,
        increment: int = 1,
        *,
        relative: ZonedDateTime | Date | DateTime | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
    ) -> te.Self: ...
    def signum(self) -> t.Literal[-1, 0, 1]: ...
    def to_signed_duration(
        self, relative: ZonedDateTime | Date | DateTime
    ) -> SignedDuration: ...
    def total(
        self,
        unit: JIFF_UNIT,
        relative: ZonedDateTime | Date | DateTime | None = None,
        days_are_24_hours: bool = False,
    ) -> int: ...
    def total_seconds(self) -> int: ...
    def try_years(self, years: int) -> te.Self: ...
    def try_months(self, months: int) -> te.Self: ...
    def try_weeks(self, weeks: int) -> te.Self: ...
    def try_days(self, days: int) -> te.Self: ...
    def try_hours(self, hours: int) -> te.Self: ...
    def try_minutes(self, minutes: int) -> te.Self: ...
    def try_seconds(self, seconds: int) -> te.Self: ...
    def try_milliseconds(self, milliseconds: int) -> te.Self: ...
    def try_microseconds(self, microseconds: int) -> te.Self: ...
    def try_nanoseconds(self, nanoseconds: int) -> te.Self: ...

    # -------------------------------------------------------------------------
    # PANIC-INDUCING METHODS
    # -------------------------------------------------------------------------
    def _years(self, years: int) -> te.Self: ...
    def _months(self, months: int) -> te.Self: ...
    def _weeks(self, weeks: int) -> te.Self: ...
    def _days(self, days: int) -> te.Self: ...
    def _hours(self, hours: int) -> te.Self: ...
    def _minutes(self, minutes: int) -> te.Self: ...
    def _seconds(self, seconds: int) -> te.Self: ...
    def _milliseconds(self, milliseconds: int) -> te.Self: ...
    def _microseconds(self, microseconds: int) -> te.Self: ...
    def _nanoseconds(self, nanoseconds: int) -> te.Self: ...

@t.final
class Timestamp(ToPy[pydt.datetime], ToPyDate, ToPyTime, ToPyDateTime):
    """
    A representation of a timestamp with second and nanosecond precision.
    """

    MIN: Timestamp
    MAX: Timestamp
    UNIX_EPOCH: Timestamp

    def __init__(
        self, second: int | None = None, nanosecond: int | None = None
    ) -> None: ...

    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def now(cls) -> Timestamp: ...
    @classmethod
    def parse(cls, s: str) -> Timestamp: ...
    @classmethod
    def from_millisecond(cls, millisecond: int) -> Timestamp: ...
    @classmethod
    def from_microsecond(cls, microsecond: int) -> Timestamp: ...
    @classmethod
    def from_nanosecond(cls, nanosecond: int) -> Timestamp: ...
    @classmethod
    def from_second(cls, second: int) -> Timestamp: ...

    # =========================================================================
    # OPERATORS/DUNDERS
    # =========================================================================
    def __add__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    def __eq__(self, other: object) -> bool: ...
    def __ge__(self, other: Timestamp) -> bool: ...
    def __gt__(self, other: Timestamp) -> bool: ...
    def __le__(self, other: Timestamp) -> bool: ...
    def __lt__(self, other: Timestamp) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __richcmp__(self, other: Timestamp, op: int) -> bool: ...

    # =========================================================================
    # OPERATORS/DUNDERS W/ OVERLOADS
    # =========================================================================
    @t.overload
    def __isub__(self, other: Timestamp) -> TimeSpan: ...
    @t.overload
    def __isub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    @t.overload
    def __sub__(self, other: Timestamp) -> TimeSpan: ...
    @t.overload
    def __sub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    @classmethod
    def from_pydatetime(cls, dt: pydt.datetime) -> Timestamp: ...
    def to_py(self) -> pydt.datetime: ...
    def to_pydate(self) -> pydt.date: ...
    def to_pydatetime(self) -> pydt.datetime: ...
    def to_pytime(self) -> pydt.time: ...

    # =========================================================================
    # STRPTIME/STRFTIME
    # =========================================================================
    def strftime(self, format: str) -> str: ...
    @classmethod
    def strptime(cls, format: str, input: str) -> Timestamp: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================

    def as_microsecond(self) -> int: ...
    def as_millisecond(self) -> int: ...
    def as_nanosecond(self) -> int: ...
    def as_second(self) -> int: ...
    def checked_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Timestamp: ...
    @t.overload
    def checked_sub(self, other: Timestamp) -> TimeSpan: ...
    @t.overload
    def checked_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Timestamp: ...
    def display_with_offset(self, offset: Offset) -> str: ...
    def in_tz(self, tz: str) -> ZonedDateTime: ...
    def intz(self, tz: str) -> ZonedDateTime:
        """Deprecated ~ use `in_tz`"""

    def is_zero(self) -> bool: ...
    def saturating_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Timestamp: ...
    def saturating_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> Timestamp: ...
    def series(self, span: TimeSpan) -> JiffSeries[Timestamp]: ...
    def signum(self) -> t.Literal[-1, 0, 1]: ...
    def string(self) -> str: ...
    def subsec_microsecond(self) -> int: ...
    def subsec_millisecond(self) -> int: ...
    def subsec_nanosecond(self) -> int: ...
    def to_zoned(self, time_zone: TimeZone) -> ZonedDateTime: ...

    # =========================================================================
    # SINCE/UNTIL
    # =========================================================================
    def _since(self, other: TimestampDifference) -> TimeSpan: ...
    def _until(self, other: TimestampDifference) -> TimeSpan: ...
    def since(
        self,
        other: Timestamp | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...
    def until(
        self,
        other: Timestamp | ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...
    def duration_since(self, other: Timestamp) -> SignedDuration: ...
    def duration_until(self, other: Timestamp) -> SignedDuration: ...
    def round(
        self,
        unit: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> Timestamp: ...
    def _round(self, options: TimestampRound) -> Timestamp: ...

@t.final
class TimestampDifference:
    def __init__(
        self,
        date: Timestamp,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> None: ...
    def smallest(self, unit: JIFF_UNIT) -> TimestampDifference: ...
    def largest(self, unit: JIFF_UNIT) -> TimestampDifference: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> TimestampDifference: ...
    def increment(self, increment: int) -> TimestampDifference: ...

@t.final
class ZonedDateTime(ToPy[pydt.datetime], ToPyDate, ToPyTime, ToPyDateTime, ToPyTzInfo):
    def __init__(
        self,
        year: int,
        month: int,
        day: int,
        hour: int = 0,
        minute: int = 0,
        second: int = 0,
        nanosecond: int = 0,
        tz: str | None = None,
    ) -> None: ...
    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    @classmethod
    def from_pydatetime(
        cls: type[ZonedDateTime], dt: pydt.datetime
    ) -> ZonedDateTime: ...
    def to_py(self) -> pydt.datetime: ...
    def to_pydate(self) -> pydt.date: ...
    def to_pydatetime(self) -> pydt.datetime: ...
    def to_pytime(self) -> pydt.time: ...
    def to_pytzinfo(self) -> pydt.tzinfo: ...
    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def now(cls: type[ZonedDateTime], tz: str | None = None) -> ZonedDateTime: ...
    @classmethod
    def utcnow(cls: type[ZonedDateTime]) -> ZonedDateTime: ...
    @classmethod
    def parse(cls: type[ZonedDateTime], s: str) -> ZonedDateTime: ...
    @classmethod
    def from_rfc2822(cls: type[ZonedDateTime], s: str) -> ZonedDateTime: ...
    @classmethod
    def parse_rfc2822(cls: type[ZonedDateTime], s: str) -> ZonedDateTime: ...
    @classmethod
    def from_parts(
        cls: type[ZonedDateTime], timestamp: Timestamp, time_zone: TimeZone
    ) -> ZonedDateTime: ...

    # =========================================================================
    # STRPTIME/STRFTIME
    # =========================================================================
    @classmethod
    def strptime(
        cls: type[ZonedDateTime], format: str, input: str
    ) -> ZonedDateTime: ...
    def strftime(self, format: str) -> str: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def year(self) -> int: ...
    @property
    def month(self) -> int: ...
    @property
    def day(self) -> int: ...
    @property
    def hour(self) -> int: ...
    @property
    def minute(self) -> int: ...
    @property
    def second(self) -> int: ...
    @property
    def millisecond(self) -> int: ...
    @property
    def microsecond(self) -> int: ...
    @property
    def nanosecond(self) -> int: ...
    @property
    def subsec_nanosecond(self) -> int: ...
    @property
    def weekday(self) -> int: ...
    @property
    def timezone(self) -> TimeZone: ...
    @property
    def tz(self) -> TimeZone: ...

    # =========================================================================
    # STRING/FORMAT
    # =========================================================================
    def string(self) -> str: ...
    def to_rfc2822(self) -> str: ...
    def format_rfc2822(self) -> str: ...
    def isoformat(self) -> str: ...

    # =========================================================================
    # OPERATORS/DUNDERS
    # =========================================================================
    def __add__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    def __eq__(self, other: object) -> bool: ...
    def __ge__(self, other: ZonedDateTime) -> bool: ...
    def __gt__(self, other: ZonedDateTime) -> bool: ...
    def __hash__(self) -> int: ...
    def __le__(self, other: ZonedDateTime) -> bool: ...
    def __lt__(self, other: ZonedDateTime) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __richcmp__(self, other: ZonedDateTime, op: int) -> bool: ...

    # =========================================================================
    # OPERATORS/DUNDERS W/ OVERLOADS
    # =========================================================================
    @t.overload
    def __isub__(self, other: ZonedDateTime) -> TimeSpan: ...
    @t.overload
    def __isub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    @t.overload
    def __sub__(self, other: ZonedDateTime) -> TimeSpan: ...
    @t.overload
    def __sub__(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def astimezone(self, tz: str) -> ZonedDateTime: ...
    def checked_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    @t.overload
    def checked_sub(self, other: ZonedDateTime) -> TimeSpan: ...
    @t.overload
    def checked_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    def date(self) -> Date: ...
    def datetime(self) -> DateTime: ...
    def iso_week_date(self) -> ISOWeekDate: ...
    def day_of_year(self) -> int: ...
    def day_of_year_no_leap(self) -> int | None: ...
    def days_in_month(self) -> int: ...
    def days_in_year(self) -> int: ...
    def duration_since(self, other: ZonedDateTime) -> SignedDuration: ...
    def duration_until(self, other: ZonedDateTime) -> SignedDuration: ...
    def end_of_day(self) -> ZonedDateTime: ...
    def era_year(self) -> tuple[int, t.Literal["CE", "BCE"]]: ...
    def first_of_month(self) -> ZonedDateTime: ...
    def first_of_year(self) -> ZonedDateTime: ...
    def in_leap_year(self) -> bool: ...
    def in_tz(self, tz: str) -> te.Self: ...
    def intz(self, tz: str) -> te.Self: ...
    def inutc(self) -> ZonedDateTime: ...
    def last_of_month(self) -> ZonedDateTime: ...
    def last_of_year(self) -> ZonedDateTime: ...
    def nth_weekday(self, nth: int, weekday: WEEKDAY) -> Date: ...
    def nth_weekday_of_month(self, nth: int, weekday: WEEKDAY) -> Date: ...
    def offset(self) -> Offset: ...
    def replace(
        self,
        obj: Date | DateTime | Offset | None = None,
        *,
        date: Date | None = None,
        time: Time | None = None,
        year: int | None = None,
        era_year: tuple[int, t.Literal["BCE", "CE"]] | None = None,
        month: int | None = None,
        day: int | None = None,
        day_of_year: int | None = None,
        day_of_year_no_leap: int | None = None,
        hour: int | None = None,
        minute: int | None = None,
        second: int | None = None,
        millisecond: int | None = None,
        microsecond: int | None = None,
        nanosecond: int | None = None,
        subsec_nanosecond: int | None = None,
        offset: Offset | None = None,
        offset_conflict: t.Any = None,
        disambiguation: t.Any = None,
    ) -> ZonedDateTime: ...
    def round(
        self,
        smallest: JIFF_UNIT | None = None,
        *,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> DateTime: ...
    def _round(self, options: ZonedDateTimeRound) -> DateTime: ...
    def saturating_add(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    @t.overload
    def saturating_sub(self, other: ZonedDateTime) -> TimeSpan: ...
    @t.overload
    def saturating_sub(
        self, other: TimeSpan | SignedDuration | Duration | pydt.timedelta
    ) -> te.Self: ...
    def start_of_day(self) -> ZonedDateTime: ...
    def time(self) -> Time: ...
    def timestamp(self) -> Timestamp: ...
    def tomorrow(self) -> ZonedDateTime: ...
    def with_time_zone(self, tz: TimeZone) -> ZonedDateTime: ...
    def yesterday(self) -> ZonedDateTime: ...
    # =========================================================================
    # SINCE/UNTIL
    # =========================================================================
    def since(
        self,
        other: ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...
    def until(
        self,
        other: ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> TimeSpan: ...

@t.final
class ZonedDateTimeDifference:
    def __init__(
        self,
        date: ZonedDateTime,
        *,
        smallest: JIFF_UNIT | None = None,
        largest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int | None = None,
    ) -> None: ...
    def smallest(self, unit: JIFF_UNIT) -> ZonedDateTimeDifference: ...
    def largest(self, unit: JIFF_UNIT) -> ZonedDateTimeDifference: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> ZonedDateTimeDifference: ...
    def increment(self, increment: int) -> ZonedDateTimeDifference: ...

@t.final
class ISOWeekDate:
    MIN: ISOWeekDate
    MAX: ISOWeekDate
    ZERO: ISOWeekDate

    def __init__(self, year: int, week: int, weekday: WEEKDAY) -> None: ...

    # =========================================================================
    # OPERATORS/DUNDERS
    # =========================================================================

    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: ISOWeekDate) -> bool: ...
    def __le__(self, other: ISOWeekDate) -> bool: ...
    def __gt__(self, other: ISOWeekDate) -> bool: ...
    def __ge__(self, other: ISOWeekDate) -> bool: ...
    def __hash__(self) -> int: ...

    # =========================================================================
    # CLASS METHODS
    # =========================================================================
    @classmethod
    def from_date(cls: type[ISOWeekDate], date: Date) -> ISOWeekDate: ...
    @classmethod
    def today(cls: type[ISOWeekDate]) -> ISOWeekDate: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def year(self) -> int: ...
    @property
    def week(self) -> int: ...
    @property
    def weekday(self) -> WEEKDAY_INT: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def date(self) -> Date: ...

@t.final
class TimestampRound:
    def __init__(
        self,
        smallest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int = 1,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> TimestampRound: ...
    def smallest(self, smallest: JIFF_UNIT) -> TimestampRound: ...
    def increment(self, increment: int) -> TimestampRound: ...
    def _smallest(self) -> JIFF_UNIT: ...
    def _mode(self) -> JIFF_ROUND_MODE: ...
    def _increment(self) -> int: ...
    def replace(
        self,
        smallest: JIFF_UNIT | None,
        mode: JIFF_ROUND_MODE | None,
        increment: int | None,
    ) -> TimestampRound: ...

@t.final
class DateTimeRound:
    def __init__(
        self,
        smallest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int = 1,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> DateTimeRound: ...
    def smallest(self, smallest: JIFF_UNIT) -> DateTimeRound: ...
    def increment(self, increment: int) -> DateTimeRound: ...
    def _smallest(self) -> JIFF_UNIT: ...
    def _mode(self) -> JIFF_ROUND_MODE: ...
    def _increment(self) -> int: ...
    def replace(
        self,
        smallest: JIFF_UNIT | None,
        mode: JIFF_ROUND_MODE | None,
        increment: int | None,
    ) -> DateTimeRound: ...

@t.final
class ZonedDateTimeRound:
    def __init__(
        self,
        smallest: JIFF_UNIT | None = None,
        mode: JIFF_ROUND_MODE | None = None,
        increment: int = 1,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def mode(self, mode: JIFF_ROUND_MODE) -> DateTimeRound: ...
    def smallest(self, smallest: JIFF_UNIT) -> DateTimeRound: ...
    def increment(self, increment: int) -> DateTimeRound: ...
    def _smallest(self) -> JIFF_UNIT: ...
    def _mode(self) -> JIFF_ROUND_MODE: ...
    def _increment(self) -> int: ...
    def replace(
        self,
        smallest: JIFF_UNIT | None,
        mode: JIFF_ROUND_MODE | None,
        increment: int | None,
    ) -> DateTimeRound: ...

@t.final
class Offset(ToPy[pydt.tzinfo], ToPyTzInfo):
    MIN: Offset
    MAX: Offset
    UTC: Offset
    ZERO: Offset

    def __init__(
        self, hours: int | None = None, seconds: int | None = None
    ) -> None: ...

    # =========================================================================
    # STRING
    # =========================================================================
    def string(self) -> str: ...

    # =========================================================================
    # OPERATORS/DUNDERS
    # =========================================================================
    def __neg__(self) -> Offset: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: Offset) -> bool: ...
    def __le__(self, other: Offset) -> bool: ...
    def __gt__(self, other: Offset) -> bool: ...
    def __ge__(self, other: Offset) -> bool: ...
    def __hash__(self) -> int: ...

    # =========================================================================
    # PYTHON CONVERSIONS
    # =========================================================================
    # __FROM__
    @classmethod
    def from_pytzinfo(cls: type[Offset], tz: pydt.tzinfo) -> Offset: ...
    # __TO__
    def to_py(self) -> pydt.tzinfo: ...
    def to_pytzinfo(self) -> pydt.tzinfo: ...

    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def seconds(self) -> int: ...
    @property
    def is_negative(self) -> bool: ...
    @property
    def is_positive(self) -> bool: ...

    # =========================================================================
    # FROM
    # =========================================================================
    @classmethod
    def utc(cls: type[Offset]) -> Offset: ...
    @classmethod
    def from_hours(cls: type[Offset], hours: int) -> Offset: ...
    @classmethod
    def from_seconds(cls: type[Offset], seconds: int) -> Offset: ...

    # =========================================================================
    # TO
    # =========================================================================
    def to_datetime(self, timestamp: Timestamp) -> DateTime: ...
    def to_timestamp(self, datetime: DateTime) -> Timestamp: ...
    def to_timezone(self) -> TimeZone: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def checked_add(self, other: Duration | SignedDuration | TimeSpan) -> Offset: ...
    def checked_sub(self, other: Duration | SignedDuration | TimeSpan) -> Offset: ...
    def duration_since(self, other: Offset) -> SignedDuration: ...
    def duration_until(self, other: Offset) -> SignedDuration: ...
    def negate(self) -> Offset: ...
    def saturating_add(self, other: Duration | SignedDuration | TimeSpan) -> Offset: ...
    def saturating_sub(self, other: Duration | SignedDuration | TimeSpan) -> Offset: ...
    def since(self, other: Offset) -> TimeSpan: ...
    def until(self, other: Offset) -> TimeSpan: ...

class JiffSeries(
    t.Generic[_T],
):
    def __iter__(self) -> t.Iterator[_T]: ...
    def __next__(self) -> _T: ...
    def take(self, n: int) -> list[_T]: ...

def date(year: int, month: int, day: int) -> Date: ...
def time(
    hour: int = 0, minute: int = 0, second: int = 0, nanosecond: int = 0
) -> Time: ...
def datetime(
    year: int,
    month: int,
    day: int,
    hour: int = 0,
    minute: int = 0,
    second: int = 0,
    nanosecond: int = 0,
) -> DateTime: ...
def zoned(
    year: int,
    month: int,
    day: int,
    hour: int = 0,
    minute: int = 0,
    second: int = 0,
    nanosecond: int = 0,
    tz: str | None = None,
) -> ZonedDateTime: ...
def timespan(
    *,
    years: int = 0,
    months: int = 0,
    weeks: int = 0,
    days: int = 0,
    hours: int = 0,
    minutes: int = 0,
    seconds: int = 0,
    milliseconds: int = 0,
    microseconds: int = 0,
    nanoseconds: int = 0,
) -> TimeSpan: ...
def offset(hours: int) -> Offset: ...

# =============================================================================
# TIMEZONE-DATABASE
# =============================================================================
@t.final
class TimeZoneDatabase:
    def __init__(self) -> None:
        """Defaults to using the `self.from_env`"""
    @t.overload
    def get(self, name: str, err: t.Literal[False]) -> TimeZone | None:
        """Returns TimeZone or None if the timezone is not found"""
    @t.overload
    def get(self, name: str, err: t.Literal[True] = True) -> TimeZone:
        """Returns TimeZone, if not found raises a ValueError"""
    def available(self) -> list[str]: ...
    def __getitem__(self, name: str) -> TimeZone: ...
    def __len__(self) -> int: ...
    def is_definitively_empty(self) -> bool: ...
    @classmethod
    def from_env(cls) -> TimeZoneDatabase: ...
    @classmethod
    def from_dir(cls, path: str) -> TimeZoneDatabase: ...
    @classmethod
    def from_concatenated_path(cls, path: str) -> TimeZoneDatabase: ...
    @classmethod
    def bundled(cls) -> TimeZoneDatabase: ...
