import json
import os
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler

DEFAULT_PROJECT_FILE = "default.project.json"

class FolderHandler(FileSystemEventHandler):
    def __init__(self, src_dir, project_file):
        self.src_dir = src_dir
        self.project_file = project_file

    def on_created(self, event):
        if event.is_directory:
            folder_name = os.path.basename(event.src_path)
            # print(f"new folder detected: {folder_name}")
            update_project(self.src_dir, self.project_file, folder_name)

def update_project(src_dir, project_file, folder_name):
    project = load_project(project_file)
    if "tree" not in project:
        project["tree"] = {"$className": "DataModel"}
    project["tree"][folder_name] = {"$path": f"{src_dir}/{folder_name}"}
    save_project(project_file, project)
    # print("project updated")

def load_project(path):
    if not os.path.exists(path):
        return {"name": "AutoProject", "tree": {"$className": "DataModel"}}
    with open(path, "r", encoding="utf-8") as f:
        return json.load(f)

def save_project(path, data):
    with open(path, "w", encoding="utf-8") as f:
        json.dump(data, f, indent=4)

def start_watcher(src_dir="src", project_file=DEFAULT_PROJECT_FILE):
    if not os.path.exists(src_dir):
        # print(f"creating source folder: {src_dir}")
        os.makedirs(src_dir)

    handler = FolderHandler(src_dir, project_file)
    observer = Observer()
    observer.schedule(handler, src_dir, recursive=False)
    observer.start()
    # print(f"watching {src_dir} for new folders... Press Ctrl+C to stop.")
    try:
        while True:
            pass
    except KeyboardInterrupt:
        observer.stop()
    observer.join()