from pathlib import Path
from typing import Optional, List

from entitykb import (
    StoredUser,
    User,
    UserStatus,
    create_index,
    exceptions,
    interfaces,
    crypto,
)


class UserStore(interfaces.IUserStore):
    """ Default user storage managed via CLI. """

    def __init__(self, root: Path, secret_key: str):
        super().__init__(root)
        self.secret_key = secret_key
        self.cache = create_index(str(root / "users"), decoder=StoredUser)

    def authenticate(self, username: str, password: str) -> Optional[str]:
        """ Check username password combo, returns an access token. """
        user: StoredUser = self.find_by_username(username)
        if user is not None:
            if crypto.verify_password(password, user.hashed_password):
                return crypto.encode_jwt_token(user.uuid, self.secret_key)

    def get_user(self, token: str) -> Optional[User]:
        """ Return user status attached to user's uuid. """
        uuid = crypto.decode_jwt_token(token=token, secret_key=self.secret_key)
        user: StoredUser = self.cache.get(f"uuid={uuid}")
        return User(**user.dict())

    def get_user_list(self) -> List[StoredUser]:
        user_list = []
        for key, user in self.cache.items():
            if key.startswith("username="):
                user_list.append(user)
        return user_list

    def add_user(self, username: str, status: UserStatus) -> str:
        """ Create user if it doesn't exist with autogenerated password. """
        if not (isinstance(username, str) and len(username) > 0):
            raise exceptions.InvalidUsername()

        if self.find_by_username(username) is not None:
            raise exceptions.DuplicateUsername()

        plain_password = crypto.generate_password()
        hashed_password = crypto.hash_password(plain_password)
        user = StoredUser(
            username=username,
            status=status,
            hashed_password=hashed_password,
        )
        self.store(user)
        return plain_password

    def set_status(self, username: str, status: UserStatus) -> bool:
        """ Change user's status to new value. """
        user: StoredUser = self.find_by_username(username)
        success = False
        if user and user.status != status:
            user.status = status
            self.store(user)
            success = True
        return success

    def reset_password(self, username: str) -> str:
        """ Resets a user's password and returns the plain-text password. """
        user: StoredUser = self.find_by_username(username)
        if user:
            plain_password = crypto.generate_password()
            user.hashed_password = crypto.hash_password(plain_password)
            self.store(user)
            return plain_password
        else:
            raise exceptions.InvalidUsername()

    def find_by_username(self, username: str) -> Optional[StoredUser]:
        """ Return full user account from username. """
        return self.cache.get(f"username={username}")

    def store(self, user: User):
        self.cache[f"username={user.username}"] = user
        self.cache[f"uuid={user.uuid}"] = user
