"""Poco embeds."""

# pylint: disable=missing-class-docstring

import dhooks_lite

from django.utils.translation import gettext as _
from eveuniverse.models import EveType

from app_utils.datetime import ldap_time_2_datetime

from structures.constants import EveTypeId
from structures.helpers import get_or_create_esi_obj
from structures.models import Notification, Webhook

from .helpers import (
    gen_corporation_link,
    gen_solar_system_text,
    target_datetime_formatted,
)
from .main import NotificationBaseEmbed


class NotificationOrbitalEmbed(NotificationBaseEmbed):
    """Base class for all orbital (aka POCO) related notification embeds."""

    def __init__(self, notification: Notification) -> None:
        super().__init__(notification)
        self._planet = self._notification.eve_planet()
        self._structure_type = get_or_create_esi_obj(
            EveType, id=EveTypeId.CUSTOMS_OFFICE
        )
        self._solar_system_link = gen_solar_system_text(
            self._notification.eve_solar_system()
        )
        self._owner_link = gen_corporation_link(str(notification.owner))
        self._aggressor_link = self.get_aggressor_link()
        self._thumbnail = dhooks_lite.Thumbnail(
            self._structure_type.icon_url(size=self.ICON_DEFAULT_SIZE)
        )


class NotificationOrbitalAttacked(NotificationOrbitalEmbed):
    def __init__(self, notification: Notification) -> None:
        super().__init__(notification)
        self._title = _("Orbital under attack")
        self._description = _(
            "The %(structure_type)s at %(planet)s in %(solar_system)s "
            "belonging to %(owner_link)s "
            "is under attack by %(aggressor)s."
        ) % {
            "structure_type": self._structure_type.name,
            "planet": self._planet.name,
            "solar_system": self._solar_system_link,
            "owner_link": self._owner_link,
            "aggressor": self._aggressor_link,
        }
        self._color = Webhook.Color.WARNING


class NotificationOrbitalReinforced(NotificationOrbitalEmbed):
    def __init__(self, notification: Notification) -> None:
        super().__init__(notification)
        reinforce_exit_time = ldap_time_2_datetime(
            self._parsed_text["reinforceExitTime"]
        )
        self._title = _("Orbital reinforced")
        self._description = _(
            "The %(structure_type)s at %(planet)s in %(solar_system)s "
            "belonging to %(owner_link)s "
            "has been reinforced by %(aggressor)s "
            "and will come out at: %(date)s."
        ) % {
            "structure_type": self._structure_type.name,
            "planet": self._planet.name,
            "solar_system": self._solar_system_link,
            "owner_link": self._owner_link,
            "aggressor": self._aggressor_link,
            "date": target_datetime_formatted(reinforce_exit_time),
        }
        self._color = Webhook.Color.DANGER
