"""This module takes care of downloading WEBKNOSSOS datasets."""

import re
from typing import Annotated, Any
from urllib.parse import urlparse

import typer

from webknossos import RemoteDataset

from ..annotation.annotation import _ANNOTATION_URL_REGEX, Annotation
from ..client import webknossos_context
from ..client._resolve_short_link import resolve_short_link
from ..dataset.abstract_dataset import _DATASET_DEPRECATED_URL_REGEX, _DATASET_URL_REGEX
from ..geometry import BoundingBox, Mag
from ._utils import parse_bbox, parse_mag, parse_path


def main(
    *,
    target: Annotated[
        Any,
        typer.Argument(
            show_default=False,
            help="Path to save your WEBKNOSSOS dataset.",
            parser=parse_path,
        ),
    ],
    url: Annotated[
        str,
        typer.Option(
            help="URL of your dataset or your annotation.",
        ),
    ],
    token: Annotated[
        str | None,
        typer.Option(
            help="Authentication token for WEBKNOSSOS instance "
            "(https://webknossos.org/auth/token).",
            rich_help_panel="WEBKNOSSOS context",
            envvar="WK_TOKEN",
        ),
    ] = None,
    bbox: Annotated[
        BoundingBox | None,
        typer.Option(
            rich_help_panel="Partial download",
            help="Bounding box that should be downloaded. "
            "The input format is x,y,z,width,height,depth. "
            "Should be a comma separated string (e.g. 0,0,0,10,10,10).",
            parser=parse_bbox,
            metavar="BBOX",
        ),
    ] = None,
    layer: Annotated[
        list[str] | None,
        typer.Option(
            rich_help_panel="Partial download",
            help="Layers that should be downloaded. "
            "For multiple layers type: --layer color --layer segmentation",
        ),
    ] = None,
    mag: Annotated[
        list[Mag] | None,
        typer.Option(
            rich_help_panel="Partial download",
            help="Mags that should be downloaded. "
            "Should be number or minus separated string (e.g. 2 or 2-2-2). "
            "For multiple mags type: --mag 1 --mag 2",
            parser=parse_mag,
            metavar="MAG",
        ),
    ] = None,
) -> None:
    """Download a dataset from a WEBKNOSSOS server."""

    layers = layer if layer else None
    mags = mag if mag else None
    url = resolve_short_link(url)
    parsed = urlparse(url)
    domain = f"{parsed.scheme}://{parsed.netloc}"

    with webknossos_context(url=domain, token=token):
        if re.match(_DATASET_URL_REGEX, url) or re.match(
            _DATASET_DEPRECATED_URL_REGEX, url
        ):
            RemoteDataset.open(url).download(
                path=target,
                bbox=bbox,
                layers=layers,
                mags=mags,
            )
        elif re.match(_ANNOTATION_URL_REGEX, url):
            Annotation.download(annotation_id_or_url=url).save(target)
        else:
            raise RuntimeError(
                "The provided URL does not lead to a dataset or annotation."
            )


if __name__ == "__main__":
    typer.run(main)
