import logging
import warnings
from inspect import getframeinfo, stack
from os import PathLike
from os.path import relpath
from typing import TYPE_CHECKING, Union

import numpy as np
from cluster_tools import Executor
from upath import UPath

from webknossos.dataset.sampling_modes import SamplingModes
from webknossos.dataset_properties import (
    DataFormat,
    LayerProperties,
    MagViewProperties,
)
from webknossos.geometry import Mag, Vec3Int, Vec3IntLike
from webknossos.geometry.mag import MagLike

from ._downsampling_utils import (
    calculate_default_coarsest_mag,
    calculate_mags_to_downsample,
    calculate_mags_to_upsample,
    determine_downsample_buffer_shape,
    determine_upsample_buffer_shape,
    downsample_cube_job,
    parse_interpolation_mode,
)
from ._upsampling_utils import upsample_cube_job
from .abstract_layer import AbstractLayer
from .view import (
    ArrayException,
    MagView,
    TensorStoreArray,
    View,
    Zarr3ArrayInfo,
    Zarr3Config,
)

if TYPE_CHECKING:
    from webknossos.dataset import Dataset, RemoteDataset
    from webknossos.dataset.layer import RemoteLayer

    from .segmentation_layer import SegmentationLayer

from webknossos.dataset.defaults import (
    DEFAULT_CHUNK_SHAPE,
    DEFAULT_SHARD_SHAPE,
)
from webknossos.utils import (
    cheap_resolve,
    copytree,
    dump_path,
    enrich_path,
    get_executor_for_args,
    is_fs_path,
    movetree,
    named_partial,
    rmtree,
    warn_deprecated,
)

logger = logging.getLogger(__name__)

COPY_COMPATIBLE_PROTOCOLS = ("", "file", "s3")


def _copy_job(args: tuple[View, View, int]) -> None:
    (source_view, target_view, _) = args
    # Copy the data form one view to the other in a buffered fashion
    target_view.write(source_view.read())


def _get_shard_shape(
    *,
    chunk_shape: Vec3Int,
    chunks_per_shard: Vec3IntLike | int | None,
    shard_shape: Vec3IntLike | int | None,
) -> Vec3Int | None:
    if shard_shape is not None and chunks_per_shard is not None:
        raise ValueError(
            "shard_shape and chunks_per_shard must not be specified at the same time."
        )

    elif shard_shape is not None:
        shard_shape = Vec3Int.from_vec_or_int(shard_shape)
        if shard_shape % chunk_shape != Vec3Int.zeros():
            raise ValueError(
                f"The chunk_shape {chunk_shape} must be a multiple of the shard_shape {shard_shape}."
            )
    elif chunks_per_shard is not None:
        warn_deprecated("chunks_per_shard", "shard_shape")
        shard_shape = Vec3Int.from_vec_or_int(chunks_per_shard) * (
            chunk_shape or DEFAULT_CHUNK_SHAPE
        )

    return shard_shape


def _is_foreign_mag(dataset_path: UPath, layer_name: str, mag_path: UPath) -> bool:
    return dataset_path / layer_name != cheap_resolve(mag_path).parent


def _find_mag_path(
    layer_path: UPath,
    mag_name: str | Mag,
    path: UPath | None = None,
) -> UPath:
    if path is not None:
        return path

    mag = Mag(mag_name)
    short_mag_file_path = layer_path / mag.to_layer_name()
    long_mag_file_path = layer_path / mag.to_long_layer_name()
    if short_mag_file_path.exists():
        return cheap_resolve(short_mag_file_path)
    elif long_mag_file_path.exists():
        return cheap_resolve(long_mag_file_path)
    else:
        raise FileNotFoundError(
            f"Could not find any valid mag `{mag}` in `{layer_path}`."
        )


class Layer(AbstractLayer):
    _dataset: "Dataset"
    _mags: dict[Mag, MagView["Layer"]]

    def __init__(
        self, dataset: "Dataset", properties: LayerProperties, read_only: bool
    ) -> None:
        """A Layer represents a portion of hierarchical data at multiple magnifications.

        A Layer consists of multiple MagViews, which store the same data in different magnifications.
        Layers are components of a Dataset and provide access to the underlying data arrays.

        Attributes:
            name (str): Name identifier for this layer
            dataset (Dataset): Parent dataset containing this layer
            path (Path): Filesystem path to this layer's data
            category (LayerCategoryType): Category of data (e.g. color, segmentation)
            dtype_per_layer (str): Deprecated, use dtype_per_channel. Data type used for the entire layer
            dtype_per_channel (np.dtype): Data type used per channel
            num_channels (int): Number of channels in the layer
            data_format (DataFormat): Format used to store the data
            default_view_configuration (LayerViewConfiguration | None): View configuration
            read_only (bool): Whether layer is read-only
            mags (dict[Mag, MagView]): Dictionary of magnification levels

        Args:
            dataset (Dataset): The parent dataset that contains this layer
            properties (LayerProperties): Properties defining this layer's attributes. Must contain num_channels.
            read_only (bool): Whether layer is read-only

        Raises:
            AssertionError: If properties.num_channels is None

        Note:
            Do not use this constructor manually. Instead use Dataset.add_layer() to create a Layer.
        """
        self._resolved_path: UPath = cheap_resolve(
            dataset.resolved_path / properties.name
        )
        super().__init__(dataset, properties, read_only)

    def _determine_read_only_and_path_for_mag(
        self, mag_properties: MagViewProperties
    ) -> tuple[bool, UPath]:
        if mag_properties.path is None:
            mag_path = _find_mag_path(self.resolved_path, mag_properties.mag)
        else:
            mag_path = enrich_path(mag_properties.path, self.dataset.resolved_path)

        mag_is_read_only = _is_foreign_mag(
            self.dataset.resolved_path, self.name, mag_path
        )
        return mag_is_read_only, mag_path

    @property
    def dataset(self) -> "Dataset":
        return self._dataset

    @property
    def path(self) -> UPath:
        """Gets the filesystem path to this layer's data. This is defined as a subdirectory of the dataset directory named like the layer.
        Therefore, this directory does not contain the actual data of any linked or remote layers or mags.

        Returns:
            UPath: Filesystem path to this layer's data directory

        Raises:
            AssertionError: If mags in layer point to different layers
        """
        return self.dataset.path / self.name

    @property
    def resolved_path(self) -> UPath:
        return self._resolved_path

    @property
    def is_foreign(self) -> bool:
        """Whether this layer's data is stored remotely relative to its dataset.
        Returns:
            bool: True if layer path parent differs from dataset path
        """
        return self.resolved_path.parent != self.dataset.resolved_path

    @property
    def is_remote_to_dataset(self) -> bool:
        warn_deprecated("is_remote_to_dataset", "is_foreign")
        return self.is_foreign

    @property
    def name(self) -> str:
        """
        Returns the name of the layer.
        """
        return self._name

    @name.setter
    def name(self, layer_name: str) -> None:
        """
        Renames the layer to `layer_name`. This changes the name of the directory on disk and updates the properties.
        Only layers on local file systems can be renamed.
        """
        from webknossos.dataset.layer.abstract_layer import _ALLOWED_LAYER_NAME_REGEX

        if layer_name == self.name:
            return
        self._ensure_metadata_writable()
        if not is_fs_path(self.path):
            raise RuntimeError(f"Cannot rename remote layer {self.path}")
        if layer_name in self.dataset.layers.keys():
            raise ValueError(
                f"Failed to rename layer {self.name} to {layer_name}: The new name already exists."
            )
        if _ALLOWED_LAYER_NAME_REGEX.match(layer_name) is None:
            raise ValueError(
                f"The layer name '{layer_name}' is invalid. It must only contain letters, numbers, underscores, hyphens and dots."
            )

        if self.path.exists():
            assert is_fs_path(self.dataset.path), (
                "Renaming layers is only supported for local paths."
            )
            self.path.rename(self.dataset.path / layer_name)
        self._path = self.dataset.path / layer_name
        self._resolved_path = cheap_resolve(self.dataset.resolved_path / layer_name)
        del self.dataset._layers[self.name]
        self.dataset._layers[layer_name] = self
        self._properties.name = layer_name
        self._name: str = layer_name

        # The MagViews need to be updated
        for mag in self._mags.values():
            if not self.is_mag_view_foreign(mag):
                mag._properties.path = dump_path(
                    self._resolved_path / mag.path.name, self.dataset.resolved_path
                )
            else:
                assert mag._properties.path is not None  # for type checking
            mag._path = (
                enrich_path(mag._properties.path, self.dataset.resolved_path)
                if mag._properties.path is not None
                else self._resolved_path / mag.path.name
            )
            # Deleting the dataset will close the file handle.
            # The new dataset will be opened automatically when needed.
            del mag._array

        self._save_layer_properties()

    def is_mag_view_foreign(self, mag: MagView) -> bool:
        """Check if this magnification's data is stored not as part of to the dataset.
        Returns:
            bool: True if data is stored in a different location than the parent dataset.
        """
        return mag.path.parent.parent != self.dataset.resolved_path

    def add_mag(
        self,
        mag: MagLike,
        *,
        chunk_shape: Vec3IntLike | int | None = None,
        shard_shape: Vec3IntLike | int | None = None,
        chunks_per_shard: int | Vec3IntLike | None = None,
        compress: bool | Zarr3Config = True,
    ) -> MagView["Layer"]:
        """Creates and adds a new magnification level to the layer.

        The new magnification can be configured with various storage parameters to
        optimize performance, notably `chunk_shape`, `shard_shape` and `compress`. Note that writing data which is not aligned with the blocks on disk may result in
        diminished performance, as full blocks will automatically be read to pad the write actions.

        Args:
            mag: Identifier for new magnification level
            chunk_shape: Shape of chunks for storage. Recommended (32,32,32) or (64,64,64). Defaults to (32,32,32).
            shard_shape: Shape of shards for storage. Must be a multiple of chunk_shape. If specified, chunks_per_shard must not be specified. Defaults to (1024, 1024, 1024).
            chunks_per_shard: Deprecated, use shard_shape. Number of chunks per shards. If specified, shard_shape must not be specified.
            compress: Whether to enable compression. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied. Defaults to True.

        Returns:
            MagView: View of newly created magnification level

        Raises:
            IndexError: If magnification already exists
            Warning: If chunk_shape is not optimal for WEBKNOSSOS performance
        """
        self._ensure_writable()
        # normalize the name of the mag
        mag = Mag(mag)

        chunk_shape = (
            DEFAULT_CHUNK_SHAPE
            if chunk_shape is None
            else Vec3Int.from_vec_or_int(chunk_shape)
        )
        shard_shape = _get_shard_shape(
            chunk_shape=chunk_shape,
            chunks_per_shard=chunks_per_shard,
            shard_shape=shard_shape,
        )
        if shard_shape is None:
            if self.data_format == DataFormat.Zarr:
                shard_shape = chunk_shape
            else:
                shard_shape = DEFAULT_SHARD_SHAPE

        if chunk_shape not in (Vec3Int.full(32), Vec3Int.full(64)):
            warnings.warn(
                "[INFO] `chunk_shape` of `32, 32, 32` or `64, 64, 64` is recommended for optimal "
                + f"performance in WEBKNOSSOS. Got {chunk_shape}."
            )

        if not shard_shape._is_power_of_two():
            raise ValueError(
                f"The shard shape must be a power of two. Got {shard_shape}."
            )

        if not chunk_shape._is_power_of_two():
            raise ValueError(
                f"The chunk shape must be a power of two. Got {chunk_shape}."
            )

        compression_mode = compress if compress is not None else True

        self._assert_mag_does_not_exist_yet(mag)
        mag_path = self._create_dir_for_mag(mag)

        mag_view = MagView["Layer"].create(
            self,
            mag,
            chunk_shape=chunk_shape,
            shard_shape=shard_shape,
            compression_mode=compression_mode,
            path=mag_path,
            read_only=False,
        )

        mag_view._array.resize(
            self.bounding_box.align_with_mag(mag, ceil=True).in_mag(mag)
        )

        self._mags[mag] = mag_view
        mag_array_info = mag_view.info
        self._properties.mags += [
            MagViewProperties(
                mag=Mag(mag_view.name),
                cube_length=(
                    mag_array_info.shard_shape.x
                    if mag_array_info.data_format == DataFormat.WKW
                    else None
                ),
                axis_order=(
                    dict(
                        zip(
                            ("c", "x", "y", "z"),
                            (0, *self.bounding_box.index_xyz),
                        )
                    )
                    if mag_array_info.data_format in (DataFormat.Zarr, DataFormat.Zarr3)
                    else None
                ),
                path=dump_path(mag_path, self.dataset.resolved_path),
            )
        ]

        self._save_layer_properties()

        return self._mags[mag]

    def _add_mag_for_existing_files(
        self,
        mag: MagLike,
        mag_path: UPath,
        read_only: bool,
        override_stored_path: str | None = None,
    ) -> MagView["Layer"]:
        """Creates a MagView for existing data files.

        Adds a magnification level by linking to data files that already exist
        on the filesystem.

        Args:
            mag: Identifier for magnification level

        Returns:
            MagView: View of existing magnification data

        Raises:
            AssertionError: If magnification already exists in layer
            ArrayException: If files cannot be opened as valid arrays
        """
        self._ensure_writable()
        mag = Mag(mag)
        assert mag not in self.mags, (
            f"Cannot add mag {mag} as it already exists for layer {self}"
        )
        self._setup_mag(mag, mag_path=mag_path, read_only=read_only)
        mag_view = self._mags[mag]
        mag_array_info = mag_view.info
        stored_path = (
            override_stored_path
            if override_stored_path is not None
            else dump_path(mag_path, self.dataset.resolved_path)
        )
        self._properties.mags.append(
            MagViewProperties(
                mag=mag,
                cube_length=(
                    mag_array_info.shard_shape.x
                    if mag_array_info.data_format == DataFormat.WKW
                    else None
                ),
                axis_order=(
                    {
                        key: value
                        for key, value in zip(
                            ("c", "x", "y", "z"),
                            (0, *self.bounding_box.index_xyz),
                        )
                    }
                    if mag_array_info.data_format in (DataFormat.Zarr, DataFormat.Zarr3)
                    else None
                ),
                path=stored_path,
            )
        )
        self._save_layer_properties()

        return mag_view

    def get_mag(self, mag: MagLike) -> MagView["Layer"]:
        return super().get_mag(mag)

    def get_or_add_mag(
        self,
        mag: MagLike,
        *,
        chunk_shape: Vec3IntLike | int | None = None,
        shard_shape: Vec3IntLike | int | None = None,
        chunks_per_shard: Vec3IntLike | int | None = None,
        compress: bool | Zarr3Config | None = None,
    ) -> MagView["Layer"]:
        """
        Creates a new mag and adds it to the dataset, in case it did not exist before.
        Then, returns the mag.

        See `add_mag` for more information.
        """

        # normalize the name of the mag
        mag = Mag(mag)

        if mag in self._mags.keys():
            mag_view = self._mags[mag]
            chunk_shape = Vec3Int.from_vec_or_int(
                chunk_shape or mag_view.info.chunk_shape
            )
            shard_shape = _get_shard_shape(
                chunk_shape=chunk_shape,
                chunks_per_shard=chunks_per_shard,
                shard_shape=shard_shape,
            )

            if chunk_shape is not None and mag_view.info.chunk_shape != chunk_shape:
                raise ValueError(
                    f"Cannot get_or_add_mag: The mag {mag} already exists, but the chunk shapes do not match. Expected {mag_view.info.chunk_shape}, got {chunk_shape}."
                )

            if shard_shape is not None and mag_view.info.shard_shape != shard_shape:
                raise ValueError(
                    f"Cannot get_or_add_mag: The mag {mag} already exists, but the shard shapes do not match. Expected {mag_view.info.shard_shape}, got {shard_shape}."
                )

            if (
                isinstance(compress, bool)
                and mag_view.info.compression_mode != compress
            ):
                raise ValueError(
                    f"Cannot get_or_add_mag: The mag {mag} already exists, but the compression modes do not match. Expected {mag_view.info.compression_mode}, got {compress}."
                )
            if isinstance(compress, Zarr3Config):
                if not self.data_format != DataFormat.Zarr3:
                    raise ValueError(
                        "Cannot get_or_add_mag: A Zarr3 config can only be supplied for Zarr3 layers."
                    )
                assert isinstance(mag_view.info, Zarr3ArrayInfo)
                if mag_view.info.codecs != compress.codecs:
                    raise ValueError(
                        f"Cannot get_or_add_mag: The mag {mag} already exists, but the codecs do not match. Expected {mag_view.info.codecs}, got {compress.codecs}."
                    )
                if mag_view.info.chunk_key_encoding != compress.chunk_key_encoding:
                    raise ValueError(
                        f"Cannot get_or_add_mag: The mag {mag} already exists, but the chunk key encoding does not match. Expected {mag_view.info.chunk_key_encoding}, got {compress.chunk_key_encoding}."
                    )

            return self.get_mag(mag)
        else:
            chunk_shape = Vec3Int.from_vec_or_int(chunk_shape or DEFAULT_CHUNK_SHAPE)
            shard_shape = _get_shard_shape(
                chunk_shape=chunk_shape,
                chunks_per_shard=chunks_per_shard,
                shard_shape=shard_shape,
            )
            return self.add_mag(
                mag,
                chunk_shape=chunk_shape,
                shard_shape=shard_shape,
                compress=compress if compress is not None else True,
            )

    def get_finest_mag(self) -> MagView["Layer"]:
        return super().get_finest_mag()

    def delete_mag(self, mag: MagLike) -> None:
        """
        Deletes the MagView from the `datasource-properties.json` and the data from disk.

        This function raises an `IndexError` if the specified `mag` does not exist.
        """
        self._ensure_writable()
        mag = Mag(mag)
        if mag not in self.mags.keys():
            raise IndexError(
                f"Deleting mag {mag} failed. There is no mag with this name"
            )
        mag_view = self.get_mag(mag)

        full_path = self._mags[mag].path
        del self._mags[mag]
        self._properties.mags = [
            res for res in self._properties.mags if Mag(res.mag) != mag
        ]
        self._save_layer_properties()
        if not self.is_mag_view_foreign(mag_view):
            # delete files on disk
            rmtree(full_path)
        else:
            # delete symlinks only
            short_mag_file_path = self.path / mag.to_layer_name()
            long_mag_file_path = self.path / mag.to_long_layer_name()
            if short_mag_file_path.exists():
                short_mag_file_path.unlink()
            elif long_mag_file_path.exists():
                long_mag_file_path.unlink()

    def add_copy_mag(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        extend_layer_bounding_box: bool = True,
        chunk_shape: Vec3IntLike | int | None = None,
        shard_shape: Vec3IntLike | int | None = None,
        chunks_per_shard: Vec3IntLike | int | None = None,
        compress: bool | None = None,
        exists_ok: bool = False,
        executor: Executor | None = None,
        progress_desc: str | None = None,
    ) -> MagView:
        """Deprecated. Use `Layer.add_mag_as_copy` instead."""
        warn_deprecated("add_copy_mag", "add_mag_as_copy")
        return self.add_mag_as_copy(
            foreign_mag_view_or_path,
            extend_layer_bounding_box=extend_layer_bounding_box,
            chunk_shape=chunk_shape,
            shard_shape=shard_shape,
            chunks_per_shard=chunks_per_shard,
            compress=compress,
            exists_ok=exists_ok,
            executor=executor,
            progress_desc=progress_desc,
        )

    def add_mag_as_copy(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        extend_layer_bounding_box: bool = True,
        chunk_shape: Vec3IntLike | int | None = None,
        shard_shape: Vec3IntLike | int | None = None,
        chunks_per_shard: Vec3IntLike | int | None = None,
        compress: bool | Zarr3Config | None = None,
        exists_ok: bool = False,
        executor: Executor | None = None,
        progress_desc: str | None = None,
    ) -> MagView["Layer"]:
        """
        Copies the data at `foreign_mag_view_or_path` which can belong to another dataset
        to the current dataset. Additionally, the relevant information from the
        `datasource-properties.json` of the other dataset are copied, too.
        """
        self._ensure_writable()
        foreign_mag_view = MagView._ensure_mag_view(foreign_mag_view_or_path)

        if progress_desc is None:
            progress_desc = f"Copying mag {foreign_mag_view.mag.to_layer_name()} from {foreign_mag_view.layer} to {self}"

        has_same_shapes = (
            (
                chunk_shape is None
                or Vec3Int.from_vec_or_int(chunk_shape)
                == foreign_mag_view.info.chunk_shape
            )
            and (
                shard_shape is None
                or Vec3Int.from_vec_or_int(shard_shape)
                == foreign_mag_view.info.shard_shape
            )
            and (
                chunks_per_shard is None
                or Vec3Int.from_vec_or_int(chunks_per_shard)
                == foreign_mag_view.info.chunks_per_shard
            )
        )
        has_same_format = self.data_format == foreign_mag_view.info.data_format
        if compress is None:
            has_same_compression = True
        elif isinstance(compress, Zarr3Config) and isinstance(
            foreign_mag_view.info, Zarr3ArrayInfo
        ):
            has_same_compression = compress == foreign_mag_view.info.zarr3_config
        else:
            has_same_compression = compress == foreign_mag_view.info.compression_mode

        uses_compatible_protocols = (
            foreign_mag_view.path.protocol in COPY_COMPATIBLE_PROTOCOLS
            and (self.path / foreign_mag_view.mag.to_layer_name()).protocol
            in COPY_COMPATIBLE_PROTOCOLS
        )

        if (
            has_same_shapes
            and has_same_format
            and has_same_compression
            and uses_compatible_protocols
        ):
            logger.debug(
                f"Optimization: Copying files from {foreign_mag_view.path} to {self.path}/{foreign_mag_view.mag} directly without re-encoding."
            )
            return self._add_fs_copy_mag(
                foreign_mag_view,
                extend_layer_bounding_box=extend_layer_bounding_box,
                exists_ok=exists_ok,
                progress_desc=progress_desc,
            )

        chunk_shape = Vec3Int.from_vec_or_int(
            chunk_shape or foreign_mag_view.info.chunk_shape
        )
        if chunks_per_shard is not None:
            if shard_shape is None:
                shard_shape = Vec3Int.from_vec_or_int(chunks_per_shard) * chunk_shape
            else:
                raise ValueError(
                    "shard_shape and chunks_per_shard must not be specified at the same time."
                )

        compress = (
            compress if compress is not None else foreign_mag_view.info.compression_mode
        )
        shard_shape = shard_shape or foreign_mag_view.info.shard_shape
        if exists_ok:
            mag_view = self.get_or_add_mag(
                mag=foreign_mag_view.mag,
                chunk_shape=chunk_shape,
                shard_shape=shard_shape,
                compress=compress,
            )
        else:
            self._assert_mag_does_not_exist_yet(foreign_mag_view.mag)
            mag_view = self.add_mag(
                mag=foreign_mag_view.mag,
                chunk_shape=chunk_shape,
                shard_shape=shard_shape,
                compress=compress,
            )

        if extend_layer_bounding_box:
            self.bounding_box = self.bounding_box.extended_by(
                foreign_mag_view.layer.bounding_box
            )

        # use the target shard shape for the copy operation
        copy_shape = mag_view.info.shard_shape * mag_view.mag.to_vec3_int()
        foreign_mag_view.for_zipped_chunks(
            func_per_chunk=_copy_job,
            target_view=mag_view,
            executor=executor,
            progress_desc=progress_desc,
            source_chunk_shape=copy_shape,
            target_chunk_shape=copy_shape,
        )

        return mag_view

    def add_symlink_mag(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        make_relative: bool = False,
        extend_layer_bounding_box: bool = True,
    ) -> MagView["Layer"]:
        """Deprecated. Use `Layer.add_mag_as_ref` instead.

        Creates a symlink to the data at `foreign_mag_view_or_path` which belongs to another dataset.
        The relevant information from the `datasource-properties.json` of the other dataset is copied to this dataset.
        Note: If the other dataset modifies its bounding box afterwards, the change does not affect this properties
        (or vice versa).
        If make_relative is True, the symlink is made relative to the current dataset path.
        Symlinked mags can only be added to layers on local file systems.
        """
        self._ensure_writable()
        warnings.warn(
            "Using symlinks is deprecated and will be removed in a future version. "
            + "Use `add_mag_as_ref` instead, which adds the mag as a reference to this layer.",
            DeprecationWarning,
            stacklevel=2,
        )
        foreign_mag_view = MagView._ensure_mag_view(foreign_mag_view_or_path)
        self._assert_mag_does_not_exist_yet(foreign_mag_view.mag)

        self_path = self.path
        self_resolved_path = self.resolved_path
        dataset_resolved_path = self.dataset.resolved_path
        assert (
            is_fs_path(self_path)
            and is_fs_path(self_resolved_path)
            and is_fs_path(dataset_resolved_path)
        ), f"Cannot create symlinks in non-local layer {self_path}"
        assert is_fs_path(foreign_mag_view.path), (
            f"Cannot create symlink to non-local mag {foreign_mag_view.path}"
        )

        foreign_normalized_mag_path = (
            UPath(relpath(foreign_mag_view.path, self_resolved_path))
            if make_relative
            else foreign_mag_view.path
        )

        (self_path / str(foreign_mag_view.mag)).symlink_to(foreign_normalized_mag_path)

        new_mag_path = (
            relpath(foreign_mag_view.path, dataset_resolved_path)
            if make_relative
            else str(foreign_mag_view.path.resolve())
        )

        mag = self._add_mag_for_existing_files(
            foreign_mag_view.mag,
            mag_path=UPath(foreign_mag_view.path),
            override_stored_path=new_mag_path,
            read_only=True,
        )

        if extend_layer_bounding_box:
            self.bounding_box = self.bounding_box.extended_by(
                foreign_mag_view.layer.bounding_box
            )
        return mag

    def add_remote_mag(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        extend_layer_bounding_box: bool = True,
    ) -> MagView["Layer"]:
        """Deprecated. Use `Layer.add_mag_as_ref` instead."""
        warn_deprecated("add_remote_mag", "add_mag_as_ref")
        return self.add_mag_as_ref(
            foreign_mag_view_or_path,
            extend_layer_bounding_box=extend_layer_bounding_box,
        )

    def add_mag_as_ref(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        mag: MagLike | None = None,
        extend_layer_bounding_box: bool = True,
    ) -> MagView["Layer"]:
        """
        Adds the mag at `foreign_mag_view_or_path` which belongs to foreign dataset.
        The relevant information from the `datasource-properties.json` of the other dataset is copied to this dataset.
        Note: If the other dataset modifies its bounding box afterwards, the change does not affect this properties
        (or vice versa).
        """
        self._ensure_writable()
        foreign_mag_view = MagView._ensure_mag_view(foreign_mag_view_or_path)
        mag = Mag(mag) if mag is not None else foreign_mag_view.mag
        self._assert_mag_does_not_exist_yet(mag)

        assert self.data_format == foreign_mag_view.info.data_format, (
            f"Cannot add a remote mag whose data format {foreign_mag_view.info.data_format} "
            + f"does not match the layers data format {self.data_format}"
        )
        assert self.dtype_per_channel == foreign_mag_view.get_dtype(), (
            f"The dtype/elementClass of the remote mag {foreign_mag_view.get_dtype()} "
            + f"must match the layer's dtype {self.dtype_per_channel}"
        )

        self._setup_mag(mag, foreign_mag_view.path, read_only=True)

        # since the remote mag view might belong to another dataset, it's property's path might be None, therefore, we get the path from the mag_view itself instead of it's properties
        self._properties.mags.append(
            MagViewProperties(
                mag=mag,
                path=dump_path(foreign_mag_view.path, self.dataset.resolved_path),
                cube_length=foreign_mag_view._properties.cube_length,
                axis_order=foreign_mag_view._properties.axis_order,
            )
        )
        self._save_layer_properties()

        if extend_layer_bounding_box:
            self.bounding_box = self.bounding_box.extended_by(
                foreign_mag_view.layer.bounding_box
            )

        return self.get_mag(mag)

    def _add_fs_copy_mag(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        extend_layer_bounding_box: bool = True,
        exists_ok: bool = False,
        progress_desc: str | None = None,
    ) -> MagView["Layer"]:
        self._ensure_writable()

        foreign_mag_view = MagView._ensure_mag_view(foreign_mag_view_or_path)
        self._assert_mag_does_not_exist_yet(foreign_mag_view.mag)

        assert foreign_mag_view.info.data_format == self.data_format, (
            f"Cannot use file-based copy, because the foreign data format {foreign_mag_view.info.data_format} does not match the layer's data format {self.data_format}."
        )

        mag_path = self.path / str(foreign_mag_view.mag)
        if not exists_ok and mag_path.exists():
            raise FileExistsError(
                f"Cannot copy {foreign_mag_view.path} to {mag_path} because it already exists."
            )
        copytree(foreign_mag_view.path, mag_path, progress_desc=progress_desc)

        mag = self._add_mag_for_existing_files(
            foreign_mag_view.mag, mag_path=mag_path, read_only=False
        )

        if extend_layer_bounding_box:
            self.bounding_box = self.bounding_box.extended_by(
                foreign_mag_view.layer.bounding_box
            )

        return mag

    def add_fs_copy_mag(
        self,
        foreign_mag_view_or_path: PathLike | UPath | str | MagView,
        *,
        extend_layer_bounding_box: bool = True,
        exists_ok: bool = False,
    ) -> MagView["Layer"]:
        """Deprecated. File-copy is automatically selected when using `Layer.add_mag_as_copy`.

        Copies the data at `foreign_mag_view_or_path` which belongs to another dataset to the current dataset via the filesystem.
        Additionally, the relevant information from the `datasource-properties.json` of the other dataset are copied, too.
        """
        caller = getframeinfo(stack()[2][0])
        warnings.warn(
            f"[DEPRECATION] Direct use of `Layer.add_fs_copy_mag` is deprecated, please use `Layer.add_mag_as_copy` instead (see {caller.filename}:{caller.lineno}), which automatically uses file-based copy if available.",
            DeprecationWarning,
            stacklevel=2,
        )
        return self._add_fs_copy_mag(
            foreign_mag_view_or_path,
            extend_layer_bounding_box=extend_layer_bounding_box,
            exists_ok=exists_ok,
        )

    def add_mag_from_zarrarray(
        self,
        mag: MagLike,
        path: PathLike | UPath,
        *,
        move: bool = False,
        extend_layer_bounding_box: bool = True,
    ) -> MagView["Layer"]:
        """
        Copies the data at `path` to the current layer of the dataset
        via the filesystem and adds it as `mag`. When `move` flag is set
        the array is moved, otherwise a copy of the zarrarray is created.
        """
        self._ensure_writable()
        source_path = enrich_path(path, self.dataset.resolved_path)

        try:
            TensorStoreArray.open(source_path)
        except ArrayException as e:
            raise ValueError(
                "The given path does not lead to a valid Zarr Array: "
            ) from e
        else:
            mag = Mag(mag)
            self._assert_mag_does_not_exist_yet(mag)
            mag_path = self.path / str(mag)
            if move:
                movetree(source_path, mag_path)
            else:
                copytree(source_path, mag_path)

            mag_view = self._add_mag_for_existing_files(
                mag, mag_path=mag_path, read_only=False
            )

            if extend_layer_bounding_box:
                # assumption: the topleft of the bounding box is still the same, the size might differ
                # axes of the layer and the zarr array provided are the same
                zarray_size = (
                    mag_view.info.shape[mag_view.info.dimension_names.index(axis)]
                    for axis in self.bounding_box.axes
                    if axis != "c"
                )
                size = self.bounding_box.size.pairmax(zarray_size)
                self.bounding_box = self.bounding_box.with_size(size)

            return mag_view

    def _create_dir_for_mag(self, mag: MagLike) -> UPath:
        mag_name = Mag(mag).to_layer_name()
        full_path = self.resolved_path / mag_name
        full_path.mkdir(parents=True, exist_ok=True)
        full_path = cheap_resolve(full_path)
        return full_path

    def _get_dataset_from_align_with_other_layers(
        self, align_with_other_layers: Union[bool, "Dataset", "RemoteDataset"]
    ) -> Union["Dataset", "RemoteDataset"] | None:
        if isinstance(align_with_other_layers, bool):
            return self.dataset if align_with_other_layers else None
        else:
            return align_with_other_layers

    def downsample(
        self,
        *,
        from_mag: Mag | None = None,
        coarsest_mag: Mag | None = None,
        interpolation_mode: str = "default",
        compress: bool | Zarr3Config = True,
        sampling_mode: str | SamplingModes = SamplingModes.ANISOTROPIC,
        align_with_other_layers: Union[bool, "Dataset", "RemoteDataset"] = True,
        buffer_shape: Vec3Int | None = None,
        force_sampling_scheme: bool = False,
        allow_overwrite: bool = False,
        only_setup_mags: bool = False,
        executor: Executor | None = None,
    ) -> None:
        """Downsample data from a source magnification to coarser magnifications.

        Downsamples the data starting from from_mag until a magnification is >= max(coarsest_mag).
        Different sampling modes control how dimensions are downsampled.

        Args:
            from_mag (Mag | None): Source magnification to downsample from. Defaults to highest existing mag.
            coarsest_mag (Mag | None): Target magnification to stop at. Defaults to calculated value.
            interpolation_mode (str): Interpolation method to use. Defaults to "default".
                Supported modes: "median", "mode", "nearest", "bilinear", "bicubic"
            compress (bool | Zarr3Config): Whether to compress the generated magnifications. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied. Defaults to True.
            sampling_mode (str | SamplingModes): How dimensions should be downsampled.
                Defaults to ANISOTROPIC.
            align_with_other_layers (bool | Dataset | RemoteDataset): Whether to align with other layers. True by default.
            buffer_shape (Vec3Int | None): Shape of processing buffer. Defaults to None.
            force_sampling_scheme (bool): Force invalid sampling schemes. Defaults to False.
            allow_overwrite (bool): Whether existing mags can be overwritten. False by default.
            only_setup_mags (bool): Only create mags without data. False by default.
            executor (Executor | None): Executor for parallel processing. None by default.

        Raises:
            AssertionError: If from_mag does not exist
            RuntimeError: If sampling scheme produces invalid magnifications
            AttributeError: If sampling_mode is invalid

        Examples:
            ```python
            from webknossos import SamplingModes

            # let 'layer' be a `Layer` with only `Mag(1)`
            assert "1" in self.mags.keys()

            layer.downsample(
                coarsest_mag=Mag(4),
                sampling_mode=SamplingModes.ISOTROPIC
            )

            assert "2" in self.mags.keys()
            assert "4" in self.mags.keys()
            ```
        """

        if from_mag is None:
            assert len(self.mags.keys()) > 0, (
                "Failed to downsample data because no existing mag was found."
            )
            from_mag = max(self.mags.keys())

        assert from_mag in self.mags.keys(), (
            f"Failed to downsample data. The from_mag ({from_mag.to_layer_name()}) does not exist."
        )

        if coarsest_mag is None:
            coarsest_mag = calculate_default_coarsest_mag(self.bounding_box.size_xyz)

        sampling_mode = SamplingModes.parse(sampling_mode)

        if self._properties.bounding_box.size.z == 1:
            if sampling_mode != SamplingModes.CONSTANT_Z:
                warnings.warn(
                    "[INFO] The sampling_mode was changed to 'CONSTANT_Z'. Downsampling 2D data with a different sampling mode mixes in black and thus leads to darkened images."
                )
                sampling_mode = SamplingModes.CONSTANT_Z

        voxel_size: tuple[float, float, float] | None = None
        if sampling_mode == SamplingModes.ANISOTROPIC:
            voxel_size = self.dataset.voxel_size
        elif sampling_mode == SamplingModes.ISOTROPIC:
            voxel_size = None
        elif sampling_mode == SamplingModes.CONSTANT_Z:
            coarsest_mag_with_fixed_z = coarsest_mag.to_list()
            coarsest_mag_with_fixed_z[2] = from_mag.to_list()[2]
            coarsest_mag = Mag(coarsest_mag_with_fixed_z)
            voxel_size = None
        else:
            raise AttributeError(
                f"Downsampling failed: {sampling_mode} is not a valid SamplingMode ({SamplingModes.ANISOTROPIC}, {SamplingModes.ISOTROPIC}, {SamplingModes.CONSTANT_Z})"
            )

        dataset_to_align_with = self._get_dataset_from_align_with_other_layers(
            align_with_other_layers
        )
        mags_to_downsample = calculate_mags_to_downsample(
            from_mag, coarsest_mag, dataset_to_align_with, voxel_size
        )

        if len(set([max(m.to_list()) for m in mags_to_downsample])) != len(
            mags_to_downsample
        ):
            msg = (
                f"[INFO] The downsampling scheme contains multiple magnifications with the same maximum value. This is not supported by WEBKNOSSOS. "
                f"Consider using a different sampling mode (e.g. {SamplingModes.ISOTROPIC}). "
                f"The calculated downsampling scheme is: {[m.to_layer_name() for m in mags_to_downsample]}"
            )
            if force_sampling_scheme:
                warnings.warn(msg)
            else:
                raise RuntimeError(msg)

        for prev_mag, target_mag in zip(
            [from_mag] + mags_to_downsample[:-1], mags_to_downsample
        ):
            self.downsample_mag(
                from_mag=prev_mag,
                target_mag=target_mag,
                interpolation_mode=interpolation_mode,
                compress=compress,
                buffer_shape=buffer_shape,
                allow_overwrite=allow_overwrite,
                only_setup_mag=only_setup_mags,
                executor=executor,
            )

    def downsample_mag(
        self,
        from_mag: Mag,
        target_mag: Mag,
        *,
        interpolation_mode: str = "default",
        compress: bool | Zarr3Config = True,
        buffer_shape: Vec3Int | None = None,
        allow_overwrite: bool = False,
        only_setup_mag: bool = False,
        executor: Executor | None = None,
    ) -> None:
        """Performs a single downsampling step between magnification levels.

        Args:
            from_mag: Source magnification level
            target_mag: Target magnification level
            interpolation_mode: Method for interpolation ("median", "mode", "nearest", "bilinear", "bicubic")
            compress: Whether to compress target data. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied. Defaults to True.
            buffer_shape: Shape of processing buffer
            allow_overwrite: Whether to allow overwriting existing mag
            only_setup_mag: Only create mag without data. This parameter can be used to prepare for parallel downsampling of multiple layers while avoiding parallel writes with outdated updates to the datasource-properties.json file.
            executor: Executor for parallel processing

        Raises:
            AssertionError: If from_mag doesn't exist or target exists without overwrite"""
        self._dataset._ensure_writable()

        assert from_mag in self.mags.keys(), (
            f"Failed to downsample data. The from_mag ({from_mag.to_layer_name()}) does not exist."
        )

        parsed_interpolation_mode = parse_interpolation_mode(
            interpolation_mode, self.category
        )

        assert from_mag <= target_mag
        assert allow_overwrite or target_mag not in self.mags, (
            "The target mag already exists. Pass allow_overwrite=True if you want to overwrite it."
        )

        prev_mag_view = self.mags[from_mag]

        mag_factors = target_mag.to_vec3_int() // from_mag.to_vec3_int()

        if target_mag in self.mags.keys() and allow_overwrite:
            target_mag_view = self.get_mag(target_mag)
        else:
            # initialize the new mag
            target_mag_view = self._initialize_mag_from_other_mag(
                target_mag,
                prev_mag_view,
                compress=compress,
            )

        if only_setup_mag:
            return

        bb_mag1 = self.bounding_box.align_with_mag(target_mag, ceil=True)

        # Get target view
        target_view = target_mag_view.get_view(absolute_bounding_box=bb_mag1)

        source_view = prev_mag_view.get_view(
            absolute_bounding_box=bb_mag1,
            read_only=True,
        )

        # perform downsampling
        with get_executor_for_args(None, executor) as executor:
            if buffer_shape is None:
                buffer_shape = determine_downsample_buffer_shape(prev_mag_view.info)
            func = named_partial(
                downsample_cube_job,
                mag_factors=mag_factors,
                interpolation_mode=parsed_interpolation_mode,
                buffer_shape=buffer_shape,
            )

            source_view.for_zipped_chunks(
                # this view is restricted to the bounding box specified in the properties
                func,
                target_view=target_view,
                executor=executor,
                progress_desc=f"Downsampling layer {self.name} from Mag {from_mag} to Mag {target_mag}",
            )

    def redownsample(
        self,
        *,
        interpolation_mode: str = "default",
        compress: bool | Zarr3Config = True,
        buffer_shape: Vec3Int | None = None,
        executor: Executor | None = None,
    ) -> None:
        """Recompute all downsampled magnifications from base mag.

        Used after modifying data in the base magnification to update
        all derived magnifications.

        Args:
            interpolation_mode: Method for interpolation
            compress: Whether to compress recomputed data. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied. Defaults to True.
            buffer_shape: Shape of processing buffer
            executor: Executor for parallel processing
        """

        mags = sorted(self.mags.keys(), key=lambda m: m.to_list())
        if len(mags) <= 1:
            # No downsampled magnifications exist. Thus, there's nothing to do.
            return
        from_mag = mags[0]
        target_mags = mags[1:]
        self.downsample_mag_list(
            from_mag=from_mag,
            target_mags=target_mags,
            interpolation_mode=interpolation_mode,
            compress=compress,
            buffer_shape=buffer_shape,
            allow_overwrite=True,
            executor=executor,
        )

    def downsample_mag_list(
        self,
        from_mag: Mag,
        target_mags: list[Mag],
        *,
        interpolation_mode: str = "default",
        compress: bool | Zarr3Config = True,
        buffer_shape: Vec3Int | None = None,
        allow_overwrite: bool = False,
        only_setup_mags: bool = False,
        executor: Executor | None = None,
    ) -> None:
        """Downsample data iteratively through multiple magnification levels.

        Performs sequential downsampling from from_mag through each magnification
        in target_mags in order.

        Args:
            from_mag (Mag): Source magnification to start from
            target_mags (List[Mag]): Ordered list of target magnifications
            interpolation_mode (str): Interpolation method to use. Defaults to "default".
            compress (bool | Zarr3Config): Whether to compress outputs. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied. Defaults to True.
            buffer_shape (Vec3Int | None): Shape of processing buffer.
            allow_overwrite (bool): Whether to allow overwriting mags. Defaults to False.
            only_setup_mags (bool): Only create mag structures without data. Defaults to False.
            executor (Executor | None): Executor for parallel processing.

        Raises:
            AssertionError: If from_mag doesn't exist or target mags not in ascending order

        See downsample_mag() for more details on parameters.
        """
        assert from_mag in self.mags.keys(), (
            f"Failed to downsample data. The from_mag ({from_mag}) does not exist."
        )

        # The lambda function is important because 'sorted(target_mags)' would only sort by the maximum element per mag
        target_mags = sorted(target_mags, key=lambda m: m.to_list())

        for i in range(len(target_mags) - 1):
            assert np.less_equal(
                target_mags[i].to_np(), target_mags[i + 1].to_np()
            ).all(), (
                f"Downsampling failed: cannot downsample {target_mags[i].to_layer_name()} to {target_mags[i + 1].to_layer_name()}. "
                f"Check 'target_mags' ({', '.join([str(mag) for mag in target_mags])}): each pair of adjacent Mags results in a downsampling step."
            )

        source_mag = from_mag
        for target_mag in target_mags:
            self.downsample_mag(
                source_mag,
                target_mag,
                interpolation_mode=interpolation_mode,
                compress=compress,
                buffer_shape=buffer_shape,
                allow_overwrite=allow_overwrite,
                only_setup_mag=only_setup_mags,
                executor=executor,
            )
            source_mag = target_mag

    def upsample(
        self,
        from_mag: Mag,
        *,
        finest_mag: Mag = Mag(1),
        compress: bool | Zarr3Config = True,
        sampling_mode: str | SamplingModes = SamplingModes.ANISOTROPIC,
        align_with_other_layers: Union[bool, "Dataset"] = True,
        buffer_shape: Vec3IntLike | None = None,
        executor: Executor | None = None,
    ) -> None:
        """Upsample data to finer magnifications.

        Upsamples from a coarser magnification to a sequence of finer magnifications,
        stopping at finest_mag. The sampling mode controls how dimensions are handled.

        Args:
            from_mag (Mag): Source coarse magnification
            finest_mag (Mag): Target finest magnification (default Mag(1))
            compress (bool | Zarr3Config): Whether to compress upsampled data. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied. Defaults to True.
            sampling_mode (str | SamplingModes): How dimensions should be upsampled:
                - 'anisotropic': Equalizes voxel dimensions based on voxel_size
                - 'isotropic': Equal upsampling in all dimensions
                - 'constant_z': Only upsamples x/y dimensions. z remains unchanged.
            align_with_other_layers: Whether to align mags with others. Defaults to True.
            buffer_shape (Vec3IntLike | None): Shape of processing buffer.
            executor (Executor | None): Executor for parallel processing.

        Raises:
            AssertionError: If from_mag doesn't exist or finest_mag invalid
            AttributeError: If sampling_mode is invalid
        """

        self._dataset._ensure_writable()

        assert from_mag in self.mags.keys(), (
            f"Failed to upsample data. The from_mag ({from_mag.to_layer_name()}) does not exist."
        )

        sampling_mode = SamplingModes.parse(sampling_mode)

        voxel_size: tuple[float, float, float] | None = None
        if sampling_mode == SamplingModes.ANISOTROPIC:
            voxel_size = self.dataset.voxel_size
        elif sampling_mode == SamplingModes.ISOTROPIC:
            voxel_size = None
        elif sampling_mode == SamplingModes.CONSTANT_Z:
            finest_mag_with_fixed_z = finest_mag.to_list()
            finest_mag_with_fixed_z[2] = from_mag.to_list()[2]
            finest_mag = Mag(finest_mag_with_fixed_z)
            voxel_size = None
        else:
            raise AttributeError(
                f"Upsampling failed: {sampling_mode} is not a valid UpsamplingMode ({SamplingModes.ANISOTROPIC}, {SamplingModes.ISOTROPIC}, {SamplingModes.CONSTANT_Z})"
            )

        dataset_to_align_with = self._get_dataset_from_align_with_other_layers(
            align_with_other_layers
        )
        mags_to_upsample = calculate_mags_to_upsample(
            from_mag, finest_mag, dataset_to_align_with, voxel_size
        )

        for prev_mag, target_mag in zip(
            [from_mag] + mags_to_upsample[:-1], mags_to_upsample
        ):
            assert prev_mag > target_mag
            assert target_mag not in self.mags

            prev_mag_view = self.mags[prev_mag]

            mag_factors = [
                t / s for (t, s) in zip(target_mag.to_list(), prev_mag.to_list())
            ]

            # initialize the new mag
            target_mag_view = self._initialize_mag_from_other_mag(
                target_mag,
                prev_mag_view,
                compress=compress,
            )

            # We need to make sure the layer's bounding box is aligned
            # with the previous mag. Otherwise, `for_zipped_chunks` will fail.
            # Saving the original layer bbox for later restore
            old_layer_bbox = self.bounding_box
            self.bounding_box = prev_mag_view.bounding_box
            bbox_mag1 = self.bounding_box.align_with_mag(prev_mag, ceil=True)
            # Get target view
            target_view = target_mag_view.get_view(absolute_bounding_box=bbox_mag1)

            # perform upsampling
            with get_executor_for_args(None, executor) as actual_executor:
                if buffer_shape is None:
                    buffer_shape = determine_upsample_buffer_shape(prev_mag_view.info)
                else:
                    buffer_shape = Vec3Int.from_vec_or_int(buffer_shape)
                func = named_partial(
                    upsample_cube_job,
                    mag_factors=mag_factors,
                    buffer_shape=buffer_shape,
                )
                prev_mag_view.get_view(
                    absolute_bounding_box=bbox_mag1
                ).for_zipped_chunks(
                    # this view is restricted to the bounding box specified in the properties
                    func,
                    target_view=target_view,
                    executor=actual_executor,
                    progress_desc=f"Upsampling from Mag {prev_mag} to Mag {target_mag}",
                )
            # Restoring the original layer bbox
            self.bounding_box = old_layer_bbox

    def _initialize_mag_from_other_mag(
        self,
        new_mag_name: str | Mag,
        other_mag: MagView,
        compress: bool | Zarr3Config,
    ) -> MagView["Layer"]:
        """Creates a new magnification based on settings from existing mag.

        Args:
            new_mag_name: Name/identifier for new mag
            other_mag: Existing mag to copy settings from
            compress: Whether to enable compression. For Zarr3 datasets, codec configuration and chunk key encoding may also be supplied.

        Returns:
            MagView: View of newly created magnification
        """
        return self.add_mag(
            new_mag_name,
            chunk_shape=other_mag.info.chunk_shape,
            shard_shape=other_mag.info.shard_shape,
            compress=compress,
        )

    def __repr__(self) -> str:
        return f"Layer({repr(self.name)}, dtype_per_channel={self.dtype_per_channel}, num_channels={self.num_channels})"

    def as_segmentation_layer(self) -> "SegmentationLayer":
        """Casts into SegmentationLayer."""
        from .segmentation_layer.segmentation_layer import SegmentationLayer

        if isinstance(self, SegmentationLayer):
            return self
        else:
            raise TypeError(f"self is not a SegmentationLayer. Got: {type(self)}")

    @classmethod
    def _ensure_layer(
        cls, layer: Union[str, PathLike, UPath, "Layer", "RemoteLayer"]
    ) -> Union["Layer", "RemoteLayer"]:
        # local import to prevent circular dependency
        from webknossos.dataset import Dataset
        from webknossos.dataset.layer.remote_layer import RemoteLayer

        if isinstance(layer, Layer | RemoteLayer):
            return layer

        layer_path = UPath(layer)
        return Dataset.open(layer_path.parent).get_layer(layer_path.name)
