import logging
from unittest import mock

import pytest
from scim2_models import SearchRequest

from canaille.app import models
from canaille.scim.casting import user_from_scim_to_canaille
from canaille.scim.client import user_from_canaille_to_scim_client
from canaille.scim.models import EnterpriseUser
from canaille.scim.models import User as SCIMUser


def test_scim_client_user_save_and_delete(scim_client_for_trusted_client, backend):
    """Test that SCIM client can create and delete users via SCIM protocol."""
    User = scim_client_for_trusted_client.get_resource_model("User")

    response = scim_client_for_trusted_client.query(User)
    assert not response.resources

    alice = models.User(
        formatted_name="Alice Alice",
        family_name="Alice",
        user_name="alice",
        emails=["john@doe.test", "johhny@doe.test"],
    )
    backend.save(alice)

    response = scim_client_for_trusted_client.query(User)
    assert len(response.resources) == 1
    assert response.resources[0].user_name == "alice"

    backend.delete(alice)
    response = scim_client_for_trusted_client.query(User)
    assert not response.resources


def test_scim_client_group_save_and_delete(
    scim_client_for_trusted_client, backend, user
):
    """Test that SCIM client can create and delete groups with members via SCIM protocol."""
    Group = scim_client_for_trusted_client.get_resource_model("Group")
    User = scim_client_for_trusted_client.get_resource_model("User")

    req = SearchRequest(filter=f'externalId eq "{user.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_user = response.resources[0] if response.resources else None

    response = scim_client_for_trusted_client.query(Group)
    assert not response.resources

    group = models.Group(
        members=[user],
        display_name="foobar",
    )
    backend.save(group)

    response = scim_client_for_trusted_client.query(Group)
    assert len(response.resources) == 1
    retrieved_group = response.resources[0]
    assert retrieved_group.display_name == "foobar"
    assert retrieved_group.members[0].value == distant_scim_user.id

    backend.delete(group)
    response = scim_client_for_trusted_client.query(Group)
    assert not response.resources


def test_scim_client_group_save_unable_to_retrieve_member_via_scim(
    scim_client_for_trusted_client, backend, user, caplog
):
    """Test that a warning is logged when a group member cannot be retrieved via SCIM."""
    Group = scim_client_for_trusted_client.get_resource_model("Group")
    User = scim_client_for_trusted_client.get_resource_model("User")

    req = SearchRequest(filter=f'externalId eq "{user.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_user = response.resources[0] if response.resources else None

    response = scim_client_for_trusted_client.query(Group)
    assert not response.resources

    group = models.Group(
        members=[user],
        display_name="foobar",
    )
    backend.save(group)

    scim_client_for_trusted_client.delete(User, distant_scim_user.id)

    backend.save(group)

    assert (
        "canaille",
        logging.WARNING,
        "Unable to find user user from group foobar via SCIM",
    ) in caplog.record_tuples

    backend.delete(group)


def test_scim_client_change_user_groups_also_updates_group_members(
    testclient,
    scim_client_for_trusted_client,
    backend,
    user,
    logged_admin,
    bar_group,
):
    """Test that changing a user's group membership automatically updates the group's member list via SCIM."""
    Group = scim_client_for_trusted_client.get_resource_model("Group")
    User = scim_client_for_trusted_client.get_resource_model("User")

    req = SearchRequest(filter=f'externalId eq "{user.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_user = response.resources[0] if response.resources else None

    req = SearchRequest(filter=f'externalId eq "{logged_admin.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_admin = response.resources[0] if response.resources else None

    user.groups = user.groups + [bar_group]
    backend.save(user)

    response = scim_client_for_trusted_client.query(Group)
    assert len(response.resources) == 1
    retrieved_bar_group = response.resources[0]
    assert retrieved_bar_group.display_name == "bar"
    assert len(retrieved_bar_group.members) == 2
    assert retrieved_bar_group.members[0].value == distant_scim_admin.id
    assert retrieved_bar_group.members[1].value == distant_scim_user.id

    user.groups = []
    backend.save(user)

    response = scim_client_for_trusted_client.query(Group)
    assert len(response.resources) == 1
    retrieved_bar_group = response.resources[0]
    assert retrieved_bar_group.display_name == "bar"
    assert len(retrieved_bar_group.members) == 1
    assert retrieved_bar_group.members[0].value == distant_scim_admin.id


def test_scim_client_user_creation_and_deletion_also_updates_their_groups(
    testclient,
    scim_client_for_trusted_client,
    backend,
    foo_group,
    bar_group,
    user,
    admin,
    logged_admin,
):
    """Test that creating or deleting a user automatically updates their group memberships via SCIM."""
    testclient.app.config["CANAILLE"]["ENABLE_REGISTRATION"] = True
    testclient.app.config["CANAILLE"]["EMAIL_CONFIRMATION"] = False

    Group = scim_client_for_trusted_client.get_resource_model("Group")
    User = scim_client_for_trusted_client.get_resource_model("User")

    req = SearchRequest(filter=f'externalId eq "{user.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_user = response.resources[0] if response.resources else None

    req = SearchRequest(filter=f'externalId eq "{admin.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_admin = response.resources[0] if response.resources else None

    response = scim_client_for_trusted_client.query(Group)
    assert len(response.resources) == 2
    retrieved_foo_group = response.resources[0]
    assert retrieved_foo_group.display_name == "foo"
    assert len(retrieved_foo_group.members) == 1
    assert retrieved_foo_group.members[0].value == distant_scim_user.id
    retrieved_bar_group = response.resources[1]
    assert retrieved_bar_group.display_name == "bar"
    assert len(retrieved_bar_group.members) == 1
    assert retrieved_bar_group.members[0].value == distant_scim_admin.id

    alice = models.User(
        formatted_name="Alice Alice",
        family_name="Alice",
        user_name="alice",
        emails=["john@doe.test", "johhny@doe.test"],
    )
    alice.groups = [bar_group, foo_group]
    backend.save(alice)

    req = SearchRequest(filter=f'externalId eq "{alice.id}"')
    response = scim_client_for_trusted_client.query(User, search_request=req)
    distant_scim_alice = response.resources[0] if response.resources else None

    response = scim_client_for_trusted_client.query(Group)
    assert len(response.resources) == 2
    retrieved_foo_group = response.resources[0]
    assert retrieved_foo_group.display_name == "foo"
    assert len(retrieved_foo_group.members) == 2
    assert retrieved_foo_group.members[0].value == distant_scim_user.id
    assert retrieved_foo_group.members[1].value == distant_scim_alice.id
    retrieved_bar_group = response.resources[1]
    assert retrieved_bar_group.display_name == "bar"
    assert len(retrieved_bar_group.members) == 2
    assert retrieved_bar_group.members[0].value == distant_scim_admin.id
    assert retrieved_bar_group.members[1].value == distant_scim_alice.id

    backend.delete(alice)

    response = scim_client_for_trusted_client.query(Group)
    assert len(response.resources) == 2
    retrieved_foo_group = response.resources[0]
    assert retrieved_foo_group.display_name == "foo"
    assert len(retrieved_foo_group.members) == 1
    assert retrieved_foo_group.members[0].value == distant_scim_user.id
    retrieved_bar_group = response.resources[1]
    assert retrieved_bar_group.display_name == "bar"
    assert len(retrieved_bar_group.members) == 1
    assert retrieved_bar_group.members[0].value == distant_scim_admin.id


def test_save_user_when_client_doesnt_support_scim(backend, user, consent, caplog):
    """Test that saving a user logs an info message when the client doesn't support SCIM."""
    backend.save(user)
    assert (
        "canaille",
        logging.INFO,
        "SCIM protocol not supported by client Client",
    ) in caplog.record_tuples


def test_save_group_when_client_doesnt_support_scim(
    backend, bar_group, consent, caplog, user
):
    """Test that saving a group logs an info message when the client doesn't support SCIM."""
    bar_group.members = [user]
    backend.save(bar_group)
    assert (
        "canaille",
        logging.INFO,
        "SCIM protocol not supported by client Client",
    ) in caplog.record_tuples


@mock.patch("scim2_client.engines.httpx.SyncSCIMClient.create")
def test_failed_scim_user_creation(
    scim_mock,
    testclient,
    scim_client_for_trusted_client,
    backend,
    caplog,
):
    """Test that a warning is logged when SCIM user creation fails."""
    scim_mock.side_effect = mock.Mock(side_effect=Exception())

    alice = models.User(
        formatted_name="Alice Alice",
        family_name="Alice",
        user_name="alice",
        emails=["john@doe.test", "johhny@doe.test"],
    )
    backend.save(alice)

    assert (
        "canaille",
        logging.WARNING,
        "SCIM User alice creation for client Some client failed",
    ) in caplog.record_tuples

    backend.delete(alice)


@mock.patch("scim2_client.engines.httpx.SyncSCIMClient.replace")
def test_failed_scim_user_update(
    scim_mock,
    testclient,
    scim_client_for_trusted_client,
    backend,
    caplog,
    user,
):
    """Test that a warning is logged when SCIM user update fails."""
    scim_mock.side_effect = mock.Mock(side_effect=Exception())

    backend.save(user)

    assert (
        "canaille",
        logging.WARNING,
        "SCIM User user update for client Some client failed",
    ) in caplog.record_tuples


@mock.patch("scim2_client.engines.httpx.SyncSCIMClient.delete")
def test_failed_scim_user_delete(
    scim_mock,
    testclient,
    scim_client_for_trusted_client,
    backend,
    caplog,
    user,
):
    """Test that a warning is logged when SCIM user deletion fails."""
    scim_mock.side_effect = mock.Mock(side_effect=Exception())

    backend.delete(user)

    assert (
        "canaille",
        logging.WARNING,
        "SCIM User user delete for client Some client failed",
    ) in caplog.record_tuples


@mock.patch("scim2_client.engines.httpx.SyncSCIMClient.create")
def test_failed_scim_group_creation(
    scim_mock,
    testclient,
    scim_client_for_trusted_client,
    backend,
    caplog,
    user,
):
    """Test that a warning is logged when SCIM group creation fails."""
    scim_mock.side_effect = mock.Mock(side_effect=Exception())

    group = models.Group(
        members=[user],
        display_name="foobar",
    )
    backend.save(group)

    assert (
        "canaille",
        logging.WARNING,
        "SCIM Group foobar creation for client Some client failed",
    ) in caplog.record_tuples

    backend.delete(group)


@mock.patch("scim2_client.engines.httpx.SyncSCIMClient.replace")
def test_failed_scim_group_update(
    scim_mock,
    testclient,
    scim_client_for_trusted_client,
    backend,
    caplog,
    bar_group,
):
    """Test that a warning is logged when SCIM group update fails."""
    scim_mock.side_effect = mock.Mock(side_effect=Exception())

    backend.save(bar_group)

    assert (
        "canaille",
        logging.WARNING,
        "SCIM Group bar update for client Some client failed",
    ) in caplog.record_tuples


@mock.patch("scim2_client.engines.httpx.SyncSCIMClient.delete")
def test_failed_scim_group_delete(
    scim_mock,
    testclient,
    scim_client_for_trusted_client,
    backend,
    caplog,
    user,
):
    """Test that a warning is logged when SCIM group deletion fails."""
    scim_mock.side_effect = mock.Mock(side_effect=Exception())

    group = models.Group(
        members=[user],
        display_name="foobar",
    )
    backend.save(group)

    backend.delete(group)

    assert (
        "canaille",
        logging.WARNING,
        "SCIM Group foobar delete for client Some client failed",
    ) in caplog.record_tuples


def test_user_from_canaille_to_scim_client_without_enterprise_user_extension(
    scim_client_for_trusted_client, user
):
    """Test that converting a Canaille user to SCIM format works without the EnterpriseUser extension."""
    User = scim_client_for_trusted_client.get_resource_model("User")

    scim_user = user_from_canaille_to_scim_client(user, User, None)
    assert isinstance(scim_user, User)
    assert not isinstance(scim_user, SCIMUser[EnterpriseUser])
    assert EnterpriseUser not in scim_user


@pytest.mark.skip("Primary is not supported at the moment")
def test_user_from_scim_to_canaille_sorts_primary_emails_first(backend):
    """Test that primary emails are sorted first when converting from SCIM to Canaille user."""
    user = models.User()

    scim_user = SCIMUser[EnterpriseUser](
        emails=[
            SCIMUser.Emails(value="secondary@example.com", primary=False),
            SCIMUser.Emails(value="primary@example.com", primary=True),
            SCIMUser.Emails(value="another@example.com", primary=False),
        ]
    )

    user_from_scim_to_canaille(scim_user, user)

    assert user.emails == [
        "primary@example.com",
        "secondary@example.com",
        "another@example.com",
    ]


@pytest.mark.skip("Primary is not supported at the moment")
def test_user_from_scim_to_canaille_sorts_primary_phones_first(backend):
    """Test that primary phone numbers are sorted first when converting from SCIM to Canaille user."""
    user = models.User()

    scim_user = SCIMUser[EnterpriseUser](
        phone_numbers=[
            SCIMUser.PhoneNumbers(value="+1234567890", primary=False),
            SCIMUser.PhoneNumbers(value="+9876543210", primary=True),
            SCIMUser.PhoneNumbers(value="+5555555555", primary=False),
        ]
    )

    user_from_scim_to_canaille(scim_user, user)

    assert user.phone_numbers == ["+9876543210", "+1234567890", "+5555555555"]


@pytest.mark.skip("Primary is not supported at the moment")
def test_user_from_scim_to_canaille_handles_no_primaries():
    """Test that emails without primary flags are preserved in order when converting from SCIM to Canaille user."""
    user = models.User()

    scim_user = SCIMUser[EnterpriseUser](
        emails=[
            SCIMUser.Emails(value="first@example.com", primary=False),
            SCIMUser.Emails(value="second@example.com", primary=False),
        ]
    )

    user_from_scim_to_canaille(scim_user, user)

    assert user.emails == ["first@example.com", "second@example.com"]
