"""Data models for the prediction system."""
from __future__ import annotations

from dataclasses import dataclass, field
from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple

import numpy as np


@dataclass
class CommitSignal:
    """Signal from a Git commit."""
    
    sha: str
    message: str
    author: str
    timestamp: datetime
    files_changed: List[str]
    lines_added: int
    lines_deleted: int
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class PRSignal:
    """Signal from a pull request."""
    
    number: int
    title: str
    description: str
    author: str
    created_at: datetime
    merged_at: Optional[datetime]
    labels: List[str]
    review_count: int
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class IssueSignal:
    """Signal from an issue discussion."""
    
    number: int
    title: str
    body: str
    author: str
    created_at: datetime
    closed_at: Optional[datetime]
    labels: List[str]
    comments: List[str]
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class DependencyDiff:
    """Difference in dependencies between versions."""
    
    added: Dict[str, str]
    removed: Dict[str, str]
    updated: Dict[str, Tuple[str, str]]  # package: (old_version, new_version)


@dataclass
class ReleaseSignal:
    """Signal from a package release."""
    
    version: str
    published_at: datetime
    dependencies: Dict[str, str]
    dependency_diff: Optional[DependencyDiff] = None
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class FeatureVector:
    """Feature vector combining structural and semantic features."""
    
    id: str
    package: str
    time_window: Tuple[datetime, datetime]
    structural: Dict[str, float]
    semantic: np.ndarray
    combined: np.ndarray
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class ClusterMetadata:
    """Metadata about a CVE cluster."""
    
    cluster_id: int
    centroid: np.ndarray
    size: int
    dominant_cwe: List[str]
    avg_cvss: float
    avg_epss: float
    example_cves: List[str]


@dataclass
class ThreatScore:
    """Threat score for a package."""
    
    package: str
    score: float
    confidence: float
    nearest_clusters: List[Tuple[int, float]]  # (cluster_id, distance)
    similar_cves: List[Tuple[str, float]]  # (cve_id, similarity)
    risk_level: str  # LOW/MEDIUM/HIGH/CRITICAL
    predicted_at: datetime
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class ThreatScenario:
    """Threat scenario generated by LLM agent."""
    
    prediction_id: str
    attack_vectors: List[str]
    affected_components: List[str]
    potential_impact: str
    likelihood: float
    reasoning: str
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class Recommendations:
    """Recommendations for threat mitigation."""
    
    scenario_id: str
    immediate_actions: List[str]
    monitoring_strategy: str
    mitigation_options: List[str]
    alternative_packages: List[str]
    timeline: str
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class ValidationResult:
    """Result of prediction validation."""
    
    prediction_id: str
    outcome: str  # TP/FP/TN/FN
    accuracy: float
    time_to_disclosure: Optional[int] = None  # days until CVE disclosed
    actual_cve: Optional[str] = None
    validation_date: datetime = field(default_factory=datetime.utcnow)


@dataclass
class PerformanceMetrics:
    """Performance metrics for the prediction model."""
    
    precision: float
    recall: float
    f1_score: float
    accuracy: float
    confusion_matrix: List[List[int]]  # 2x2 matrix
    total_predictions: int
    timestamp: datetime = field(default_factory=datetime.utcnow)
