import asyncio
from datetime import datetime, timedelta, timezone
import re

from hishel import AsyncCacheClient
from pydantic import BaseModel

from uv_secure.package_info.dependency_file_parser import Dependency


class Downloads(BaseModel):
    last_day: int | None = None
    last_month: int | None = None
    last_week: int | None = None


class Info(BaseModel):
    author: str | None = None
    author_email: str | None = None
    bugtrack_url: str | None = None
    classifiers: list[str]
    description: str
    description_content_type: str | None = None
    docs_url: str | None = None
    download_url: str | None = None
    downloads: Downloads
    dynamic: list[str] | str | None = None
    home_page: str | None = None
    keywords: str | list[str] | None = None
    license: str | None = None
    license_expression: str | None = None
    license_files: list[str] | None = None
    maintainer: str | None = None
    maintainer_email: str | None = None
    name: str
    package_url: str | None = None
    platform: str | None = None
    project_url: str | None = None
    project_urls: dict[str, str] | None = None
    provides_extra: list[str] | None = None
    release_url: str
    requires_dist: list[str] | None = None
    requires_python: str | None = None
    summary: str | None = None
    version: str
    yanked: bool
    yanked_reason: str | None = None


class Digests(BaseModel):
    blake2b_256: str
    md5: str
    sha256: str


class Url(BaseModel):
    comment_text: str | None = None
    digests: Digests
    downloads: int
    filename: str
    has_sig: bool
    md5_digest: str
    packagetype: str
    python_version: str
    requires_python: str | None = None
    size: int
    upload_time: datetime
    upload_time_iso_8601: datetime
    url: str
    yanked: bool
    yanked_reason: str | None = None


class Vulnerability(BaseModel):
    id: str
    details: str
    fixed_in: list[str] | None = None
    aliases: list[str] | None = None
    link: str | None = None
    source: str | None = None
    summary: str | None = None
    withdrawn: str | None = None


class PackageInfo(BaseModel):
    info: Info
    last_serial: int
    urls: list[Url]
    vulnerabilities: list[Vulnerability]
    direct_dependency: bool = False

    @property
    def age(self) -> timedelta | None:
        """Return age of the package"""
        release_date = min(
            (url.upload_time_iso_8601 for url in self.urls), default=None
        )
        if release_date is None:
            return None
        return datetime.now(tz=timezone.utc) - release_date


def _canonicalize_name(name: str) -> str:
    """Converts a package name to its canonical form for PyPI URLs"""
    return re.sub(r"[_.]+", "-", name).lower()


async def _download_package(
    http_client: AsyncCacheClient, dependency: Dependency, disable_cache: bool
) -> PackageInfo:
    """Queries the PyPi JSON API for vulnerabilities of a given dependency."""
    canonical_name = _canonicalize_name(dependency.name)
    url = f"https://pypi.org/pypi/{canonical_name}/{dependency.version}/json"
    response = await http_client.get(
        url, extensions={"cache_disabled": True} if disable_cache else None
    )
    response.raise_for_status()
    return PackageInfo(**response.json(), direct_dependency=dependency.direct)


async def download_packages(
    dependencies: list[Dependency], http_client: AsyncCacheClient, disable_cache: bool
) -> list[PackageInfo | BaseException]:
    """Fetch vulnerabilities for all dependencies concurrently."""
    tasks = [_download_package(http_client, dep, disable_cache) for dep in dependencies]
    return await asyncio.gather(*tasks, return_exceptions=True)
