# Hapax Observability with OpenLIT

Hapax integrates with OpenLIT to provide comprehensive observability for your LLM applications. This integration gives you:

1. **Auto-Instrumentation**:
   - Automatic tracing of LLM operations
   - Built-in support for LangChain operations
   - Token usage tracking
   - Cost monitoring
   - Latency measurements

2. **Evaluation Features**:
   - Bias detection
   - Toxicity detection
   - Hallucination detection
   - Custom evaluation metrics

3. **Real-Time Monitoring**:
   - Pre-built dashboards
   - Performance metrics
   - Error tracking
   - Cost analytics

## Quick Start

1. Install dependencies:
```bash
pip install openlit
```

2. Initialize OpenLIT in your code:
```python
import openlit
import os

# Initialize OpenLIT
openlit.init(
    environment="development",
    application_name="my-hapax-app",
    otlp_endpoint="http://localhost:4318"
)

# Your existing Hapax code
from hapax_py import llm_function

@llm_function(
    name="summarize",
    model="gpt-3.5-turbo",
    provider="openai"
)
def summarize_text(text: str) -> str:
    # Your function implementation
    pass
```

3. Start the OpenLIT server:
```bash
docker compose up -d
```

4. View your traces and metrics at `http://localhost:3000`

## Features

### 1. LangChain Integration

OpenLIT automatically instruments LangChain operations including:
- Document loading
- Text splitting
- LLM calls
- Chain execution
- Agent operations

### 2. Evaluation

```python
import openlit

# Initialize evaluators
bias_detector = openlit.evals.Bias(provider="openai")
toxicity_detector = openlit.evals.Toxicity(provider="openai")
hallucination_detector = openlit.evals.Hallucination(provider="openai")

# Evaluate responses
result = bias_detector.measure(
    prompt="Your prompt",
    text="Response to evaluate"
)

print(result)  # Shows score, classification, explanation, and verdict
```

### 3. Real-Time Monitoring

The OpenLIT dashboard provides:
- Request volumes
- Latency distributions
- Error rates
- Token usage
- Cost tracking
- Custom metrics

## Best Practices

1. **Environment Setup**:
   - Use different environments (development, staging, production)
   - Set appropriate sampling rates
   - Configure retention policies

2. **Metrics Collection**:
   - Enable token counting for cost tracking
   - Add custom attributes for better filtering
   - Use semantic conventions for consistency

3. **Error Handling**:
   - Add error context to spans
   - Set appropriate status codes
   - Include error messages and stack traces

4. **Performance Optimization**:
   - Use async operations where possible
   - Configure appropriate batch sizes
   - Set reasonable timeouts

## Configuration Options

```python
openlit.init(
    environment="production",
    application_name="my-app",
    otlp_endpoint="http://localhost:4318",
    trace_content=True,  # Include prompt/response content in traces
    disable_metrics=False,  # Enable metrics collection
    collect_gpu_stats=True,  # Track GPU usage if available
)
```

## Dashboard Access

1. Default credentials:
   - URL: http://localhost:3000
   - Email: user@openlit.io
   - Password: openlituser

2. Available views:
   - Request Overview
   - Latency Analysis
   - Error Tracking
   - Cost Analytics
   - Custom Metrics
   - Evaluation Results

## Troubleshooting

1. **No traces visible**:
   - Check OpenLIT server is running
   - Verify OTLP endpoint configuration
   - Ensure correct port forwarding

2. **Missing metrics**:
   - Check metrics collection is enabled
   - Verify proper initialization
   - Check sampling configuration

3. **Performance issues**:
   - Adjust batch size settings
   - Check network connectivity
   - Monitor resource usage
