"""
Example showing automatic server management with OpenLIT observability
"""

import os
import time
from openai import OpenAI
from hapax_py import llm_function
import openlit

# Initialize OpenAI client
openai_client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))

# Initialize OpenLIT
openlit.init(
    environment="development",
    application_name="auto-server",
    otlp_endpoint="http://localhost:4318",
    trace_content=True
)

@llm_function(
    name="classify_task",
    model="gpt-3.5-turbo",
    provider="openai"
)
def classify_task(task: str) -> dict:
    """Classify a task by type and complexity"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": "You are a task classification system. Analyze tasks and return:"
                "\n- Type: coding, writing, analysis, or other"
                "\n- Complexity: simple, medium, or complex"
                "\n- Estimated time: in minutes"
                "\nRespond in a simple format like: type|complexity|minutes"
            },
            {
                "role": "user",
                "content": f"Classify this task: {task}"
            }
        ]
    )
    result = response.choices[0].message.content.split("|")
    return {
        "type": result[0],
        "complexity": result[1],
        "estimated_minutes": int(result[2])
    }

@llm_function(
    name="generate_steps",
    model="gpt-3.5-turbo",
    provider="openai"
)
def generate_steps(task: str, task_type: str) -> list:
    """Generate steps to complete a task"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": f"You are an expert at breaking down {task_type} tasks into steps."
                "\nProvide 3-5 clear, actionable steps."
                "\nRespond with one step per line."
            },
            {
                "role": "user",
                "content": f"Break down this task into steps: {task}"
            }
        ]
    )
    return response.choices[0].message.content.split("\n")

@llm_function(
    name="estimate_resources",
    model="gpt-3.5-turbo",
    provider="openai"
)
def estimate_resources(task: str, complexity: str) -> dict:
    """Estimate required computing resources for a task"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": "You are a resource estimation system. For each task, estimate:"
                "\n- CPU cores needed"
                "\n- Memory (GB)"
                "\n- Storage (GB)"
                "\nRespond in format: cores|memory|storage"
            },
            {
                "role": "user",
                "content": f"Estimate resources for this {complexity} task: {task}"
            }
        ]
    )
    result = response.choices[0].message.content.split("|")
    return {
        "cpu_cores": int(result[0]),
        "memory_gb": int(result[1]),
        "storage_gb": int(result[2])
    }

def process_task(task: str):
    """Process a task with full observability"""
    with openlit.trace(operation_name="task_processing") as span:
        span.set_attribute("task", task)
        
        # Step 1: Classify task
        with openlit.trace(operation_name="task_classification") as classify_span:
            print("\nClassifying task...")
            classification = classify_task(task)
            print(f"Type: {classification['type']}")
            print(f"Complexity: {classification['complexity']}")
            print(f"Estimated time: {classification['estimated_minutes']} minutes")
            classify_span.set_attribute("classification", str(classification))
        
        # Step 2: Generate steps
        with openlit.trace(operation_name="step_generation") as steps_span:
            print("\nGenerating steps...")
            steps = generate_steps(task, classification["type"])
            print("\nSteps:")
            for i, step in enumerate(steps, 1):
                print(f"{i}. {step}")
            steps_span.set_attribute("num_steps", len(steps))
            steps_span.set_attribute("steps", str(steps))
        
        # Step 3: Estimate resources
        with openlit.trace(operation_name="resource_estimation") as resource_span:
            print("\nEstimating resources...")
            resources = estimate_resources(task, classification["complexity"])
            print(f"CPU Cores: {resources['cpu_cores']}")
            print(f"Memory: {resources['memory_gb']} GB")
            print(f"Storage: {resources['storage_gb']} GB")
            resource_span.set_attribute("resources", str(resources))
        
        # Evaluate the entire process
        evaluator = openlit.evals.All(provider="openai")
        eval_result = evaluator.measure(
            prompt=f"Process this task: {task}",
            text=str({
                "classification": classification,
                "steps": steps,
                "resources": resources
            })
        )
        
        # Add final metrics to span
        span.set_attribute("evaluation", str(eval_result))
        span.set_attribute("total_steps", len(steps))
        span.set_attribute("estimated_duration", classification["estimated_minutes"])
        span.set_attribute("total_resources", sum(resources.values()))

def main():
    tasks = [
        "Build a simple web scraper in Python",
        "Write a blog post about machine learning",
        "Analyze customer feedback data",
        "Create a presentation about blockchain"
    ]
    
    # Process each task with tracing
    with openlit.trace(operation_name="batch_processing") as span:
        span.set_attribute("num_tasks", len(tasks))
        
        for i, task in enumerate(tasks, 1):
            print(f"\nProcessing Task {i}: {task}")
            print("=" * 50)
            process_task(task)
            time.sleep(1)  # Small delay between tasks
    
    print("\nView traces and evaluations at: http://localhost:3000")

if __name__ == "__main__":
    main()
