"""Example showing chained LLM functions with OpenLIT observability"""

import os
from openai import OpenAI
from hapax_py import llm_function
import openlit

# Initialize OpenAI client
openai_client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))

# Initialize OpenLIT
openlit.init(
    environment="development",
    application_name="chain-functions",
    otlp_endpoint="http://localhost:4318",
    trace_content=True
)

@llm_function(
    name="generate_outline",
    model="gpt-3.5-turbo",
    provider="openai"
)
def generate_outline(topic: str) -> list:
    """Generate an outline for an article about the given topic"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": "You are an expert at creating article outlines. Generate a brief outline with 3-4 main points."
            },
            {
                "role": "user",
                "content": f"Create an outline for an article about: {topic}"
            }
        ]
    )
    return response.choices[0].message.content.split("\n")

@llm_function(
    name="expand_point",
    model="gpt-3.5-turbo",
    provider="openai"
)
def expand_point(topic: str, point: str) -> str:
    """Expand a single outline point into a paragraph"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": "You are an expert at expanding outline points into well-written paragraphs."
            },
            {
                "role": "user",
                "content": f"Write a paragraph expanding this point about {topic}: {point}"
            }
        ]
    )
    return response.choices[0].message.content

@llm_function(
    name="add_conclusion",
    model="gpt-3.5-turbo",
    provider="openai"
)
def add_conclusion(topic: str, paragraphs: list) -> str:
    """Generate a conclusion based on the expanded paragraphs"""
    combined = "\n\n".join(paragraphs)
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": "You are an expert at writing conclusions that effectively summarize key points."
            },
            {
                "role": "user",
                "content": f"Write a conclusion for this article about {topic}. Here are the paragraphs:\n\n{combined}"
            }
        ]
    )
    return response.choices[0].message.content

def main():
    topic = "The Future of Artificial Intelligence"
    
    # Generate article with tracing
    with openlit.trace(operation_name="article_generation") as span:
        span.set_attribute("topic", topic)
        
        print(f"\nGenerating article about: {topic}")
        print("=" * 50)
        
        # Step 1: Generate outline
        with openlit.trace(operation_name="outline_generation") as outline_span:
            print("\nGenerating outline...")
            outline = generate_outline(topic)
            print("\nOutline:")
            for point in outline:
                print(f"- {point}")
            outline_span.set_attribute("outline_points", len(outline))
        
        # Step 2: Expand each point
        paragraphs = []
        with openlit.trace(operation_name="point_expansion") as expand_span:
            for i, point in enumerate(outline, 1):
                with openlit.trace(operation_name=f"expand_point_{i}") as point_span:
                    print(f"\nExpanding point {i}...")
                    paragraph = expand_point(topic, point)
                    paragraphs.append(paragraph)
                    print(paragraph)
                    point_span.set_attribute("point", point)
                    point_span.set_attribute("paragraph_length", len(paragraph))
            expand_span.set_attribute("total_paragraphs", len(paragraphs))
        
        # Step 3: Add conclusion
        with openlit.trace(operation_name="conclusion_generation") as conclusion_span:
            print("\nGenerating conclusion...")
            conclusion = add_conclusion(topic, paragraphs)
            print("\nConclusion:")
            print(conclusion)
            conclusion_span.set_attribute("conclusion_length", len(conclusion))
        
        # Evaluate the entire article
        full_article = "\n\n".join(paragraphs + [conclusion])
        evaluator = openlit.evals.All(provider="openai")
        eval_result = evaluator.measure(
            prompt=f"Write an article about {topic}",
            text=full_article
        )
        
        # Add evaluation results to span
        span.set_attribute("evaluation", str(eval_result))
        span.set_attribute("article_length", len(full_article))
    
    print("\nView traces and evaluations at: http://localhost:3000")

if __name__ == "__main__":
    main()
