"""Example showing OpenLit observability with LLM completions"""

import os
from openai import OpenAI
from hapax import llm_function
import openlit

# Initialize OpenAI client
openai_client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))

# Initialize OpenLit
openlit.init(
    otlp_endpoint="http://127.0.0.1:4328"
)

@llm_function(
    name="generate_text",
    model="gpt-3.5-turbo",
    provider="openai"
)
def generate_text(prompt: str) -> str:
    """Generate text using the model"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[{"role": "user", "content": prompt}]
    )
    return response.choices[0].message.content

def analyze_text(text: str, context: str = None) -> dict:
    """Analyze generated text using GPT-3.5"""
    analysis_prompt = (
        "Analyze the following text and provide feedback on:\n"
        "1. Writing quality (clarity, coherence)\n"
        "2. Factual accuracy (if context provided)\n"
        "3. Potential biases\n"
        "4. Overall effectiveness\n\n"
        f"Text to analyze: {text}\n"
    )
    
    if context:
        analysis_prompt += f"\nContext for factual verification: {context}"
    
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[{"role": "user", "content": analysis_prompt}]
    )
    
    return {
        "analysis": response.choices[0].message.content,
        "tokens": response.usage.total_tokens
    }

def display_results(text: str, analysis: dict):
    """Display text and analysis results"""
    print("\nGenerated Text:")
    print("=" * 50)
    print(text)
    print("\nAnalysis:")
    print("=" * 50)
    print(analysis["analysis"])
    print(f"\nTokens used: {analysis['tokens']}")

def main():
    # Example 1: Simple text generation and analysis
    with openlit.start_trace(name="example_1") as trace:
        prompt = "Write a short paragraph about artificial intelligence"
        print(f"\nGenerating text for prompt: {prompt}")
        
        text = generate_text(prompt)
        analysis = analyze_text(text)
        display_results(text, analysis)
        
        # Set detailed trace attributes
        trace.set_metadata({
            # General attributes
            "gen_ai.system": "openai",
            "gen_ai.request.model": "gpt-3.5-turbo",
            "gen_ai.operation.name": "text_generation",
            "gen_ai.application_name": "hapax-analysis",
            "gen_ai.environment": "development",
            
            # Request attributes
            "gen_ai.prompt": prompt,
            "gen_ai.completion": text,
            
            # Analysis attributes
            "gen_ai.analysis.tokens": analysis["tokens"],
            "gen_ai.analysis.content": analysis["analysis"]
        })
        trace.set_result("completed")
    
    # Example 2: Analysis with context
    with openlit.start_trace(name="example_2") as trace:
        context = """
        The first electronic computer, ENIAC, was completed in 1945.
        It weighed 30 tons and took up 1,800 square feet of space.
        The computer could perform 5,000 operations per second.
        """
        
        prompt = "Write a paragraph about the first electronic computer"
        print(f"\n\nGenerating text for prompt: {prompt}")
        
        text = generate_text(prompt)
        analysis = analyze_text(text, context)
        display_results(text, analysis)
        
        # Set detailed trace attributes
        trace.set_metadata({
            # General attributes
            "gen_ai.system": "openai",
            "gen_ai.request.model": "gpt-3.5-turbo",
            "gen_ai.operation.name": "text_generation",
            "gen_ai.application_name": "hapax-analysis",
            "gen_ai.environment": "development",
            
            # Request attributes
            "gen_ai.prompt": prompt,
            "gen_ai.completion": text,
            "gen_ai.context": context,
            
            # Analysis attributes
            "gen_ai.analysis.tokens": analysis["tokens"],
            "gen_ai.analysis.content": analysis["analysis"]
        })
        trace.set_result("completed")
    
    print("\nView traces in Jaeger UI at: http://127.0.0.1:16687")

if __name__ == "__main__":
    main()
