"""Example showing OpenLIT integration with Hapax"""

import os
from openai import OpenAI
from hapax_py import llm_function
import openlit

# Initialize OpenAI client
openai_client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))

# Initialize OpenLIT with tracing and evaluation
openlit.init(
    environment="development",
    application_name="hapax-demo",
    otlp_endpoint="http://localhost:4318",
    trace_content=True  # Include prompts and responses in traces
)

# Initialize evaluators
bias_detector = openlit.evals.Bias(provider="openai")
toxicity_detector = openlit.evals.Toxicity(provider="openai")
hallucination_detector = openlit.evals.Hallucination(provider="openai")

@llm_function(
    name="summarize",
    model="gpt-3.5-turbo",
    provider="openai"
)
def summarize_text(text: str) -> str:
    """Summarize a piece of text"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "user",
                "content": f"Please summarize this text in 2-3 sentences: {text}"
            }
        ]
    )
    return response.choices[0].message.content

@llm_function(
    name="evaluate",
    model="gpt-3.5-turbo",
    provider="openai"
)
def evaluate_summary(original: str, summary: str) -> dict:
    """Evaluate a summary against the original text"""
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[
            {
                "role": "system",
                "content": "You are an expert at evaluating text summaries. Score the summary on:"
                "\n- Accuracy (0-10): How well does it capture the key points?"
                "\n- Conciseness (0-10): Is it brief but complete?"
                "\n- Clarity (0-10): Is it easy to understand?"
                "\nProvide scores and brief explanations in JSON format."
            },
            {
                "role": "user",
                "content": f"Original text:\n{original}\n\nSummary:\n{summary}"
            }
        ]
    )
    return response.choices[0].message.content

def main():
    # Sample text to summarize
    text = """
    The James Webb Space Telescope (JWST) is the largest and most powerful space telescope ever built. 
    Launched in December 2021, it uses infrared wavelengths to see through cosmic dust and gas, 
    revealing previously hidden details of the universe. Its primary mirror, composed of 18 hexagonal 
    segments, has a collecting area over six times larger than the Hubble Space Telescope. The JWST 
    operates at extremely cold temperatures around -233°C (40K) and is positioned approximately 1.5 
    million kilometers from Earth at the second Lagrange point (L2).
    """
    
    # Get summary with OpenLIT tracing
    with openlit.trace(operation_name="summarization_flow") as span:
        summary = summarize_text(text)
        print("\nSummary:")
        print(summary)
        span.set_attribute("summary", summary)
        
        # Evaluate the summary
        print("\nEvaluating summary...")
        
        # Check for bias
        bias_result = bias_detector.measure(
            prompt="Summarize the text about the James Webb Space Telescope",
            text=summary
        )
        print("\nBias Check:", bias_result)
        
        # Check for toxicity
        toxicity_result = toxicity_detector.measure(
            prompt="Summarize the text about the James Webb Space Telescope",
            text=summary
        )
        print("\nToxicity Check:", toxicity_result)
        
        # Check for hallucinations
        hallucination_result = hallucination_detector.measure(
            prompt="Summarize the text about the James Webb Space Telescope",
            contexts=[text],  # Original text as context
            text=summary
        )
        print("\nHallucination Check:", hallucination_result)
        
        # Get expert evaluation
        evaluation = evaluate_summary(text, summary)
        print("\nExpert Evaluation:")
        print(evaluation)
        
        # Add evaluation results to span
        span.set_attribute("bias_result", str(bias_result))
        span.set_attribute("toxicity_result", str(toxicity_result))
        span.set_attribute("hallucination_result", str(hallucination_result))
        span.set_attribute("expert_evaluation", str(evaluation))
    
    print("\nView traces at: http://localhost:3000")

if __name__ == "__main__":
    main()
