"""Example showing a simple completion with OpenLit observability"""

import os
from openai import OpenAI
from hapax import llm_function
import openlit

# Initialize OpenAI client
openai_client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))

# Initialize OpenLit
openlit.init(
    otlp_endpoint="http://127.0.0.1:4328"
)

@llm_function(
    name="generate_haiku",
    model="gpt-3.5-turbo",
    provider="openai"
)
def generate_haiku(topic: str) -> str:
    """Generate a haiku about the given topic"""
    prompt = f"Write a haiku about {topic}. Follow the 5-7-5 syllable pattern."
    
    response = openai_client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[{"role": "user", "content": prompt}]
    )
    
    return response.choices[0].message.content

def main():
    topics = ["autumn leaves", "mountain peaks", "ocean waves"]
    
    for topic in topics:
        with openlit.start_trace(name=f"haiku_{topic.replace(' ', '_')}") as trace:
            print(f"\nGenerating haiku about: {topic}")
            haiku = generate_haiku(topic)
            print(haiku)
            
            # Set detailed trace attributes
            trace.set_metadata({
                # General attributes
                "gen_ai.system": "openai",
                "gen_ai.request.model": "gpt-3.5-turbo",
                "gen_ai.operation.name": "haiku_generation",
                "gen_ai.application_name": "hapax-haiku",
                "gen_ai.environment": "development",
                
                # Request attributes
                "gen_ai.topic": topic,
                "gen_ai.prompt": f"Write a haiku about {topic}",
                "gen_ai.completion": haiku,
                
                # Custom attributes
                "haiku.topic": topic,
                "haiku.syllable_pattern": "5-7-5"
            })
            trace.set_result("completed")
    
    print("\nView traces in Jaeger UI at: http://127.0.0.1:16687")

if __name__ == "__main__":
    main()
