"""Command-line interface for Hapax with OpenLIT integration"""

import os
import sys
import time
from typing import Optional
import typer
from rich.console import Console
from rich.panel import Panel
from rich.progress import Progress, SpinnerColumn, TextColumn
import openlit
from openai import OpenAI
from .decorators import llm_function

app = typer.Typer()
console = Console()

# Initialize OpenLIT
openlit.init(
    environment="development",
    application_name="hapax-cli",
    otlp_endpoint="http://localhost:4318",
    trace_content=True
)

# Initialize evaluators
bias_detector = openlit.evals.Bias(provider="openai")
toxicity_detector = openlit.evals.Toxicity(provider="openai")
hallucination_detector = openlit.evals.Hallucination(provider="openai")

@llm_function(
    name="chat",
    model="gpt-3.5-turbo",
    provider="openai"
)
def chat_completion(prompt: str) -> str:
    """Get a chat completion from the model"""
    client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))
    response = client.chat.completions.create(
        model="gpt-3.5-turbo",
        messages=[{"role": "user", "content": prompt}]
    )
    return response.choices[0].message.content

def evaluate_response(prompt: str, response: str):
    """Evaluate the response using OpenLIT evaluators"""
    results = {}
    
    # Check for bias
    results["bias"] = bias_detector.measure(
        prompt=prompt,
        text=response
    )
    
    # Check for toxicity
    results["toxicity"] = toxicity_detector.measure(
        prompt=prompt,
        text=response
    )
    
    # Check for hallucinations
    results["hallucination"] = hallucination_detector.measure(
        prompt=prompt,
        text=response
    )
    
    return results

@app.command()
def chat(
    prompt: str = typer.Argument(..., help="The prompt to send to the model"),
    evaluate: bool = typer.Option(False, "--evaluate", "-e", help="Evaluate the response"),
    model: str = typer.Option("gpt-3.5-turbo", "--model", "-m", help="Model to use"),
    show_metrics: bool = typer.Option(False, "--metrics", help="Show metrics after completion")
):
    """Chat with an LLM model with observability"""
    
    # Check API key
    if not os.getenv("OPENAI_API_KEY"):
        console.print("[red]Error: OPENAI_API_KEY environment variable not set[/red]")
        sys.exit(1)
    
    with openlit.trace(operation_name="cli_chat") as span:
        span.set_attribute("prompt", prompt)
        span.set_attribute("model", model)
        
        # Show prompt
        console.print(Panel(prompt, title="[blue]Prompt[/blue]", border_style="blue"))
        
        # Get response with progress
        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            console=console,
        ) as progress:
            progress.add_task(description="Getting response...", total=None)
            response = chat_completion(prompt)
        
        # Show response
        console.print(Panel(response, title="[green]Response[/green]", border_style="green"))
        span.set_attribute("response", response)
        
        # Evaluate if requested
        if evaluate:
            console.print("\n[yellow]Evaluating response...[/yellow]")
            results = evaluate_response(prompt, response)
            
            # Show evaluation results
            console.print("\n[bold]Evaluation Results:[/bold]")
            
            for eval_type, result in results.items():
                color = "green" if result["verdict"] == "no" else "red"
                console.print(f"\n{eval_type.title()}:")
                console.print(f"  Classification: [{color}]{result['classification']}[/{color}]")
                console.print(f"  Explanation: {result['explanation']}")
                console.print(f"  Score: {result['score']}")
                
                # Add to span
                span.set_attribute(f"eval_{eval_type}_verdict", result["verdict"])
                span.set_attribute(f"eval_{eval_type}_score", result["score"])
        
        if show_metrics:
            console.print("\n[bold]View metrics at:[/bold] http://localhost:3000")

@app.command()
def server(
    action: str = typer.Argument(..., help="Action to perform (start/stop)"),
    port: int = typer.Option(8080, "--port", "-p", help="Port to run server on")
):
    """Manage the Hapax server with observability"""
    
    if action == "start":
        with openlit.trace(operation_name="server_start") as span:
            span.set_attribute("port", port)
            
            console.print(f"Starting Hapax server on port {port}...")
            console.print("View metrics at: http://localhost:3000")
            
            # Server start logic here...
            time.sleep(2)  # Simulated startup
            
            console.print("[green]Server started successfully![/green]")
    
    elif action == "stop":
        with openlit.trace(operation_name="server_stop"):
            console.print("Stopping Hapax server...")
            
            # Server stop logic here...
            time.sleep(1)  # Simulated shutdown
            
            console.print("[green]Server stopped successfully![/green]")
    
    else:
        console.print(f"[red]Unknown action: {action}[/red]")
        sys.exit(1)

if __name__ == "__main__":
    app()
