"""
Client for interacting with Hapax Gateway
"""

import os
from typing import Any, Dict, Optional

import httpx
from pydantic import BaseModel, Field

class HapaxConfig(BaseModel):
    """Configuration for Hapax Gateway"""
    gateway_url: str = Field(
        default_factory=lambda: os.environ.get("HAPAX_GATEWAY_URL", "http://localhost:8080")
    )
    api_key: Optional[str] = Field(
        default_factory=lambda: os.environ.get("HAPAX_API_KEY")
    )

class HapaxClient:
    """Client for deploying and managing LLM functions with Hapax Gateway"""

    def __init__(
        self,
        gateway_url: Optional[str] = None,
        api_key: Optional[str] = None,
    ):
        """
        Initialize Hapax client

        Args:
            gateway_url: URL of Hapax Gateway, defaults to HAPAX_GATEWAY_URL env var
            api_key: API key for authentication, defaults to HAPAX_API_KEY env var
        """
        self.config = HapaxConfig(
            gateway_url=gateway_url,
            api_key=api_key,
        )
        self.client = httpx.Client(
            base_url=self.config.gateway_url,
            headers={"Authorization": f"Bearer {self.config.api_key}"} if self.config.api_key else {},
        )

    def deploy_function(self, func: Any) -> Dict[str, Any]:
        """
        Deploy an LLM function to Hapax Gateway

        Args:
            func: Function decorated with @llm_function

        Returns:
            Deployment information
        """
        if not hasattr(func, "_hapax_metadata"):
            raise ValueError("Function must be decorated with @llm_function")

        # Get function metadata
        metadata = func._hapax_metadata

        # Create deployment configuration
        config = {
            "name": metadata["name"],
            "model": metadata.get("model"),
            "provider": metadata.get("provider"),
            "function": func,  # This will be serialized appropriately by Hapax
            **metadata
        }

        # Deploy to Hapax Gateway
        response = self.client.post("/deployments", json=config)
        response.raise_for_status()
        return response.json()
