"""
Decorators for defining LLM functions that will be deployed behind Hapax Gateway
"""

import functools
from typing import Any, Callable, Optional
import threading
import openlit

# Initialize OpenLit with default settings
openlit.init(otlp_endpoint="http://127.0.0.1:4318")

# Global server instance
_server: Optional['HapaxServer'] = None  # noqa: F821
_server_lock = threading.Lock()

def ensure_server_running():
    """Ensure Hapax server is running"""
    global _server
    
    with _server_lock:
        if _server is None:
            from .server import HapaxServer
            _server = HapaxServer()
            _server.start()

def llm_function(
    name: Optional[str] = None,
    model: Optional[str] = None,
    provider: Optional[str] = None,
    **kwargs: Any,
) -> Callable:
    """
    Decorator to mark a function as an LLM function that will be deployed behind Hapax Gateway.
    Automatically adds OpenLit observability.

    Args:
        name: Optional name for the function endpoint
        model: Optional specific model to use
        provider: Optional specific provider to use
        **kwargs: Additional configuration options

    Returns:
        Decorated function with OpenLit observability
    """
    def decorator(func: Callable) -> Callable:
        @functools.wraps(func)
        def wrapper(*args: Any, **kwargs: Any) -> Any:
            # Ensure Hapax server is running
            ensure_server_running()
            
            # Start tracing span using OpenLit
            with openlit.start_trace(name=name or func.__name__) as trace:
                try:
                    # Set general attributes
                    trace.set_metadata({
                        "gen_ai.system": provider or "unknown",
                        "gen_ai.request.model": model,
                        "gen_ai.operation.name": "llm",
                        "gen_ai.application_name": "hapax",
                        "gen_ai.environment": "production",
                    })

                    # Call function and record metrics
                    result = func(*args, **kwargs)
                    
                    # Add usage metrics if available
                    if hasattr(result, "usage"):
                        trace.set_metadata({
                            "gen_ai.usage.input_tokens": result.usage.prompt_tokens,
                            "gen_ai.usage.output_tokens": result.usage.completion_tokens,
                            "gen_ai.usage.total_tokens": result.usage.total_tokens
                        })
                    
                    # Set the result
                    if result:
                        trace.set_result(str(result))
                    
                    return result
                except Exception as e:
                    # Record error in trace
                    trace.set_metadata({
                        "error.type": e.__class__.__name__,
                        "error.message": str(e)
                    })
                    raise

        # Store Hapax deployment metadata on the function
        wrapper._hapax_metadata = {
            "name": name or func.__name__,
            "model": model,
            "provider": provider,
            **kwargs
        }
        return wrapper
    return decorator
