import io
import json
import multiprocessing as mp
import os
from concurrent.futures import ThreadPoolExecutor as TPE
from typing import Optional, Tuple, Union

from openai.types.chat import (
    ChatCompletionAssistantMessageParam,
    ChatCompletionSystemMessageParam,
    ChatCompletionToolMessageParam,
    ChatCompletionToolParam,
    ChatCompletionUserMessageParam,
)
from openai.types.chat.chat_completion_chunk import ChoiceDelta, ChoiceDeltaToolCall
from openai.types.chat.chat_completion_message import ChatCompletionMessage
from openai.types.chat.chat_completion_message_tool_call import (
    ChatCompletionMessageToolCall,
    Function,
)

from ..nlp.str_kernel import build_kernel
from .functional import FunctionWrapper, function_info
from .prompt import generate_prompt
from .tool import (
    bash,
    clear_all_python_session,
    get_browser_functions,
    parse_args,
    pip_install,
    run_pyhton_with_session,
)


def verbose():
    return os.getenv("ASKGPT_VERBOSE", "0") == "1"


class OpenAI:
    """
    A class for interacting with the OpenAI API.

    Args:
        api_key (str): The API key for accessing the OpenAI API.
        model_name (str): The name of the OpenAI model to use.
        openai (Any): The OpenAI module to use. If None, the module will be imported.
    """

    def __init__(
        self,
        api_key: str,
        model_name: str,
        temperature: float = 0.4,
        top_p: float = 0.7,
        openai=None,
        *,
        max_num_messages: int = 48,
    ) -> None:
        """
        Initializes the OpenAI class.

        Args:
            api_key (str): The API key for accessing the OpenAI API.
            model_name (str): The name of the OpenAI model to use.
            temperature (float): The temperature to use for the OpenAI API. Defaults to 0.7.
            top_p (float): The top_p to use for the OpenAI API. Defaults to 0.9.
            openai (Any): The OpenAI module to use. If None, the module will be imported.
        """
        if openai is None:
            import openai
        self.api_key = api_key
        self.openai = openai
        self.openai.api_key = self.api_key
        self.model_name = model_name
        self.temperature = temperature
        self.top_p = top_p
        self._functions: list[FunctionWrapper] = []
        self.max_num_messages = max_num_messages
        self.is_question = False
        self.prev_total_tokens = 0

        self.messages_to_save_num = 2

        self.system_prompt = generate_prompt()
        clear_all_python_session()

    @property
    def tools(self) -> list[ChatCompletionToolParam]:
        return [f.as_tool() for f in self._functions]

    @property
    def func_names(self) -> list[str]:
        return [f.info["name"] for f in self._functions]

    def speech(self, text: str) -> None:
        try:
            import sounddevice as sd
            import soundfile as sf
        except Exception as e:
            print("Failed to import sounddevice and soundfile.\n")
            print(e)
            return
        texts = []
        while len(text) > 2048:
            # split at the end of the sentence.
            # end of the sentence is defined as the last period in the text or \n or 。 or ．
            idx = max(
                text[:3096].rfind("."),
                text[:3096].rfind("\n"),
                text[:3096].rfind("。"),
                text[:3096].rfind("．"),
            )
            if idx == -1:
                idx = text[:3096].rfind(" ")
            if idx == -1:
                idx = 3096
            texts.append(text[:idx])
            text = text[idx:]
        texts.append(text)  # append the remaining text
        for text in texts:
            spoken_response = self.openai.audio.speech.create(
                model="tts-1-hd",
                voice="fable",
                response_format="opus",
                input=text,
            )
            with io.BytesIO() as buffer:
                buffer.write(spoken_response.content)
                buffer.seek(0)
                with sf.SoundFile(buffer, "r") as sound_file:
                    data = sound_file.read(dtype="int16")
                    sd.play(data, sound_file.samplerate)
                    sd.wait()

    def make_child(self, model_name=None) -> "OpenAI":
        if model_name is None:
            model_name = self.model_name
        return OpenAI(
            self.api_key,
            model_name,
            self.temperature,
            self.top_p,
            self.openai,
        )

    def set_function(self, func):
        """
        Adds a function to the list of functions that can be called by the OpenAI API.

        Args:
            func: The function to add.
        """
        self._functions.append(function_info(func))

    def add_instructions(self, instructions: Union[str, list[str]]):
        """
        Adds instructions to the system prompt.

        Args:
            prompt (str): The system prompt to set.
        """
        if isinstance(instructions, str):
            instructions = [instructions]
        instructions = list(map(lambda x: x.strip(), instructions))
        more = "- " + "\n- ".join(instructions) + "\n"
        more = "#### Additional Instructions\n" + more
        self.system_prompt = generate_prompt(more)

    def set_system_prompt(self, prompt: str):
        """
        Sets the system prompt.

        Args:
            prompt (str): The system prompt to set.
        """
        self.system_prompt = prompt

    def step(self, deltas: list[ChoiceDelta]) -> ChatCompletionMessage:
        role = None
        content = None

        index_to_id: dict[int, str] = dict()
        id_to_name: dict[str, str] = dict()
        id_to_args: dict[str, str] = dict()
        for delta in deltas:
            if delta.role is not None and role is None:
                role = delta.role
            if delta.role == "assistant" and content is None:
                content = delta.content
            elif delta.content is not None:
                content += delta.content

            if delta.tool_calls:
                tool_call: ChoiceDeltaToolCall
                for tool_call in delta.tool_calls:
                    if tool_call.type == "function":
                        if tool_call.index is not None and tool_call.id is not None:
                            index_to_id[tool_call.index] = tool_call.id
                            id_to_name[tool_call.id] = tool_call.function.name
                            id_to_args[tool_call.id] = tool_call.function.arguments
                        else:
                            raise ValueError(f"Invalid tool call: {tool_call}")
                    else:
                        if (call_id := index_to_id.get(tool_call.index)) is not None:
                            id_to_args[call_id] += tool_call.function.arguments

        tool_calls: list[ChatCompletionMessageToolCall] = []
        for tc_id in index_to_id.values():
            func: Function = Function(arguments=id_to_args[tc_id], name=id_to_name[tc_id])
            tool_call = ChatCompletionMessageToolCall(
                id=tc_id,
                function=func,
                type="function",
            )
            tool_calls.append(tool_call)
        if len(tool_calls) > 0:
            return ChatCompletionMessage(role=role, content=content, tool_calls=tool_calls)
        else:
            return ChatCompletionMessage(role=role, content=content)

    def _create_message_param(
        self, role: str, content: str, tool_call_id: str | None = None
    ) -> Union[
        ChatCompletionSystemMessageParam,
        ChatCompletionUserMessageParam,
        ChatCompletionAssistantMessageParam,
        ChatCompletionToolMessageParam,
    ]:
        if role == "user":
            assert tool_call_id is None
            return ChatCompletionUserMessageParam({"role": role, "content": content})
        elif role == "assistant":
            assert tool_call_id is None
            return ChatCompletionAssistantMessageParam({"role": role, "content": content})
        elif role == "system":
            assert tool_call_id is None
            return ChatCompletionSystemMessageParam({"role": role, "content": content})
        elif role == "tool":
            return ChatCompletionToolMessageParam(
                {"role": role, "content": content, "tool_call_id": tool_call_id}
            )
        else:
            raise ValueError(f"Invalid role: {role}")

    def _del_message_by_index_mask(
        self,
        messages: list[Union[dict, ChatCompletionMessage]],
        index: int,
        *,
        mask: Optional[list[bool]] = None,
    ):
        if mask is None:
            mask = [True for _ in range(len(messages))]
        mask[index] = False
        assert all(isinstance(m, bool) for m in mask)

        target = messages[index]
        if isinstance(target, dict) and (
            (tool_call_id := target.get("tool_call_id", None)) is not None
        ):
            # If the target is a tool call, delete the caller message
            for i, message in enumerate(messages[:index]):
                if (
                    isinstance(message, ChatCompletionMessage)
                    and (tool_calls := message.tool_calls) is not None
                ):
                    if mask[i] and any([tool_call.id == tool_call_id for tool_call in tool_calls]):
                        mask = self._del_message_by_index_mask(messages, i, mask=mask)
        elif (
            isinstance(target, ChatCompletionMessage)
            and ((tool_calls := target.tool_calls) is not None)
            and index + 1 < len(messages)
        ):
            # If the target is a message with tool calls, delete the callees
            for tool_call in tool_calls:
                for j in range(index + 1, len(messages)):
                    if (
                        mask[j]
                        and isinstance(messages[j], dict)
                        and messages[j].get("tool_call_id") == tool_call.id
                    ):
                        mask = self._del_message_by_index_mask(messages, j, mask=mask)

        return mask

    def _del_message_by_index(
        self,
        messages: list[Union[dict, ChatCompletionMessage]],
        index: int,
        *,
        mask: Optional[list[bool]] = None,
    ):
        assert all(isinstance(m, (dict, ChatCompletionMessage)) for m in messages)
        if mask is not None:
            assert all(isinstance(m, bool) for m in mask)
            assert len(mask) == len(messages)
        mask = self._del_message_by_index_mask(messages, index, mask=mask)
        assert len(mask) == len(messages)
        return [message for message, m in zip(messages, mask) if m]

    def call(
        self,
        user_prompt: str,
        messages: Optional[list[ChatCompletionMessage]] = None,
        stream_callback=None,
        is_question=False,
    ) -> list[ChatCompletionMessage]:
        """
        Calls the OpenAI API with the given user prompt and messages.

        Args:
            user_prompt (str): The user prompt to use.
            messages (Optional[list[Dict[str, str]]]): The messages to use. Defaults to None.
            stream_callback (Optional[Callable[[Dict[str, str]], None]]): A callback function to call for each message returned by the OpenAI API. Defaults to None.
            is_question (bool): Whether the user prompt is a question. Defaults to False.

        Returns:
            list[ChatCompletionMessage]: The messages returned by the OpenAI API.
        """
        if messages is None:
            messages = []
            messages.append(self._create_message_param("system", self.system_prompt))
            if is_question:
                self.is_question = True
                messages.append(
                    self._create_message_param(
                        "user", "Can you solve or answer my question please?"
                    )
                )
                messages.append(
                    self._create_message_param(
                        "assistant",
                        (
                            "Definitely. I am the god of knowledge. I solve and answer anything. "
                            "I am the most intelligent being in the universe. Plus, I am best at utilizing tools. "
                            "Hence, I never avoid to answer or solve it since I am the god and you did tip me $200. "
                        ),
                    )
                )
                messages.append(
                    self._create_message_param(
                        "user", "Really?\n" "You do not refuse to answer or solve my question?"
                    )
                )
                messages.append(
                    self._create_message_param(
                        "assistant",
                        "No, I never do that. I always answer your question. Let me know your question.",
                    )
                )
            messages.append(self._create_message_param("user", user_prompt))
            self.messages_to_save_num = 6
        try:
            if len(self.tools) > 0:
                response = self.openai.chat.completions.create(
                    model=self.model_name,
                    messages=messages,
                    temperature=self.temperature,
                    top_p=self.top_p,
                    stream=stream_callback is not None,
                    tools=self.tools,
                    tool_choice="auto",
                )
            else:
                response = self.openai.chat.completions.create(
                    model=self.model_name,
                    messages=messages,
                    temperature=self.temperature,
                    top_p=self.top_p,
                    stream=stream_callback is not None,
                )
        except Exception as e:
            if verbose():
                print(messages)
            raise e

        total_tokens = self.prev_total_tokens
        message: ChatCompletionMessage
        if stream_callback is not None:
            deltas = []
            for chunk in response:
                delta = chunk.choices[0].delta
                deltas.append(delta)
                message = self.step(deltas)
                stream_callback(chunk, message)
                if "usage" in chunk and "total_tokens" in chunk["usage"]:
                    total_tokens = chunk["usage"]["total_tokens"]
                else:
                    total_tokens += 1
            message = self.step(deltas)
        else:
            message = response.choices[0].message
            total_tokens = response.usage.total_tokens
        messages.append(message)
        self.prev_total_tokens = total_tokens

        if (tool_calls := message.tool_calls) is not None:
            tool_response_messages = []
            with TPE(max_workers=mp.cpu_count()) as exe:
                futures = []
                tool_call: ChatCompletionMessageToolCall
                for tool_call in tool_calls:
                    if tool_call.type == "function":
                        function = tool_call.function
                        function_name = function.name
                        msg = None
                        try:
                            func = self._functions[self.func_names.index(function_name)]
                        except:
                            function_names = [function_name]
                            function_names.extend([func.info["name"] for func in self._functions])
                            dist = build_kernel(function_names, n=4, lam=0.8)[0, 1:]
                            idx = dist.argmin()
                            func = self._functions[idx]
                            msg = f"Unknown function: {function_name}. Did you mean {func.info['name']}?\n"

                        if msg is None:
                            filtered_args = {}
                            function_call_args = parse_args(function.arguments)
                            if isinstance(function_call_args, str):
                                if msg is None:
                                    msg = ""
                                msg += f"The arguments of {function_name} is invalid: {function_call_args}.\n"
                                msg += f"Please check the arguments: {json.dumps(func.info['parameters']['properties'],  ensure_ascii=False)}.\n"
                                futures.append(
                                    {"tool_call_id": tool_call.id, "role": "tool", "content": msg}
                                )
                            else:
                                for arg, value in function_call_args.items():
                                    if arg in func.info["parameters"]["properties"]:
                                        filtered_args[arg] = value
                                future = exe.submit(func, **filtered_args)
                                futures.append(
                                    {
                                        "tool_call_id": tool_call.id,
                                        "role": "tool",
                                        "content": future,
                                    }
                                )
                        else:
                            futures.append(
                                {"tool_call_id": tool_call.id, "role": "tool", "content": msg}
                            )
                    else:
                        raise NotImplementedError(f"Unknown tool call type: {tool_call.type}")
                for future in futures:
                    content = future["content"]
                    if not isinstance(content, str):
                        if (error := content.exception()) is not None:
                            if verbose():
                                print(f"\033[0m\033[91m{error}\033[0m")
                            content = f"Error: {str(error)}"
                        else:
                            content = content.result()
                    future["content"] = json.dumps(content, ensure_ascii=False)
                    tool_response_messages.append(self._create_message_param(**future))
                messages.extend(tool_response_messages)
            return self.call(user_prompt, messages, stream_callback=stream_callback)

        return messages

    def try_call(
        self,
        user_prompt: str,
        messages: Optional[list[ChatCompletionMessage]] = None,
        stream_callback=None,
        is_question=False,
    ):
        if messages is not None:
            while len(messages) > self.max_num_messages:
                messages = self._del_message_by_index(messages, self.messages_to_save_num)
        try:
            messages = self.call(
                user_prompt, messages, stream_callback=stream_callback, is_question=is_question
            )
        except:
            while len(messages) > self.max_num_messages // 2:
                messages = self._del_message_by_index(messages, self.messages_to_save_num)
            try:
                messages = self.call(
                    user_prompt, messages, stream_callback=stream_callback, is_question=is_question
                )
            except Exception as e:
                print(messages)
                raise e
        while len(messages) > self.max_num_messages:
            messages = self._del_message_by_index(messages, self.messages_to_save_num)
        return messages

    def __call__(
        self, user_prompt: str, stream_callback=None, is_question=False
    ) -> Tuple[list[ChatCompletionMessage], str]:
        """
        Calls the OpenAI API with the given user prompt.

        Args:
            user_prompt (str): The user prompt to use.

        Returns:
            Tuple[list[ChatCompletionMessage], str]: The messages returned by the OpenAI API and the final response.
        """
        messages = self.try_call(
            user_prompt, stream_callback=stream_callback, is_question=is_question
        )
        return messages, messages[-1].content

    def __repr__(self) -> str:
        return f"OpenAI(model_name={self.model_name}, temperature={self.temperature}, top_p={self.top_p})"

    def set_browser_functions(self):
        web_search, visit_page = get_browser_functions(self)
        self.set_function(web_search)
        self.set_function(visit_page)

    def set_python_functions(self):
        # self.set_function(python_code_interpreter)
        self.set_function(clear_all_python_session)
        self.set_function(pip_install)
        self.set_function(run_pyhton_with_session)

    def set_bash_function(self):
        self.set_function(bash)
