import typing as t
from dataclasses import dataclass

import click

from unstructured.ingest.cli.base.src import BaseSrcCmd
from unstructured.ingest.cli.interfaces import (
    CliConfig,
    FileOrJson,
)
from unstructured.ingest.connector.fsspec.gcs import GcsWriteConfig, SimpleGcsConfig

CMD_NAME = "gcs"


@dataclass
class GcsCliConfig(SimpleGcsConfig, CliConfig):
    @staticmethod
    def get_cli_options() -> t.List[click.Option]:
        help_string = """
        Options:
        - ``None``, GCSFS will attempt to guess your credentials in the
      following order: gcloud CLI default, gcsfs cached token, google compute
      metadata service, anonymous.
        - ``'google_default'``, your default gcloud credentials will be used,
          which are typically established by doing ``gcloud login`` in a terminal.
        - ``'cache'``, credentials from previously successful gcsfs
          authentication will be used (use this after "browser" auth succeeded)
        - ``'anon'``, no authentication is performed, and you can only
          access data which is accessible to allUsers (in this case, the project and
          access level parameters are meaningless)
        - ``'browser'``, you get an access code with which you can
          authenticate via a specially provided URL
        - if ``'cloud'``, we assume we are running within google compute
          or google container engine, and query the internal metadata directly for
          a token.
        - you may supply a token generated by the
          [gcloud](https://cloud.google.com/sdk/docs/)
          utility; this is either a python dictionary or the name of a file
          containing the JSON returned by logging in with the gcloud CLI tool.
        """
        options = [
            click.Option(
                ["--service-account-key"],
                default=None,
                type=FileOrJson(allow_raw_str=True),
                help=help_string,
            ),
        ]
        return options


def get_base_src_cmd() -> BaseSrcCmd:
    cmd_cls = BaseSrcCmd(
        cmd_name=CMD_NAME,
        cli_config=GcsCliConfig,
        is_fsspec=True,
    )
    return cmd_cls


def get_base_dest_cmd():
    from unstructured.ingest.cli.base.dest import BaseDestCmd

    cmd_cls = BaseDestCmd(
        cmd_name=CMD_NAME,
        cli_config=GcsCliConfig,
        write_config=GcsWriteConfig,
        is_fsspec=True,
    )
    return cmd_cls
