from dataclasses import dataclass

import click

from unstructured.ingest.v2.cli.base import DestCmd, SrcCmd
from unstructured.ingest.v2.cli.cmds.fsspec.fsspec import (
    FsspecCliDownloadConfig,
    FsspecCliIndexerConfig,
    FsspecCliUploaderConfig,
)
from unstructured.ingest.v2.cli.interfaces import CliConfig
from unstructured.ingest.v2.cli.utils import FileOrJson
from unstructured.ingest.v2.processes.connectors.fsspec.gcs import (
    CONNECTOR_TYPE,
)


@dataclass
class GcsCliDownloadConfig(FsspecCliDownloadConfig):
    pass


@dataclass
class GcsCliIndexerConfig(FsspecCliIndexerConfig):
    pass


@dataclass
class GcsCliConnectionConfig(CliConfig):
    @staticmethod
    def get_cli_options() -> list[click.Option]:
        help_string = """
          Options:
          - ``None``, GCSFS will attempt to guess your credentials in the
        following order: gcloud CLI default, gcsfs cached token, google compute
        metadata service, anonymous.
          - ``'google_default'``, your default gcloud credentials will be used,
            which are typically established by doing ``gcloud login`` in a terminal.
          - ``'cache'``, credentials from previously successful gcsfs
            authentication will be used (use this after "browser" auth succeeded)
          - ``'anon'``, no authentication is performed, and you can only
            access data which is accessible to allUsers (in this case, the project and
            access level parameters are meaningless)
          - ``'browser'``, you get an access code with which you can
            authenticate via a specially provided URL
          - if ``'cloud'``, we assume we are running within google compute
            or google container engine, and query the internal metadata directly for
            a token.
          - you may supply a token generated by the
            [gcloud](https://cloud.google.com/sdk/docs/)
            utility; this is either a python dictionary or the name of a file
            containing the JSON returned by logging in with the gcloud CLI tool.
          """
        options = [
            click.Option(
                ["--service-account-key"],
                default=None,
                type=FileOrJson(allow_raw_str=True),
                help=help_string,
            ),
        ]
        return options


@dataclass
class GcsUploaderConfig(FsspecCliUploaderConfig):
    pass


gcs_src_cmd = SrcCmd(
    cmd_name=CONNECTOR_TYPE,
    indexer_config=GcsCliIndexerConfig,
    connection_config=GcsCliConnectionConfig,
    downloader_config=GcsCliDownloadConfig,
)

gcs_dest_cmd = DestCmd(
    cmd_name=CONNECTOR_TYPE,
    connection_config=GcsCliConnectionConfig,
    uploader_config=GcsUploaderConfig,
)
