"""
Configuration Management
========================

Handles loading, validation, and management of CodeSentinel configuration.
"""

import json
import os
from pathlib import Path
from typing import Dict, Any, Optional, List, Tuple


class ConfigManager:
    """Manages CodeSentinel configuration files."""

    def __init__(self, config_path: Path = None, **kwargs):
        """
        Initialize configuration manager.

        Args:
            config_path: Path to the configuration file.
        """
        # Backward-compat: some tests pass config_file=...
        if config_path is None and 'config_file' in kwargs:
            config_path = kwargs.get('config_file')

        self.config_path = Path(config_path) if config_path is not None else Path.cwd() / 'codesentinel.json'
        # Legacy attribute for backward compatibility with tests
        self.config_file = str(self.config_path)
        self.config = {}
        self.config_loaded = False

    def load_config(self) -> Dict[str, Any]:
        """
        Load configuration from file.

        Returns:
            Dict containing configuration data.
        """
        try:
            if self.config_path.exists():
                with open(self.config_path, 'r') as f:
                    self.config = json.load(f)
                self.config_loaded = True
            else:
                # Legacy behavior for tests: return empty when not present
                self.config = {}
                self.config_loaded = False

        except Exception as e:
            print(f"Warning: Could not load config from {self.config_path}: {e}")
            self.config = self._create_default_config()

        # Do not inject defaults here to preserve strict load semantics expected by tests.
        # Policy defaults are applied at runtime by components (e.g., DevAudit) if missing.

        return self.config

    def save_config(self, data: Optional[Dict[str, Any]] = None):
        """Save configuration to file.

        Args:
            data: Optional configuration dict to save; when None saves current self.config.
        """
        try:
            if data is not None:
                # Work on a shallow copy to avoid mutating caller's dict
                self.config = dict(data)

            # Sanitize secrets before writing to disk (security patch v1.0.1)
            def _prune_secrets(cfg: Dict[str, Any]) -> Dict[str, Any]:
                try:
                    # Top-level alerts.email shape
                    alerts = cfg.get("alerts")
                    if isinstance(alerts, dict):
                        # Newer shape: alerts -> email
                        email_cfg = alerts.get("email")
                        if isinstance(email_cfg, dict) and "password" in email_cfg:
                            email_cfg = email_cfg.copy()
                            email_cfg.pop("password", None)
                            alerts["email"] = email_cfg

                        # Newer shape: alerts -> slack
                        slack_cfg = alerts.get("slack")
                        if isinstance(slack_cfg, dict) and "access_token" in slack_cfg:
                            slack_cfg = slack_cfg.copy()
                            slack_cfg.pop("access_token", None)
                            alerts["slack"] = slack_cfg

                    # Legacy shape: alerts -> channels -> email/slack
                    channels = alerts.get("channels") if isinstance(alerts, dict) else None
                    if isinstance(channels, dict):
                        if isinstance(channels.get("email"), dict):
                            email_ch = channels["email"].copy()
                            email_ch.pop("password", None)
                            channels["email"] = email_ch
                        if isinstance(channels.get("slack"), dict):
                            slack_ch = channels["slack"].copy()
                            # Do not remove webhook here (runtime requires it); tokens removed
                            slack_ch.pop("access_token", None)
                            channels["slack"] = slack_ch

                    # GitHub tokens
                    github = cfg.get("github")
                    if isinstance(github, dict):
                        gh = github.copy()
                        gh.pop("access_token", None)
                        gh.pop("token", None)
                        cfg["github"] = gh
                except Exception:
                    # Best-effort; never fail save due to pruning
                    pass
                return cfg

            self.config = _prune_secrets(self.config)
            self.config_path.parent.mkdir(parents=True, exist_ok=True)
            with open(self.config_path, 'w') as f:
                json.dump(self.config, f, indent=2)
            # Best-effort: restrict permissions on POSIX systems to owner-only
            try:
                if os.name != 'nt':
                    os.chmod(self.config_path, 0o600)
            except Exception:
                # Do not fail save due to permission set errors
                pass
        except Exception as e:
            raise Exception(f"Could not save config to {self.config_path}: {e}")

    def _create_default_config(self) -> Dict[str, Any]:
        """Create default configuration."""
        return {
            "version": "1.0.0",
            "enabled": True,
            "policy": {
                # Non-destructive, feature-preserving operations are mandatory
                "non_destructive": True,
                "feature_preservation": True,
                "conflict_resolution": "merge-prefer-existing",
                "principles": ["SECURITY", "EFFICIENCY", "MINIMALISM"],
                "hierarchy": ["CORE_CONCEPTS", "PERMANENT_DIRECTIVES", "PERSISTENT_POLICIES"]
            },
            "dev_audit": {
                "trigger_tokens": ["!!!!"],
                "enforce_policy": True,
            },
            "integrity": {
                "enabled": False,  # Disabled by default until baseline generated
                "hash_algorithm": "sha256",
                "whitelist_patterns": [
                    "**/.codesentinel_integrity.json",  # Baseline file itself
                    "**/test_*.py",  # Test files can change frequently
                    "**/__pycache__/**",  # Excluded anyway, but explicit
                    "**/.*",  # Hidden files (git, env, etc.)
                ],
                "critical_files": [
                    # Files that must not be modified without authorization
                    ".github/copilot-instructions.md",
                    "SECURITY.md",
                    "docs/POLICY.md",
                    "codesentinel/core/dev_audit.py",
                    "codesentinel/utils/config.py"
                ],
                "auto_update_baseline": False,  # Require explicit update
                "alert_on_violation": True  # Send alerts on integrity violations
            },
            "alerts": {
                "enabled": True,
                "channels": {
                    "console": {"enabled": True},
                    "file": {"enabled": True, "log_file": "codesentinel.log"},
                    "email": {"enabled": False},
                    "slack": {"enabled": False}
                },
                "alert_rules": {
                    "critical_security_issues": True,
                    "task_failures": True,
                    "dependency_vulnerabilities": True
                }
            },
            "github": {
                "copilot": {"enabled": False},
                "api": {"enabled": False},
                "repository": {"enabled": False}
            },
            "ide": {
                "vscode": {"enabled": False}
            },
            "maintenance": {
                "daily": {"enabled": True, "schedule": "09:00"},
                "weekly": {"enabled": True, "schedule": "Monday 10:00"},
                "monthly": {"enabled": True, "schedule": "1st 11:00"}
            },
            "logging": {
                "level": "INFO",
                "file": "codesentinel.log",
                "max_size": "10MB",
                "retention": "30 days"
            }
        }

    def _ensure_defaults(self) -> None:
        """Ensure critical default sections exist and are valid.

        This guarantees that '!!!!' semantics and security-first, non-destructive
        policies persist even if the config was created by an older version.
        """
        # Insert policy defaults if missing
        if "policy" not in self.config or not isinstance(self.config.get("policy"), dict):
            self.config.setdefault("policy", {})
        policy = self.config["policy"]
        policy.setdefault("non_destructive", True)
        policy.setdefault("feature_preservation", True)
        policy.setdefault("conflict_resolution", "merge-prefer-existing")
        policy.setdefault("principles", ["SECURITY", "EFFICIENCY", "MINIMALISM"])

        # Dev audit trigger defaults
        if "dev_audit" not in self.config or not isinstance(self.config.get("dev_audit"), dict):
            self.config.setdefault("dev_audit", {})
        da = self.config["dev_audit"]
        triggers = da.get("trigger_tokens") or ["!!!!"]
        if "!!!!" not in triggers:
            triggers.append("!!!!")
        da["trigger_tokens"] = triggers
        da.setdefault("enforce_policy", True)

    def get(self, key: str, default: Any = None) -> Any:
        """
        Get configuration value.

        Args:
            key: Configuration key (dot-separated for nested keys).
            default: Default value if key not found.

        Returns:
            Configuration value or default.
        """
        keys = key.split('.')
        value = self.config

        for k in keys:
            if isinstance(value, dict) and k in value:
                value = value[k]
            else:
                return default

        return value

    def set(self, key: str, value: Any):
        """
        Set configuration value.

        Args:
            key: Configuration key (dot-separated for nested keys).
            value: Value to set.
        """
        keys = key.split('.')
        config = self.config

        # Navigate to the parent of the target key
        for k in keys[:-1]:
            if k not in config:
                config[k] = {}
            config = config[k]

        # Set the value
        config[keys[-1]] = value

    def validate_config(self, data: Optional[Dict[str, Any]] = None) -> Tuple[bool, List[str]]:
        """
        Validate configuration.

        Args:
            data: Optional configuration to validate; defaults to current config.

        Returns:
            Tuple (is_valid, errors)
        """
        cfg = data if data is not None else self.config
        errors: List[str] = []

        if not isinstance(cfg, dict):
            return False, ["Configuration must be a dictionary"]

        # Minimal validation matching tests' expectations
        alerts = cfg.get('alerts', {})
        if 'email' in alerts and alerts['email'].get('enabled'):
            if not alerts['email'].get('smtp_server'):
                errors.append("Email alerts enabled but 'smtp_server' missing")

        return len(errors) == 0, errors