# StrataRegula CLI Reference v0.2.0

Complete command-line interface reference for StrataRegula.

## Table of Contents
- [Basic Commands](#basic-commands)
- [Compilation Options](#compilation-options)
- [Visualization Features](#visualization-features)
- [Diagnostics](#diagnostics)
- [Advanced Usage](#advanced-usage)
- [Examples](#examples)

## Basic Commands

### `strataregula compile`
Main compilation command for processing YAML configurations with pattern expansion.

```bash
strataregula compile [OPTIONS]
```

**Required Options:**
- `--traffic PATH` - Input YAML file with service patterns

**Optional Arguments:**
- `--prefectures PATH` - YAML file defining region hierarchy
- `--output PATH` - Output file path (default: stdout)
- `--format FORMAT` - Output format: `python`, `json`, `yaml` (default: python)

**Examples:**
```bash
# Basic compilation
strataregula compile --traffic services.yaml

# With custom output format
strataregula compile --traffic services.yaml --format json

# With region hierarchy
strataregula compile --traffic services.yaml --prefectures regions.yaml --output result.py
```

## Compilation Options

### Output Formats

#### Python Format (Default)
Generates optimized Python code with O(1) lookup functions:

```bash
strataregula compile --traffic config.yaml --format python
```

**Output:**
```python
# Generated by StrataRegula v0.2.0
DIRECT_MAPPING = {
    "web.frontend.response": 200,
    "api.v1.timeout": 30,
    # ...
}

def get_service_time(service_name: str, default: float = 0.0) -> float:
    return DIRECT_MAPPING.get(service_name, default)
```

#### JSON Format
Clean JSON output for API integration:

```bash
strataregula compile --traffic config.yaml --format json
```

**Output:**
```json
{
  "direct_mapping": {
    "web.frontend.response": 200,
    "api.v1.timeout": 30
  },
  "metadata": {
    "timestamp": "2025-08-26T14:30:22Z",
    "version": "0.2.0"
  }
}
```

#### YAML Format
Human-readable YAML output:

```bash
strataregula compile --traffic config.yaml --format yaml
```

## Visualization Features (v0.2.0)

### `--dump-compiled-config`
Shows the internal compiled configuration for debugging and verification.

```bash
strataregula compile --traffic config.yaml --dump-compiled-config
```

### `--dump-format FORMAT`
Specifies visualization format (used with `--dump-compiled-config`).

**Available formats:**
- `json` - Structured JSON output
- `yaml` - Human-readable YAML
- `python` - Python code representation  
- `table` - Tabular view with columns
- `tree` - Hierarchical tree structure

#### Tree Format Example
```bash
strataregula compile --traffic config.yaml --dump-compiled-config --dump-format tree
```

**Output:**
```
Configuration Tree:
services/
├── web/
│   ├── frontend/
│   │   ├── response: 200ms
│   │   └── timeout: 5000ms
│   └── backend/
│       ├── response: 300ms
│       └── timeout: 8000ms
└── api/
    ├── v1/
    │   └── rate_limit: 1000/hour
    └── v2/
        └── rate_limit: 2000/hour
```

#### Table Format Example
```bash
strataregula compile --traffic config.yaml --dump-compiled-config --dump-format table
```

**Output:**
```
┌─────────────────────────┬─────────┬──────────────┬────────────┐
│ Pattern                 │ Value   │ Type         │ Source     │
├─────────────────────────┼─────────┼──────────────┼────────────┤
│ web.frontend.response   │ 200     │ service_time │ expansion  │
│ web.backend.response    │ 300     │ service_time │ expansion  │
│ api.v1.rate_limit      │ 1000    │ config       │ direct     │
│ api.v2.rate_limit      │ 2000    │ config       │ direct     │
└─────────────────────────┴─────────┴──────────────┴────────────┘
```

## Diagnostics

### `strataregula doctor`
Environment compatibility and diagnostic tool.

```bash
strataregula doctor [OPTIONS]
```

**Options:**
- `--verbose`, `-v` - Show detailed diagnostic information
- `--fix-suggestions`, `-f` - Show fix suggestions for detected issues

#### Basic Check
```bash
strataregula doctor
```

**Output:**
```
🔍 StrataRegula Environment Check

✅ Python Version: 3.9.16 (compatible)
✅ YAML Support: PyYAML 6.0 installed
⚠️  Performance Monitoring: psutil not available
✅ Output Formatting: Rich 13.3.5 installed
✅ Plugin System: All dependencies available

Overall Status: ✅ Compatible
```

#### Verbose Check
```bash
strataregula doctor --verbose
```

**Output includes:**
- Detailed Python environment information
- Package versions and compatibility
- Plugin system status
- Performance benchmarks
- Memory and CPU information (if psutil available)

#### Fix Suggestions
```bash
strataregula doctor --fix-suggestions
```

**Example output:**
```
🛠️  FIX SUGGESTIONS:

Performance Monitoring (Optional):
  pip install 'strataregula[performance]'
  
For pyenv users with build issues:
  pyenv install 3.9.16
  pyenv global 3.9.16
  pip install --upgrade pip
  pip install strataregula

For dependency conflicts:
  pip install --upgrade --force-reinstall strataregula
```

### `strataregula examples`
Shows usage examples and common patterns.

```bash
strataregula examples
```

**Output includes:**
- Basic usage examples
- Advanced pattern examples
- CLI feature demonstrations
- Plugin usage examples

## Advanced Usage

### Global Options

Available for all commands:

- `--verbose`, `-v` - Enable verbose output
- `--quiet`, `-q` - Suppress non-essential output
- `--version` - Show version information

### Plugin Control

Control plugin system behavior:

```bash
# With plugin system (default)
strataregula compile --traffic config.yaml

# Programmatic plugin control (Python API)
from strataregula.core.config_compiler import ConfigCompiler

# Enable plugins
compiler = ConfigCompiler(use_plugins=True)

# Disable plugins for maximum performance
compiler = ConfigCompiler(use_plugins=False)
```

### Performance Tuning

```bash
# For large configurations, increase memory limit
strataregula compile --traffic large_config.yaml --format json > output.json

# Monitor compilation with verbose output
strataregula compile --traffic config.yaml --verbose

# Profile performance
time strataregula compile --traffic config.yaml --quiet
```

### Batch Processing

```bash
# Process multiple files
for config in configs/*.yaml; do
    strataregula compile --traffic "$config" --output "output/$(basename $config .yaml).py"
done

# Combine with visualization
strataregula compile --traffic config.yaml --dump-compiled-config --dump-format json | jq '.'
```

## Examples

### Basic Pattern Expansion
```bash
# Input: services.yaml
# service_times:
#   web.*.response: 200
#   api.*.timeout: 30

strataregula compile --traffic services.yaml
```

### Multi-Format Output
```bash
# Generate Python, JSON, and YAML outputs
strataregula compile --traffic config.yaml --format python --output config.py
strataregula compile --traffic config.yaml --format json --output config.json  
strataregula compile --traffic config.yaml --format yaml --output config.yaml
```

### Debugging Workflow
```bash
# 1. Check environment
strataregula doctor

# 2. View what patterns were found
strataregula compile --traffic config.yaml --dump-compiled-config --dump-format tree

# 3. Generate final output
strataregula compile --traffic config.yaml --output result.py

# 4. Validate result
python -c "import result; print(result.get_service_time('web.frontend.response'))"
```

### Integration with Other Tools
```bash
# Use with jq for JSON processing
strataregula compile --traffic config.yaml --format json | jq '.direct_mapping'

# Pipe to file and format
strataregula compile --traffic config.yaml --format python | black - > formatted_output.py

# Combine with git for version control
strataregula compile --traffic config.yaml --output config_v$(git rev-parse --short HEAD).py
```

### Pipeline Usage
```bash
# In CI/CD pipeline
strataregula doctor --fix-suggestions || exit 1
strataregula compile --traffic $CONFIG_FILE --format json --output deployment_config.json
```

## Error Handling

### Common Exit Codes
- `0` - Success
- `1` - Configuration error or compilation failure
- `130` - Interrupted by user (Ctrl+C)

### Troubleshooting

**File not found:**
```bash
strataregula compile --traffic nonexistent.yaml
# Error: Configuration file not found: nonexistent.yaml
```

**Invalid YAML:**
```bash
# Check file syntax first
python -c "import yaml; yaml.safe_load(open('config.yaml'))"
```

**Plugin errors:**
```bash
# Run with verbose output to see plugin details
strataregula compile --traffic config.yaml --verbose
```

**Memory issues with large files:**
```bash
# Use streaming for large configurations
strataregula compile --traffic large_config.yaml --format json
```

---

For more information, see:
- [Documentation Index](../index.md) - Main documentation hub
- [Plugin Quick Start](../developer-guide/PLUGIN_QUICKSTART.md) - Plugin development guide
- [API Reference](../api-reference/API_REFERENCE.md) - Complete Python API documentation
- [Examples](../examples/examples.md) - Real-world usage patterns