# coding: utf-8

import json
import uuid

from stix2.canonicalization.Canonicalize import canonicalize


class MalwareAnalysis:
    """Main MalwareAnalysis class for OpenCTI

    Manages malware analysis reports and results in the OpenCTI platform.

    :param opencti: instance of :py:class:`~pycti.api.opencti_api_client.OpenCTIApiClient`
    """

    def __init__(self, opencti):
        self.opencti = opencti
        self.properties = """
            id
            standard_id
            entity_type
            parent_types
            spec_version
            created_at
            updated_at
            status {
                id
                template {
                  id
                  name
                  color
                }
            }
            createdBy {
                ... on Identity {
                    id
                    standard_id
                    entity_type
                    parent_types
                    spec_version
                    identity_class
                    name
                    description
                    roles
                    contact_information
                    x_opencti_aliases
                    created
                    modified
                    objectLabel {
                        id
                        value
                        color
                    }
                }
                ... on Organization {
                    x_opencti_organization_type
                    x_opencti_reliability
                }
                ... on Individual {
                    x_opencti_firstname
                    x_opencti_lastname
                }
            }
            objectOrganization {
                id
                standard_id
                name
            }
            objectMarking {
                id
                standard_id
                entity_type
                definition_type
                definition
                created
                modified
                x_opencti_order
                x_opencti_color
            }
            objectLabel {
                id
                value
                color
            }
            externalReferences {
                edges {
                    node {
                        id
                        standard_id
                        entity_type
                        source_name
                        description
                        url
                        hash
                        external_id
                        created
                        modified
                    }
                }
            }
            revoked
            confidence
            created
            modified
            product
            result_name
            result
            submitted
            analysis_started
            analysis_ended
            version
            configuration_version
            analysis_engine_version
            analysis_definition_version
            modules
        """
        self.properties_with_files = """
            id
            standard_id
            entity_type
            parent_types
            spec_version
            created_at
            updated_at
            status {
                id
                template {
                  id
                  name
                  color
                }
            }
            createdBy {
                ... on Identity {
                    id
                    standard_id
                    entity_type
                    parent_types
                    spec_version
                    identity_class
                    name
                    description
                    roles
                    contact_information
                    x_opencti_aliases
                    created
                    modified
                    objectLabel {
                        id
                        value
                        color
                    }
                }
                ... on Organization {
                    x_opencti_organization_type
                    x_opencti_reliability
                }
                ... on Individual {
                    x_opencti_firstname
                    x_opencti_lastname
                }
            }
            objectOrganization {
                id
                standard_id
                name
            }
            objectMarking {
                id
                standard_id
                entity_type
                definition_type
                definition
                created
                modified
                x_opencti_order
                x_opencti_color
            }
            objectLabel {
                id
                value
                color
            }
            externalReferences {
                edges {
                    node {
                        id
                        standard_id
                        entity_type
                        source_name
                        description
                        url
                        hash
                        external_id
                        created
                        modified
                        importFiles {
                            edges {
                                node {
                                    id
                                    name
                                    size
                                    metaData {
                                        mimetype
                                        version
                                    }
                                }
                            }
                        }
                    }
                }
            }
            revoked
            confidence
            created
            modified
            product
            result_name
            result
            submitted
            analysis_started
            analysis_ended
            version
            configuration_version
            analysis_engine_version
            analysis_definition_version
            modules
            importFiles {
                edges {
                    node {
                        id
                        name
                        size
                        metaData {
                            mimetype
                            version
                        }
                    }
                }
            }
        """

    @staticmethod
    def generate_id(result_name, product=None, submitted=None):
        result_name = result_name.lower().strip()
        data = {"result_name": result_name, "product": product}
        if submitted is not None:
            data = {**data, "submitted": submitted}
        data = canonicalize(data, utf8=False)
        id = str(uuid.uuid5(uuid.UUID("00abedb4-aa42-466c-9c01-fed23315a9b7"), data))
        return "malware-analysis--" + id

    @staticmethod
    def generate_id_from_data(data):
        return MalwareAnalysis.generate_id(
            data["result_name"], data["product"], data.get("submitted")
        )

    """
        List Malware analysis objects

        :param filters: the filters to apply
        :param search: the search keyword
        :param first: return the first n rows from the after ID (or the beginning if not set)
        :param after: ID of the first row for pagination
        :return List of MalwareAnalysis objects
    """

    def list(self, **kwargs):
        filters = kwargs.get("filters", None)
        search = kwargs.get("search", None)
        first = kwargs.get("first", 500)
        after = kwargs.get("after", None)
        order_by = kwargs.get("orderBy", None)
        order_mode = kwargs.get("orderMode", None)
        custom_attributes = kwargs.get("customAttributes", None)
        get_all = kwargs.get("getAll", False)
        with_pagination = kwargs.get("withPagination", False)
        with_files = kwargs.get("withFiles", False)

        self.opencti.app_logger.info(
            "Listing Malware analyses with filters", {"filters": json.dumps(filters)}
        )
        query = (
            """
            query MalwareAnalyses($filters: FilterGroup, $search: String, $first: Int, $after: ID, $orderBy: MalwareAnalysesOrdering, $orderMode: OrderingMode) {
                malwareAnalyses(filters: $filters, search: $search, first: $first, after: $after, orderBy: $orderBy, orderMode: $orderMode) {
                    edges {
                        node {
                            """
            + (
                custom_attributes
                if custom_attributes is not None
                else (self.properties_with_files if with_files else self.properties)
            )
            + """
                        }
                    }
                    pageInfo {
                        startCursor
                        endCursor
                        hasNextPage
                        hasPreviousPage
                        globalCount
                    }
                }
            }
        """
        )
        result = self.opencti.query(
            query,
            {
                "filters": filters,
                "search": search,
                "first": first,
                "after": after,
                "orderBy": order_by,
                "orderMode": order_mode,
            },
        )

        if get_all:
            final_data = []
            data = self.opencti.process_multiple(result["data"]["malwareAnalyses"])
            final_data = final_data + data
            while result["data"]["malwareAnalyses"]["pageInfo"]["hasNextPage"]:
                after = result["data"]["malwareAnalyses"]["pageInfo"]["endCursor"]
                self.opencti.app_logger.info(
                    "Listing Malware analyses", {"after": after}
                )
                result = self.opencti.query(
                    query,
                    {
                        "filters": filters,
                        "search": search,
                        "first": first,
                        "after": after,
                        "orderBy": order_by,
                        "orderMode": order_mode,
                    },
                )
                data = self.opencti.process_multiple(result["data"]["malwareAnalyses"])
                final_data = final_data + data
            return final_data
        else:
            return self.opencti.process_multiple(
                result["data"]["malwareAnalyses"], with_pagination
            )

    """
        Read a Malware analysis object

        :param id: the id of the Malware analysis
        :param filters: the filters to apply if no id provided
        :return Malware analysis object
    """

    def read(self, **kwargs):
        id = kwargs.get("id", None)
        filters = kwargs.get("filters", None)
        custom_attributes = kwargs.get("customAttributes", None)
        with_files = kwargs.get("withFiles", False)
        if id is not None:
            self.opencti.app_logger.info("Reading Malware analysis", {"id": id})
            query = (
                """
                query MalwareAnalysis($id: String!) {
                    malwareAnalysis(id: $id) {
                        """
                + (
                    custom_attributes
                    if custom_attributes is not None
                    else (self.properties_with_files if with_files else self.properties)
                )
                + """
                    }
                }
             """
            )
            result = self.opencti.query(query, {"id": id})
            return self.opencti.process_multiple_fields(
                result["data"]["malwareAnalysis"]
            )
        elif filters is not None:
            result = self.list(filters=filters)
            if len(result) > 0:
                return result[0]
            else:
                return None
        else:
            self.opencti.app_logger.error(
                "[opencti_malwareAnalysis] Missing parameters: id or filters"
            )
            return None

    """
        Create a Malware analysis object

        :param name: the name of the Malware analysis
        :return Malware analysis object
    """

    def create(self, **kwargs):
        stix_id = kwargs.get("stix_id", None)
        created_by = kwargs.get("createdBy", None)
        object_marking = kwargs.get("objectMarking", None)
        object_label = kwargs.get("objectLabel", None)
        external_references = kwargs.get("externalReferences", None)
        revoked = kwargs.get("revoked", None)
        confidence = kwargs.get("confidence", None)
        lang = kwargs.get("lang", None)
        created = kwargs.get("created", None)
        modified = kwargs.get("modified", None)
        product = kwargs.get("product", None)
        result_name = kwargs.get("result_name", None)
        result = kwargs.get("result", None)
        submitted = kwargs.get("submitted", None)
        analysis_started = kwargs.get("analysis_started", None)
        analysis_ended = kwargs.get("analysis_ended", None)
        version = kwargs.get("version", None)
        configuration_version = kwargs.get("configuration_version", None)
        analysis_engine_version = kwargs.get("analysis_engine_version", None)
        analysis_definition_version = kwargs.get("analysis_definition_version", None)
        modules = kwargs.get("modules", None)
        hostVm = kwargs.get("hostVm", None)
        operatingSystem = kwargs.get("operatingSystem", None)
        installedSoftware = kwargs.get("installedSoftware", None)
        sample = kwargs.get("sample", None)
        analysisSco = kwargs.get("analysisSco", None)
        x_opencti_stix_ids = kwargs.get("x_opencti_stix_ids", None)
        granted_refs = kwargs.get("objectOrganization", None)
        x_opencti_workflow_id = kwargs.get("x_opencti_workflow_id", None)
        update = kwargs.get("update", False)

        if product is not None and result_name is not None:
            self.opencti.app_logger.info(
                "Creating Malware analysis", {"product": product}
            )
            query = """
                mutation MalwareAnalysisAdd($input: MalwareAnalysisAddInput!) {
                    malwareAnalysisAdd(input: $input) {
                        id
                        standard_id
                        entity_type
                        parent_types
                    }
                }
            """
            result = self.opencti.query(
                query,
                {
                    "input": {
                        "stix_id": stix_id,
                        "createdBy": created_by,
                        "objectMarking": object_marking,
                        "objectLabel": object_label,
                        "objectOrganization": granted_refs,
                        "externalReferences": external_references,
                        "revoked": revoked,
                        "confidence": confidence,
                        "lang": lang,
                        "created": created,
                        "modified": modified,
                        "product": product,
                        "result_name": result_name,
                        "result": result,
                        "submitted": submitted,
                        "analysis_started": analysis_started,
                        "analysis_ended": analysis_ended,
                        "version": version,
                        "configuration_version": configuration_version,
                        "analysis_engine_version": analysis_engine_version,
                        "analysis_definition_version": analysis_definition_version,
                        "modules": modules,
                        "hostVm": hostVm,
                        "operatingSystem": operatingSystem,
                        "installedSoftware": installedSoftware,
                        "analysisSample": sample,
                        "analysisSco": analysisSco,
                        "x_opencti_stix_ids": x_opencti_stix_ids,
                        "x_opencti_workflow_id": x_opencti_workflow_id,
                        "update": update,
                    }
                },
            )
            return self.opencti.process_multiple_fields(
                result["data"]["malwareAnalysisAdd"]
            )
        else:
            self.opencti.app_logger.error(
                "[opencti_malwareAnalysis] Missing parameters: product and result_name"
            )

    """
        Import an Malware analysis object from a STIX2 object

        :param stixObject: the Stix-Object Malware analysis
        :return Malware analysis object
    """

    def import_from_stix2(self, **kwargs):
        stix_object = kwargs.get("stixObject", None)
        extras = kwargs.get("extras", {})
        update = kwargs.get("update", False)
        if stix_object is not None:
            # Search in extensions
            if "x_opencti_stix_ids" not in stix_object:
                stix_object["x_opencti_stix_ids"] = (
                    self.opencti.get_attribute_in_extension("stix_ids", stix_object)
                )
            if "x_opencti_granted_refs" not in stix_object:
                stix_object["x_opencti_granted_refs"] = (
                    self.opencti.get_attribute_in_extension("granted_refs", stix_object)
                )
            if "x_opencti_workflow_id" not in stix_object:
                stix_object["x_opencti_workflow_id"] = (
                    self.opencti.get_attribute_in_extension("workflow_id", stix_object)
                )

            return self.create(
                stix_id=stix_object["id"],
                createdBy=(
                    extras["created_by_id"] if "created_by_id" in extras else None
                ),
                objectMarking=(
                    extras["object_marking_ids"]
                    if "object_marking_ids" in extras
                    else None
                ),
                objectLabel=(
                    extras["object_label_ids"] if "object_label_ids" in extras else None
                ),
                externalReferences=(
                    extras["external_references_ids"]
                    if "external_references_ids" in extras
                    else None
                ),
                revoked=stix_object["revoked"] if "revoked" in stix_object else None,
                confidence=(
                    stix_object["confidence"] if "confidence" in stix_object else None
                ),
                lang=stix_object["lang"] if "lang" in stix_object else None,
                created=stix_object["created"] if "created" in stix_object else None,
                modified=stix_object["modified"] if "modified" in stix_object else None,
                product=stix_object["product"] if "product" in stix_object else None,
                result_name=(
                    stix_object["result_name"] if "result_name" in stix_object else None
                ),
                result=stix_object["result"] if "result" in stix_object else None,
                submitted=(
                    stix_object["submitted"] if "submitted" in stix_object else None
                ),
                analysis_started=(
                    stix_object["analysis_started"]
                    if "analysis_started" in stix_object
                    else None
                ),
                analysis_ended=(
                    stix_object["analysis_ended"]
                    if "analysis_ended" in stix_object
                    else None
                ),
                version=stix_object["version"] if "version" in stix_object else None,
                configuration_version=(
                    stix_object["configuration_version"]
                    if "configuration_version" in stix_object
                    else None
                ),
                analysis_engine_version=(
                    stix_object["analysis_engine_version"]
                    if "analysis_engine_version" in stix_object
                    else None
                ),
                analysis_definition_version=(
                    stix_object["analysis_definition_version"]
                    if "analysis_definition_version" in stix_object
                    else None
                ),
                operatingSystem=(
                    stix_object["operating_system_ref"]
                    if "operating_system_ref" in stix_object
                    else None
                ),
                analysisSco=(
                    stix_object["analysis_sco_refs"]
                    if "analysis_sco_refs" in stix_object
                    else None
                ),
                hostVm=(
                    stix_object["host_vm_ref"] if "host_vm_ref" in stix_object else None
                ),
                installedSoftware=(
                    stix_object["installed_software_refs"]
                    if "installed_software_refs" in stix_object
                    else None
                ),
                sample=(
                    stix_object["sample_ref"] if "sample_ref" in stix_object else None
                ),
                modules=stix_object["modules"] if "modules" in stix_object else None,
                x_opencti_stix_ids=(
                    stix_object["x_opencti_stix_ids"]
                    if "x_opencti_stix_ids" in stix_object
                    else None
                ),
                objectOrganization=(
                    stix_object["x_opencti_granted_refs"]
                    if "x_opencti_granted_refs" in stix_object
                    else None
                ),
                x_opencti_workflow_id=(
                    stix_object["x_opencti_workflow_id"]
                    if "x_opencti_workflow_id" in stix_object
                    else None
                ),
                update=update,
            )
        else:
            self.opencti.app_logger.error(
                "[opencti_malware_analysis] Missing parameters: stixObject"
            )
