"""
Support values on MoaT-KV
"""

from __future__ import annotations

import logging

from .device import Register as BaseRegister
from moat.util import Path, NotGiven
import anyio

logger = logging.getLogger(__name__)


class Register(BaseRegister):
    """
    One possibly-complex Modbus register that's mirrored from and/or to MoaT-KV
    """

    def __init__(self, *a, mt_kv=None, mt_ln=None, tg=None, is_server=False, **kw):
        super().__init__(*a, **kw)
        self.mt_kv = mt_kv
        self.mt_ln = mt_ln
        self.tg = tg
        self.is_server = is_server

        if self.src is None and self.dest is None:
            if self.data.get("slot", "write") != "write":
                if getattr(self.data, "_is_orig", False):
                    logger.warning("%s:%s: no source/destination", self.unit, self.path)
            return

    async def start(self):
        await super().start()

        # logger.info("%s:%s: Polling", self.unit, self.path)
        tg = self.tg

        if (dest := self.dest) is not None:
            if self.is_server:
                slot = None
            else:
                slot = self.data.get("slot", None)
                if slot is None:
                    logger.warning("%s:%s: no read slot", self.unit, self.path)

            # logger.info("%s:%s: Write %s", self.unit, self.path, dest)
            if isinstance(dest, Path):
                dest = (dest,)

            for d in dest:
                if d.mark in ("r","R"):
                    if self.mt_ln is not None:
                        tg.start_soon(self.to_dlink_raw, d, d.mark=="R")
                    else:
                        tg.start_soon(self.to_dkv_raw, d)
                elif d.mark in ("l","L") or self.mt_kv is None:
                    tg.start_soon(self.to_dlink, d, d.mark!="l")
                else:
                    tg.start_soon(self.to_dkv, d)

        if self.src is not None:
            slot = self.data.get("slot", None) if self.dest is None else None
            # if a slot is set AND src is set AND dst is not set,
            # then we want to do a periodic write (keepalive etc.).
            if self.src.mark in ("r","R"):
                if self.mt_ln is not None:
                    mon = self.mt_ln.monitor(self.src)
                else:
                    mon = self.mt_kv.msg_monitor(self.src)
            elif self.src.mark in ("l","L") or self.mt_kv is None:
                mon = self.mt_ln.d_watch(self.src, meta=True, mark=self.src.mark!="l")
            else:
                mon = self.mt_kv.watch(self.src, fetch=True, max_depth=0)

            # logger.info("%s:%s: Watch %s", self.unit, self.path,self.src)

            if self.is_server or slot in (None, "write"):
                await tg.start(self.from_dkv, mon)
            else:
                tg.start_soon(self.from_dkv_p, mon, self.slot.write_delay)

    @property
    def src(self):
        return self.data.get("src")

    @property
    def dest(self):
        return self.data.get("dest")

    def set(self, val):
        self.reg.set(val)

    async def to_dkv(self, dest):
        """Copy a Modbus value to MoaT-KV"""
        async for val in self:
            logger.debug("%s R %r", self.path, val)
            await self.mt_kv.set(dest, value=val, idem=self.data.get("idem", True))

    async def to_dlink(self, dest, retain=False):
        """Copy a Modbus value to MoaT-Link"""
        async for val in self:
            logger.debug("%s L %r", self.path, val)
            await self.mt_ln.d_set(dest, val, retain=retain)

    async def to_dkv_raw(self, dest):
        """Copy a Modbus value to MQTT"""
        async for val in self:
            logger.debug("%s r %r", self.path, val)
            await self.mt_kv.msg_send(list(dest), val)

    async def to_dlink_raw(self, dest, retain=False):
        """Copy a Modbus value to MQTT"""
        async for val in self:
            logger.debug("%s r %r", self.path, val)
            await self.mt_ln.send(list(dest), val, retain=retain)

    async def from_dkv(self, mon, *, task_status):
        """Copy an MQTT value to Modbus"""
        async with mon as mon_:
            async for val in mon_:
                if task_status is not None:
                    task_status.started()
                    task_status = None
                if val is None:  # Link message
                    continue
                if isinstance(val,tuple):  # Link client
                    logger.debug("%s W %r", self.path, val)
                    val=val[0]
                elif "value" not in val:  # KV message
                    logger.debug("%s Wx", self.path)
                    continue
                else:
                    logger.debug("%s W %r", self.path, val)
                    val=val.value
                await self._set(val)

    async def from_dkv_p(self, mon, slot):
        """Copy an MQTT value to Modbus, with periodic refresh"""
        evt = anyio.Event()
        val = NotGiven

        async def per():
            nonlocal evt, val
            while True:
                if val is NotGiven:
                    await evt.wait()
                else:
                    with anyio.move_on_after(slot):
                        await evt.wait()
                if val is NotGiven:
                    continue

                logger.debug("%s Wr %r", self.path, val)
                await self._set(val)

        async with mon as mon_, anyio.create_task_group() as tg:
            tg.start_soon(per)
            first = True
            async for val_ in mon_:
                if isinstance(val,tuple):
                    val = val[0]
                else:
                    try:
                        val = val_.value
                    except AttributeError:
                        if first:
                            first = False
                            continue
                        val = NotGiven
                logger.debug("%s w %r", self.path, val)
                evt.set()
                evt = anyio.Event()

    async def _set(self, value):
        self.value = value

        if (dest := self.dest) is not None and self.data.get("mirror", False):
            if isinstance(dest, Path):
                dest = (dest,)

            for d in dest:
                if d.mark == "r":
                    if self.mt_ln is not None:
                        await self.mt_ln.send(d, self.value)
                    else:
                        await self.mt_kv.msg_send(list(d), self.value)
                elif d.mark == "R":
                    await self.mt_ln.send(d, value, retain=True)
                elif d.mark in ("l","L") or self.mt_kv is None:
                    await self.mt_ln.d_set(d, value, retain, d.mark!="l")
                else:
                    await self.mt_kv.set(d, value=value, idem=self.data.get("idem", True))
