"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const iam = require("@aws-cdk/aws-iam");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment snapshot
// --------------------------------------------------------------
test('Test minimal deployment snapshot', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template'
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new core_1.Stack();
    const pattern = new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template'
    });
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new core_1.Stack();
    const existingRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        description: 'existing role for SageMaker integration',
        inlinePolicies: {
            InvokePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:endpoint/my-endpoint`],
                        actions: ['sagemaker:InvokeEndpoint']
                    })]
            })
        }
    });
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template',
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        apiGatewayExecutionRole: existingRole,
        resourceName: 'my-resource',
        responseMappingTemplate: 'my-response-vtl-template'
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: true,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Resource', {
        PathPart: 'my-resource'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'application/json': 'my-response-vtl-template',
                    },
                    StatusCode: '200'
                },
                {
                    StatusCode: '500',
                    SelectionPattern: '5\\d{2}'
                },
                {
                    StatusCode: '400',
                    SelectionPattern: '4\\d{2}'
                }
            ]
        },
        MethodResponses: [
            { StatusCode: '200' },
            { StatusCode: '500' },
            { StatusCode: '400' }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Description: 'existing role for SageMaker integration'
    });
});
//# sourceMappingURL=data:application/json;base64,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