from __future__ import annotations

import logging
import time
from typing import Optional

from boneio.const import ID, MODEL, NAME, SENSOR
from boneio.helper.config import ConfigHelper
from boneio.helper.filter import Filter
from boneio.helper.ha_discovery import modbus_sensor_availabilty_message
from boneio.message_bus.basic import MessageBus

_LOGGER = logging.getLogger(__name__)


class BaseSensor(Filter):

    _ha_type_ = SENSOR

    def __init__(
        self,
        name: str,
        parent: dict,
        message_bus: MessageBus,
        config_helper: ConfigHelper,
        unit_of_measurement: Optional[str] = None,
        state_class: Optional[str]   = None,
        device_class: Optional[str] = None,
        value_type: Optional[str] = None,
        return_type: Optional[str] = None,
        filters: list = [],
        user_filters: Optional[list] = [],
        ha_filter: str = "round(2)",
    ) -> None:
        self._name = name
        self._parent = parent
        self._decoded_name = self._name.replace(" ", "")
        self._decoded_name_low = self._name.replace(" ", "").lower()
        self._unit_of_measurement = unit_of_measurement
        self._state_class = state_class
        self._device_class = device_class
        self._message_bus = message_bus
        self._config_helper = config_helper
        self._user_filters = user_filters
        self._filters = filters
        self._value = None
        self._return_type = return_type
        self._value_type = value_type
        self._ha_filter = ha_filter
        self._timestamp = time.time()
        self._topic = (
            f"{self._config_helper.ha_discovery_prefix}/{self._ha_type_}/{self._config_helper.topic_prefix}{self._parent[ID]}"
            f"/{self._parent[ID]}{self._decoded_name_low.replace('_', '')}/config"
        )

    def set_user_filters(self, user_filters: list) -> None:
        self._user_filters = user_filters

    def set_value(self, value, timestamp: float) -> None:
        value = self._apply_filters(value=value)
        value = self._apply_filters(
            value=value,
            filters=self._user_filters,
        )
        self._value = value
        self._timestamp = timestamp

    @property
    def return_type(self) -> str:
        return self._return_type

    def get_value(self):
        return self._value

    @property
    def value_type(self) -> str:
        return self._value_type

    @property
    def state(self) -> str | float:
        """Give back state value."""
        return self._value

    @property
    def decoded_name(self) -> str:
        return self._decoded_name_low

    @property
    def name(self) -> str:
        """Return name of the sensor."""
        return self._name

    @property
    def unit_of_measurement(self) -> str:
        return self._unit_of_measurement

    @property
    def last_timestamp(self) -> float:
        return self._timestamp

    @property
    def write_address(self) -> int | None:
        return None

    @property
    def id(self) -> str:
        return f"{self._parent[ID]}{self._decoded_name_low}"

    def send_ha_discovery(self):
        payload = self.discovery_message()
        _LOGGER.debug(
            "Sending %s discovery message for %s of %s",
            self._ha_type_,
            self._name,
            self._parent[ID],
        )
        self._config_helper.add_autodiscovery_msg(
            topic=self._topic, payload=payload, ha_type=self._ha_type_
        )
        self._message_bus.send_message(topic=self._topic, payload=payload)

    def discovery_message(self):
        value_template = f"{{{{ value_json.{self.decoded_name} | {self._ha_filter} }}}}" if self._ha_filter else f"{{{{ value_json.{self.decoded_name} }}}}" 
        
        kwargs = {
            "unit_of_measurement": self.unit_of_measurement,
            "state_class": self._state_class,
            "value_template": value_template,
            "sensor_id": self.name,
        }
        if self._device_class:
            kwargs["device_class"] = self._device_class
        return modbus_sensor_availabilty_message(
            topic=self._config_helper.topic_prefix,
            id=self._parent[ID],
            name=self._parent[NAME],
            state_topic_base=str(self.base_address),
            model=self._parent[MODEL],
            **kwargs,
        )

    def encode_value(self, value: int | float) -> int:
        raise NotImplementedError


class ModbusBaseSensor(BaseSensor):
    def __init__(
        self,
        name: str,
        parent: dict,
        register_address: int,
        base_address: int,
        message_bus: MessageBus,
        config_helper: ConfigHelper,
        unit_of_measurement: Optional[str] = None,
        state_class: Optional[str] = None,
        device_class: Optional[str] = None,
        value_type: Optional[str] = None,
        return_type: Optional[str] = None,
        filters: Optional[list] = None,
        user_filters: Optional[list] = [],
        ha_filter: str = "",
    ) -> None:
        """
        Initialize single sensor.
        :param name: name of sensor
        :param register_address: address of register
        :param base_address: address of base
        :param unit_of_measurement: unit of measurement
        :param state_class: state class
        :param device_class: device class
        :param value_type: type of value
        :param return_type: type of return
        :param user_filters: list of user filters
        :param filters: list of filters
        :param send_ha_autodiscovery: function for sending HA autodiscovery
        """
        super().__init__(
            name=name,
            parent=parent,
            unit_of_measurement=unit_of_measurement,
            state_class=state_class,
            device_class=device_class,
            value_type=value_type,
            return_type=return_type,
            filters=filters,
            message_bus=message_bus,
            config_helper=config_helper,
            user_filters=user_filters,
            ha_filter=ha_filter,
        )
        self._register_address = register_address
        self._base_address = base_address

    @property
    def address(self) -> int:
        return self._register_address

    @property
    def base_address(self) -> int:
        return self._base_address
    