"""MCP23017 Relay module."""

import logging

from adafruit_mcp230xx.mcp23017 import MCP23017, DigitalInOut

from boneio.const import COVER, MCP, OFF, ON, SWITCH
from boneio.relay.basic import BasicRelay

_LOGGER = logging.getLogger(__name__)


class MCPRelay(BasicRelay):
    """Represents MCP Relay output"""

    def __init__(
        self,
        pin: int,
        mcp: MCP23017,
        mcp_id: str,
        output_type: str = SWITCH,
        restored_state: bool = False,
        **kwargs
    ) -> None:
        """Initialize MCP relay."""
        self._pin: DigitalInOut = mcp.get_pin(pin)
        if output_type == COVER:
            """Just in case to not restore state of covers etc."""
            restored_state = False
        super().__init__(
            **kwargs, output_type=output_type, restored_state=restored_state
        )
        self._pin_id = pin
        self._expander_id = mcp_id

        self.init_with_check_if_can_restore_state(restored_state=restored_state)
        _LOGGER.debug("Setup MCP with pin %s", self._pin_id)

    def init_with_check_if_can_restore_state(self, restored_state: bool) -> None:
        if restored_state:
            interlock_manager = getattr(self, "_interlock_manager", None)
            interlock_groups = getattr(self, "_interlock_groups", None)
            if self._interlock_manager and self._interlock_groups:
                if not interlock_manager.can_turn_on(self, interlock_groups):
                    _LOGGER.warning(
                        f"Interlock active: cannot restore ON state for {self._pin_id} at startup"
                    )
                    restored_state = False
        self._pin.switch_to_output(value=restored_state)


    @property
    def expander_type(self) -> str:
        """Check expander type."""
        return MCP

    @property
    def pin_id(self) -> int:
        """Return PIN id."""
        return self._pin_id

    @property
    def is_active(self) -> bool:
        """Is relay active."""
        return self.pin.value

    @property
    def pin(self) -> DigitalInOut:
        """PIN of the relay"""
        return self._pin

    def turn_on(self, time=None) -> None:
        """Call turn on action."""
        self.pin.value = True
        self._state = ON
        if not time:
            self._execute_momentary_turn(momentary_type=ON)

    def turn_off(self, time=None) -> None:
        """Call turn off action."""
        self.pin.value = False
        self._state = OFF
        if not time:
            self._execute_momentary_turn(momentary_type=OFF)
