# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""模型中心服务模块

当前封装的能力：
- 分页查询模型列表
- 获取单个模型详情
- 新建模型
- 编辑模型
- 删除模型
- 上传模型
- 下载模型
- 查询模型 DB 信息
"""

from __future__ import annotations

import io
import os
import time

import httpx
import minio
from loguru import logger

from ..exceptions import APIError
from ..models.artifact import StsResp
from ..models.common import APIWrapper
from ..models.model_center import (
    ListModelsRequest,
    ListModelsResponse,
    ModelCardDetail,
    CreateModelRequest,
    CreateModelResponse,
    EditModelRequest,
    ModelDb,
)
from ..utils.di import SimpleS3Client, DataUploader
from ..utils.s3 import parse_s3_path_strict, download_dir_from_s3_to_local

_BASE = "/model-center/api/v1"
MODEL_STATUS_INIT = "init"
MODEL_STATUS_UPLOADING = "uploading"
MODEL_STATUS_READY = "ready"
MODEL_STATUS_FAILED = "failed"


class ModelCenterService:
    """模型中心业务封装"""

    def __init__(self, http: httpx.Client):
        self._model = _Model(http)

    def list_models(self, payload: ListModelsRequest) -> ListModelsResponse:
        """分页查询模型列表

        Args:
            payload: 查询参数（分页、名称过滤等）

        Returns:
            ListModelsResponse: 包含分页信息与模型数据
        """
        return self._model.list(payload)

    def get_model(self, model_id: int) -> ModelCardDetail:
        """获取模型详情

        Args:
            model_id: 模型 ID

        Returns:
            ModelCardDetail: 模型详情（含 README、模型树、基模型等）
        """
        return self._model.get(model_id)

    def create_model(self, payload: CreateModelRequest) -> int:
        """创建模型

        Args:
            payload: 创建模型所需字段

        Returns:
            int: 后端生成的模型 ID
        """
        return self._model.create(payload)

    def edit_model(self, model_id: int, payload: EditModelRequest) -> None:
        """编辑模型信息

        Args:
            model_id: 模型ID
            payload: 编辑模型信息
        """
        self._model.edit(model_id, payload)

    def delete_model(self, model_id: int) -> None:
        """删除模型

        Args:
            model_id: 目标模型 ID
        """
        self._model.delete(model_id)

    def get_model_db(self, id: int | None = None, name: str | None = None) -> ModelDb:
        """通过 id 或 name 查询模型 DB 信息

        Args:
            id: 模型id
            name: 模型名称

        Returns:
            ModelDb: 模型在DB中的信息
        """
        return self._model.get_model_db(id=id, name=name)

    def upload(
        self,
        model_name: str | None = None,
        local_dir: str | None = None,
        model_id: int | None = None,
        timeout_seconds: int = 3600,
    ) -> None:
        """上传模型

        Args:
            local_dir: 本地模型目录
            model_id: 模型 id
            model_name: 模型名称
            timeout_seconds: 超时时间
        """
        return self._model.upload(
            local_dir=local_dir,
            model_id=model_id,
            model_name=model_name,
            timeout_seconds=timeout_seconds,
        )

    def download(self, *, local_dir: str, model_id: int | None = None, model_name: str | None = None) -> None:
        """下载模型

        Args:
            local_dir: 要下载到的本地目录
            model_id: 模型id
            model_name: 模型名称
        """
        return self._model.download(local_dir=local_dir, model_id=model_id, model_name=model_name)

    @property
    def model(self) -> _Model:
        return self._model


class _Model:

    def __init__(self, http: httpx.Client):
        self._http = http

    def list(self, payload: ListModelsRequest) -> ListModelsResponse:
        resp = self._http.get(f"{_BASE}/models", params=payload.model_dump(by_alias=True, exclude_none=True))
        wrapper = APIWrapper[ListModelsResponse].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")
        return wrapper.data

    def get(self, model_id: int) -> ModelCardDetail:
        resp = self._http.get(f"{_BASE}/models/{model_id}")
        wrapper = APIWrapper[ModelCardDetail].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")
        return wrapper.data

    def create(self, form: CreateModelRequest) -> int:
        files = {
            "name": (None, form.name),
            "is_public": (None, "true" if bool(form.is_public) else "false"),
        }
        if form.description is not None:
            files["description"] = (None, form.description)
        if form.tags is not None:
            files["tags"] = (None, form.tags)

        if form.readme_content:
            md_text = (
                form.readme_content
                if (form.readme_content and form.readme_content.strip())
                else f"# {form.name}\n\nAutogenerated at {int(time.time())}\n"
            )
            files["readme_file"] = ("README.md", io.BytesIO(md_text.encode("utf-8")), "text/markdown")

        resp = self._http.post(f"{_BASE}/models", files=files)
        wrapper = APIWrapper[CreateModelResponse].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")
        return wrapper.data.id

    def edit(self, model_id: int, payload: EditModelRequest) -> None:
        resp = self._http.put(f"{_BASE}/models/{model_id}", json=payload.model_dump(by_alias=True, exclude_none=True))
        wrapper = APIWrapper[dict].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")

    def delete(self, model_id: int) -> None:
        resp = self._http.delete(f"{_BASE}/models/{model_id}")
        wrapper = APIWrapper[dict].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")

    def get_model_db(self, *, id: int | None = None, name: str | None = None) -> ModelDb:
        if id is None and (name is None or name == ""):
            raise ValueError("id or name is required")

        params = {"id": id} if id is not None else {"name": name}
        resp = self._http.get(f"{_BASE}/models/db", params=params)
        wrapper = APIWrapper[ModelDb].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")
        return wrapper.data

    def _get_sts(self) -> StsResp:
        resp = self._http.get(f"{_BASE}/models/get-sts")
        wrapper = APIWrapper[StsResp].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")
        return wrapper.data

    def upload(
        self,
        *,
        local_dir: str,
        model_id: int | None = None,
        model_name: str | None = None,
        timeout_seconds: int = 3600,
    ) -> None:
        if (model_id is None) and (not model_name):
            raise ValueError("id or name is required")
        if not local_dir or not os.path.exists(local_dir):
            raise ValueError(f"local_dir not exists: {local_dir}")

        # 1. get db info
        db_info = self.get_model_db(id=model_id, name=model_name)
        resolved_id = db_info.id
        s3_target = db_info.object_storage_path
        s3_csv_path = db_info.csv_file_path
        s3_status_path = db_info.task_status_s3_path

        # 2. upload file to s3
        sts = self._get_sts()
        s3_client = SimpleS3Client(
            sts.endpoint, sts.access_key_id, sts.secret_access_key, session_token=sts.session_token
        )
        uploader = DataUploader(
            task_id=resolved_id,
            local_path=local_dir,
            s3_target=s3_target,
            csv_path=s3_csv_path,
            status_path=s3_status_path,
            num_workers=40,
        )
        stats = uploader.run(s3_client)

        # 3. invoke model center upload interface
        payload = {
            "model_id": resolved_id,
            "object_cnt": stats.uploaded_count,
            "data_size": stats.uploaded_size,
        }
        resp = self._http.post(f"{_BASE}/models/upload", json=payload)
        wrapper = APIWrapper[dict].model_validate(resp.json())
        if wrapper.code != 0:
            raise APIError(f"backend code {wrapper.code}: {wrapper.msg}")

        logger.info(f"模型上传本地处理完成，model_id={resolved_id}，等待服务端处理，3s 后开始轮询状态…")
        time.sleep(3)

        start_ts = time.time()
        poll_interval = 10
        while True:
            db_cur = self.get_model_db(id=resolved_id)
            status = (db_cur.status or "").lower()

            if status == MODEL_STATUS_READY:
                logger.info(f"模型处理成功：model_id={resolved_id}, status=ready")
                return

            if status == MODEL_STATUS_FAILED:
                logger.error(f"模型处理失败：model_id={resolved_id}, status=failed")
                raise APIError(f"模型处理失败：model_id={resolved_id}, status=failed")

            elapsed = time.time() - start_ts
            if elapsed > timeout_seconds:
                logger.error(f"等待模型就绪超时：model_id={resolved_id}, waited={int(elapsed)}s")
                raise TimeoutError(f"等待模型就绪超时：model_id={resolved_id}, waited={int(elapsed)}s")

            logger.info(f"[Model Upload] id={resolved_id} 已等待 {int(elapsed)}s，当前 status={status}，继续轮询…")
            time.sleep(poll_interval)

    def download(self, *, local_dir: str, model_id: int | None = None, model_name: str | None = None) -> None:
        if (model_id is None) and (not model_name):
            raise ValueError("id or name is required")
        if not local_dir:
            raise ValueError("local_dir is required")

        db_info = self.get_model_db(id=model_id, name=model_name)

        # 判断是否允许下载
        status = (db_info.status or "").lower()
        if status != "ready":
            raise APIError(f"model is not ready for download (current status: {db_info.status})")
        if not (db_info.parquet_index_path or "").strip():
            raise APIError("parquet index path is required and cannot be empty")

        s3_dir_path = db_info.object_storage_path
        if not s3_dir_path or not s3_dir_path.startswith("s3://"):
            raise APIError(f"invalid object_storage_path: {s3_dir_path}")

        sts = self._get_sts()
        s3_client = minio.Minio(
            sts.endpoint,
            access_key=sts.access_key_id,
            secret_key=sts.secret_access_key,
            session_token=sts.session_token,
            secure=False,
        )

        bucket, object_name = parse_s3_path_strict(s3_dir_path)
        if not bucket or not object_name:
            raise APIError(f"invalid s3 path: {s3_dir_path}")

        os.makedirs(local_dir, exist_ok=True)
        download_dir_from_s3_to_local(s3_client, bucket, object_name, local_dir)
