"""Provide configuration for Sphinx."""

# Configuration file for the Sphinx documentation builder.
#
# This file only contains a selection of the most common options. For a full
# list see the documentation:
# https://www.sphinx-doc.org/en/master/usage/configuration.html

import datetime
from functools import partial
from pathlib import Path

from setuptools_scm import get_version


# Check if sphinx-multiversion is installed
use_multiversion = False
try:
    import sphinx_multiversion  # noqa: F401

    use_multiversion = True
except ImportError:
    pass


# -- Path setup --------------------------------------------------------------

PROJECT_ROOT_DIR = Path(__file__).parents[1].resolve()
get_scm_version = partial(get_version, root=PROJECT_ROOT_DIR)

# -- Project information -----------------------------------------------------

github_url = "https://github.com"
github_repo_org = "juaml"
github_repo_name = "julearn"
github_repo_slug = f"{github_repo_org}/{github_repo_name}"
github_repo_url = f"{github_url}/{github_repo_slug}"

project = github_repo_name
author = f"{project} Contributors"
copyright = f"{datetime.date.today().year}, {author}"

# The version along with dev tag
release = get_scm_version(
    version_scheme="guess-next-dev",
    local_scheme="no-local-version",
)

# -- General configuration ---------------------------------------------------

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    # Built-in extensions:
    "sphinx.ext.autodoc",  # include documentation from docstrings
    "sphinx.ext.autosummary",  # generate autodoc summaries
    "sphinx.ext.doctest",  # test snippets in the documentation
    "sphinx.ext.extlinks",  # markup to shorten external links
    "sphinx.ext.intersphinx",  # link to other projects` documentation
    "sphinx.ext.mathjax",  # math support for HTML outputs in Sphinx
    # Third-party extensions:
    "sphinx_gallery.gen_gallery",  # HTML gallery of examples
    "numpydoc",  # support for NumPy style docstrings
    "sphinx_copybutton",  # copy button for code blocks
    "sphinxcontrib.towncrier.ext",  # towncrier fragment support
    "bokeh.sphinxext.bokeh_plot",  # bokeh plot support
]

if use_multiversion:
    extensions.append("sphinx_multiversion")

# Add any paths that contain templates here, relative to this directory.
templates_path = [
    "_templates",
]

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path.
exclude_patterns = [
    "_build",
    "Thumbs.db",
    ".DS_Store",
    "auto_examples/99_docs/*",
]

nitpicky = True

nitpick_ignore_regex = [
    ("py:class", "numpy._typing.*"),
    ("py:obj", "trimboth"),  # python 3.11 error
    ("py:obj", "tmean"),  # python 3.11 error
    ("py:obj", "subclass"),  # python 3.11 error
    ("py:class", "typing.Any"),  # python 3.11 error
    # ('py:class', 'numpy.typing.ArrayLike')
    ("py:obj", "sqlalchemy.engine.Engine"),  # ignore sqlalchemy
    # Sklearn doc issue to be solved in next release
    ("py:class", "pipeline.Pipeline"),
    ("py:class", "sklearn.utils.metadata_routing.MetadataRequest"),
    ("py:class", "julearn.inspect._pipeline.PipelineInspector"),
]


# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#
html_theme = "furo"
html_title = "julearn documentation"
html_logo = "images/julearn_logo_it.png"

# These paths are either relative to html_static_path
# or fully qualified paths (eg. https://...)
html_css_files = [
    "css/custom.css",
]

html_js_files = [
    "js/custom.js",
]

html_use_modindex = False

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = ["_static"]

html_sidebars = {
    "**": [
        "sidebar/scroll-start.html",
        "sidebar/brand.html",
        "sidebar/search.html",
        "sidebar/navigation.html",
        "versions.html",
        "sidebar/ethical-ads.html",
        "sidebar/scroll-end.html",
    ]
}

# -- sphinx.ext.autodoc configuration ----------------------------------------

autoclass_content = "both"
autodoc_typehints = "description"
autodoc_typehints_description_target = "documented"

# -- sphinx.ext.autosummary configuration ------------------------------------

autosummary_generate = True

autodoc_default_options = {
    "imported-members": True,
    "inherited-members": True,
    "undoc-members": True,
    "member-order": "bysource",
    #  We cannot have __init__: it causes duplicated entries
    #  'special-members': '__init__',
}


# -- sphinx.ext.intersphinx configuration ------------------------------------

intersphinx_mapping = {
    "python": ("https://docs.python.org/3/", None),
    "sklearn": ("https://scikit-learn.org/stable", None),
    "nilearn": ("https://nilearn.github.io/stable/", None),
    "nibabel": ("https://nipy.org/nibabel/", None),
    "numpy": ("https://numpy.org/doc/stable/", None),
    "pandas": ("https://pandas.pydata.org/pandas-docs/dev", None),
    # "sqlalchemy": ("https://docs.sqlalchemy.org/en/20/", None),
    "joblib": ("https://joblib.readthedocs.io/en/latest/", None),
    "scipy": ("https://docs.scipy.org/doc/scipy/", None),
    "skopt": ("https://scikit-optimize.readthedocs.io/en/latest", None),
    "optuna": ("https://optuna.readthedocs.io/en/stable", None),
    "optuna_integration": (
        "https://optuna-integration.readthedocs.io/en/stable",
        None,
    ),
}

# -- sphinx.ext.extlinks configuration ---------------------------------------

extlinks = {
    "gh": (f"{github_repo_url}/issues/%s", "#%s"),
}

# -- numpydoc configuration --------------------------------------------------

numpydoc_show_class_members = False
numpydoc_xref_param_type = False
numpydoc_xref_aliases = {
    "Path": "pathlib.Path",
    "Nifti1Image": "nibabel.nifti1.Nifti1Image",
    "Nifti2Image": "nibabel.nifti2.Nifti2Image",
    # "Engine": "sqlalchemy.engine.Engine",
}
numpydoc_xref_ignore = {
    "of",
    "shape",
    "optional",
    "or",
    "the",
    "options",
    "function",
    "object",
    "class",
    "objects",
    "Engine",
    "positive",
    "negative",
    "compatible",
    "TransformerLike",
    "ModelLike",
    "EstimatorLike",
}
# numpydoc_validation_checks = {
#     "all",
#     "GL01",
#     "GL02",
#     "GL03",
#     "ES01",
#     "SA01",
#     "EX01",
# }

# -- Sphinx-Gallery configuration --------------------------------------------

ex_dirs = [
    "00_starting",
    "01_model_comparison",
    "02_inspection",
    "03_complex_models",
    "04_confounds",
    "05_customization",
    "99_docs",
]

example_dirs = []
gallery_dirs = []
for t_dir in ex_dirs:
    example_dirs.append(f"../examples/{t_dir}")
    gallery_dirs.append(f"auto_examples/{t_dir}")

sphinx_gallery_conf = {
    "doc_module": "julearn",
    "backreferences_dir": "api/generated",
    "examples_dirs": example_dirs,
    "gallery_dirs": gallery_dirs,
    "nested_sections": True,
    "filename_pattern": "/(plot|run)_",
    "download_all_examples": False,
}

# -- sphinx-multiversion configuration ---------------------------------------

smv_rebuild_tags = False
smv_tag_whitelist = r"^v\d+\.\d+.\d+$"
smv_branch_whitelist = r"main"
smv_released_pattern = r"^tags/v.*$"

# -- sphinxcontrib-towncrier configuration -----------------------------------

towncrier_draft_autoversion_mode = "draft"
towncrier_draft_include_empty = True
towncrier_draft_working_directory = PROJECT_ROOT_DIR
