# Jaeger Interface

This sub-package ships a **thin synchronous wrapper** around the
[Jaeger Query Service](https://www.jaegertracing.io/docs/) HTTP API so
that you can **search for and retrieve traces** directly from Python
with minimal boilerplate.

> The client relies on `requests` (already included in the SDK’s
> dependencies) and uses *pydantic* for full type-safety.

---

## Installation

`veris-ai` already lists both `requests` and `pydantic` as hard
requirements, so **no additional dependencies are required**.

```bash
pip install veris-ai
```

---

## Quick-start

```python
from veris_ai.jaeger_interface import JaegerClient, SearchQuery
import json
from veris_ai.jaeger_interface.models import Trace
# Replace with the URL of your Jaeger Query Service instance
client = JaegerClient("http://localhost:16686")

# --- 1. Search traces --------------------------------------------------
resp = client.search(
    SearchQuery(
        service="veris-agent",
        limit=3,
        operation="CustomSpanData",
        tags={"bt.metadata.session_id":"oRL1_IhMP3s7T7mYrixCW"}
    )
)

# save to json
with open("resp.json", "w") as f:
    f.write(resp.model_dump_json(indent=2))

# Guard clause
if not resp or not resp.data:
    print("No data found")
    exit(1)

# Print trace ids
for trace in resp.data:
    if isinstance(trace, Trace):
        print("TRACE ID:", trace.traceID, len(trace.spans), "spans")

# --- 2. Retrieve a specific trace -------------------------------------
if isinstance(resp.data, list):
    trace_id = resp.data[0].traceID
else:
    trace_id = resp.data.traceID

detailed = client.get_trace(trace_id)
# save detailed to json
with open("detailed.json", "w") as f:
    f.write(detailed.model_dump_json(indent=2))
```

---

## API Reference

### `JaegerClient`

| Method | Description |
| -------- | ----------- |
| `search(query: SearchQuery) -> SearchResponse` | Search for traces matching the given parameters (wrapper around `/api/traces`). |
| `get_trace(trace_id: str) -> GetTraceResponse` | Fetch a single trace by ID  (wrapper around `/api/traces/{id}`). |

### `SearchQuery`

`SearchQuery` is a *pydantic* model for building the query-string sent to
Jaeger’s `/api/traces` endpoint.

Parameter logic:

* **service** – mandatory, sets the top-level service filter.
* **operation** – optional; a trace is kept only if *any* of its spans has
  this exact `operationName`.
* **tags** – dict of key/value filters; *all* pairs must match on a single
  span (logical AND). Example: `{"error": "true"}`.
* **limit** – applied after all filters and trims the result list.

Any other fields are forwarded untouched, so you can experiment with new
Jaeger parameters without waiting for an SDK update.

---

## Compatibility

The implementation targets **Jaeger v1.x** REST endpoints. For clusters
backed by **OpenSearch** storage the same endpoints apply. Should you
need API v3 support feel free to open an issue or contribution—thanks!

---

## Licence

This package is released under the **MIT license**.