"""Synchronous Jaeger Query Service client built on **requests**.

This implementation keeps dependencies minimal while providing fully-typed
*pydantic* models for both **request** and **response** bodies.
"""

from __future__ import annotations

from typing import TYPE_CHECKING, Self

import requests

from .models import GetTraceResponse, SearchQuery, SearchResponse

if TYPE_CHECKING:
    from types import TracebackType

__all__ = ["JaegerClient"]


class JaegerClient:  # noqa: D101
    def __init__(
        self,
        base_url: str,
        *,
        timeout: float | None = 10.0,
        session: requests.Session | None = None,
        headers: dict[str, str] | None = None,
    ) -> None:
        """Create a new *JaegerClient* instance.

        Args:
            base_url: Base URL of the Jaeger Query Service (e.g. ``http://localhost:16686``).
            timeout: Request timeout in **seconds** (applied to every call).
            session: Optional pre-configured :class:`requests.Session` to reuse.
            headers: Optional default headers to send with every request.
        """
        # Normalise to avoid trailing slash duplicates
        self._base_url = base_url.rstrip("/")
        self._timeout = timeout
        self._external_session = session  # If provided we won't close it
        self._headers = headers or {}

    # ---------------------------------------------------------------------
    # Internal helpers
    # ---------------------------------------------------------------------

    def _make_session(self) -> tuple[requests.Session, bool]:  # noqa: D401
        """Return a *(session, should_close)* tuple.

        If an external session was supplied we **must not** close it after the
        request, hence the boolean flag letting callers know whether they are
        responsible for closing the session.
        """
        if self._external_session is not None:
            return self._external_session, False

        # Reuse the session opened via the context manager if available
        if hasattr(self, "_session_ctx"):
            return self._session_ctx, False

        session = requests.Session()
        session.headers.update(self._headers)
        return session, True

    # ---------------------------------------------------------------------
    # Public API
    # ---------------------------------------------------------------------

    def search(self, query: SearchQuery) -> SearchResponse:  # noqa: D401
        """Search traces using the *v1* ``/api/traces`` endpoint.

        Args:
            query: :class:`~veris_ai.jaeger_interface.models.SearchQuery` instance.

        Returns:
            Parsed :class:`~veris_ai.jaeger_interface.models.SearchResponse` model.
        """
        params = query.to_params()
        session, should_close = self._make_session()
        try:
            url = f"{self._base_url}/api/traces"
            response = session.get(url, params=params, timeout=self._timeout)
            response.raise_for_status()
            data = response.json()
        finally:
            if should_close:
                session.close()
        return SearchResponse.model_validate(data)  # type: ignore[arg-type]

    def get_trace(self, trace_id: str) -> GetTraceResponse:  # noqa: D401
        """Retrieve a single trace by *trace_id*.

        Args:
            trace_id: The Jaeger trace identifier.

        Returns:
            Parsed :class:`~veris_ai.jaeger_interface.models.GetTraceResponse` model.
        """
        if not trace_id:
            error_msg = "trace_id must be non-empty"
            raise ValueError(error_msg)

        session, should_close = self._make_session()
        try:
            url = f"{self._base_url}/api/traces/{trace_id}"
            response = session.get(url, timeout=self._timeout)
            response.raise_for_status()
            data = response.json()
        finally:
            if should_close:
                session.close()
        return GetTraceResponse.model_validate(data)  # type: ignore[arg-type]

    # ------------------------------------------------------------------
    # Context-manager helpers (optional but convenient)
    # ------------------------------------------------------------------

    def __enter__(self) -> Self:
        """Enter the context manager."""
        self._session_ctx, self._should_close_ctx = self._make_session()
        return self

    def __exit__(
        self,
        exc_type: type[BaseException] | None,
        exc: BaseException | None,
        tb: TracebackType | None,
    ) -> None:
        """Exit the context manager."""
        # Only close if we created the session
        if getattr(self, "_should_close_ctx", False):
            self._session_ctx.close()
