from __future__ import annotations

import json
from typing import Any

from pydantic import BaseModel, ConfigDict, Field, field_validator

__all__ = [
    "Tag",
    "Process",
    "Span",
    "Trace",
    "SearchResponse",
    "GetTraceResponse",
    "SearchQuery",
]


class Tag(BaseModel):
    """A Jaeger tag key/value pair."""

    key: str
    value: Any
    type: str | None = None  # Jaeger uses an optional *type* field in v1


class Process(BaseModel):
    """Represents the *process* section of a Jaeger trace."""

    serviceName: str = Field(alias="serviceName")  # noqa: N815
    tags: list[Tag] | None = None


class Span(BaseModel):
    """Represents a single Jaeger span."""

    traceID: str  # noqa: N815
    spanID: str  # noqa: N815
    operationName: str  # noqa: N815
    startTime: int  # noqa: N815
    duration: int
    tags: list[Tag] | None = None
    references: list[dict[str, Any]] | None = None
    processID: str | None = None  # noqa: N815

    model_config = ConfigDict(extra="allow")


class Trace(BaseModel):
    """A full Jaeger trace as returned by the Query API."""

    traceID: str  # noqa: N815
    spans: list[Span]
    process: Process | dict[str, Process] | None = None
    warnings: list[str] | None = None

    model_config = ConfigDict(extra="allow")


class _BaseResponse(BaseModel):
    data: list[Trace] | Trace | None = None
    errors: list[str] | None = None

    # Allow any additional keys returned by Jaeger so that nothing gets
    # silently dropped if the backend adds new fields we don’t know about.

    model_config = ConfigDict(extra="allow")


class SearchResponse(_BaseResponse):
    """Response model for *search* or *find traces* requests."""

    total: int | None = None
    limit: int | None = None


class GetTraceResponse(_BaseResponse):
    """Response model for *get trace by id* requests."""

    # Same as base but alias for clarity


# ---------------------------------------------------------------------------
# Query models
# ---------------------------------------------------------------------------


class SearchQuery(BaseModel):
    """Minimal set of query parameters for the `/api/traces` endpoint.

    Parameter interaction rules:

    * **service** – global filter; *all* returned traces must belong to this
      service.
    * **operation** – optional secondary filter; returned traces must contain
      *at least one span* whose ``operationName`` equals the provided value.
    * **tags** – dictionary of key‒value pairs; each trace must include a span
      that matches **all** of the pairs (logical AND).
    * **limit** – applied *after* all other filters; truncates the final list
      of traces to the requested maximum.

    Any additional/unknown parameters are forwarded thanks to
    ``extra = "allow"`` – this keeps the model future-proof.
    """

    # NOTE: Only the fields that are reliably supported by Jaeger’s REST API and
    # work with the user’s deployment are kept. The model remains *open* to any
    # extra parameters thanks to `extra = "allow"`.

    service: str = Field(
        ...,
        description="Service name to search for. Example: 'veris-agent'",
    )

    limit: int | None = Field(
        None,
        description="Maximum number of traces to return. Example: 10",
    )

    tags: dict[str, Any] | None = Field(
        None,
        description=(
            "Dictionary of tag filters (AND-combined). "
            "Example: {'error': 'true', 'bt.metrics.time_to_first_token': '0.813544'}"
        ),
    )

    operation: str | None = Field(
        None,
        description="Operation name to search for. Example: 'process_chat_message'",
    )

    model_config = ConfigDict(
        extra="allow",  # allow additional query params implicitly
        populate_by_name=True,
        str_to_lower=False,
    )

    @field_validator("tags", mode="before")
    @classmethod
    def _empty_to_none(cls, v: dict[str, Any] | None) -> dict[str, Any] | None:  # noqa: D401, ANN102
        return v or None

    def to_params(self) -> dict[str, Any]:  # noqa: D401
        """Translate the model into a *requests*/*httpx* compatible params dict."""
        # Dump using aliases so ``span_kind`` becomes ``spanKind`` automatically.
        params: dict[str, Any] = self.model_dump(exclude_none=True, by_alias=True)

        # Convert tags to a JSON string if necessary – this matches what the UI sends.
        if "tags" in params and isinstance(params["tags"], dict):
            params["tags"] = json.dumps(params["tags"])

        return params
