#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
This module  exposes exceptions used by refinery.
"""
from __future__ import annotations


class RefineryImportMissing(ModuleNotFoundError):
    """
    A special variant of the `ModuleNotFoundError` exception which is raised when a dependency of a
    refinery unit is not installed in the current environment. The exception also provides hints
    about what package has to be installed in order to make that module available.
    """
    def __init__(self, missing: str, *dependencies: str):
        super().__init__()
        import shlex
        self.missing = missing
        self.install = ' '.join(shlex.quote(dist) for dist in dependencies)
        self.dependencies = dependencies


class MissingModule:
    """
    This class can wrap a module import that is currently missing. If any attribute of the missing
    module is accessed, it raises `refinery.units.RefineryImportMissing`.
    """
    def __init__(self, name, dist=None):
        self.name = name
        self.dist = dist or name

    def __getattr__(self, key: str):
        if key.startswith('__') and key.endswith('__'):
            raise AttributeError(key)
        raise RefineryImportMissing(self.name, self.dist)


class RefineryCriticalException(RuntimeError):
    """
    If this exception is thrown, processing of the entire input stream
    is aborted instead of just aborting the processing of the current
    chunk.
    """
    pass


class RefineryPotentialUserError(RuntimeError):
    """
    This exception can be raised by a unit to inform the user about a
    suspected input error.
    """
    pass


class RefineryException(RuntimeError):
    """
    This is an exception that was not generated by an external library.
    """
    pass
