#   Copyright ETH 2018 - 2023 Zürich, Scientific IT Services
# 
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
# 
#        http://www.apache.org/licenses/LICENSE-2.0
#   
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#
import re
import time

import pytest

from pybis import Openbis
from pybis import ImagingControl


def test_token(openbis_instance):
    assert openbis_instance.token is not None
    assert openbis_instance.is_token_valid(openbis_instance.token) is True
    assert openbis_instance.is_session_active() is True


def http_only():
    with pytest.raises(Exception):
        new_instance = Openbis("http://localhost")
        assert new_instance is None

    new_instance = Openbis(
        url="http://localhost",
        allow_http_but_do_not_use_this_in_production_and_only_within_safe_networks=True,
    )
    assert new_instance is not None


def test_cached_token(other_openbis_instance):
    assert other_openbis_instance.is_token_valid() is True

    other_openbis_instance.logout()
    assert other_openbis_instance.is_token_valid() is False


def test_create_perm_id(openbis_instance):
    permId = openbis_instance.create_permId()
    assert permId is not None
    m = re.search("([0-9]){17}-([0-9]*)", permId)
    ts = m.group(0)
    assert ts is not None
    count = m.group(1)
    assert count is not None


def test_get_samples_update_in_transaction(openbis_instance):
    """
        Update samples in transaction without overriding parents/children
    """
    name_suffix = str(time.time())

    st = openbis_instance.new_sample_type('TEST_YEAST',
                                          'YEAST-',
                                          autoGeneratedCode=True,
                                          description="some new yest type for test")
    st.save()
    st.assign_property('name')

    # Create new space
    space = openbis_instance.new_space(code='space_name' + name_suffix, description='')
    space.save()

    # Create new project
    project = space.new_project(code='project_code' + name_suffix)
    project.save()

    # Create new experiment
    experiment = openbis_instance.new_experiment(
        code='MY_NEW_EXPERIMENT',
        type='DEFAULT_EXPERIMENT',
        project=project.code
    )
    experiment.save()

    # Create parent sample
    sample1 = openbis_instance.new_sample(
        type='TEST_YEAST',
        space=space.code,
        experiment=experiment.identifier,
        parents=[],
        children=[],
        props={"name": "sample1"}
    )
    sample1.save()

    # Create child sample
    sample2 = openbis_instance.new_sample(
        type='TEST_YEAST',
        space=space.code,
        experiment=experiment.identifier,
        parents=[sample1],
        children=[],
        props={"name": "sample2"}
    )
    sample2.save()

    # Verify samples parent/child relationship
    sample1 = openbis_instance.get_sample(
        sample_ident=sample1.identifier,
        space=space.code,
        props="*"
    )
    sample2 = openbis_instance.get_sample(
        sample_ident=sample2.identifier,
        space=space.code,
        props="*"
    )
    assert sample1.children == [sample2.identifier]
    assert sample2.parents == [sample1.identifier]

    trans = openbis_instance.new_transaction()
    # get samples that have parents and update name
    samples = openbis_instance.get_samples(space=space.code, props="*", withParents="*")
    for sample in samples:
        sample.props["name"] = 'new name for sample2'
        trans.add(sample)
    # get samples that have children and update name
    samples = openbis_instance.get_samples(space=space.code, props="*", withChildren="*")
    for sample in samples:
        sample.props["name"] = 'new name for sample1'
        trans.add(sample)
    trans.commit()

    # Verify that name has been changed and parent/child relationship remains
    sample1 = openbis_instance.get_sample(
        sample_ident=sample1.identifier,
        space=space.code,
        props="*"
    )
    sample2 = openbis_instance.get_sample(
        sample_ident=sample2.identifier,
        space=space.code,
        props="*"
    )
    assert sample1.props["name"] == 'new name for sample1'
    assert sample1.children == [sample2.identifier]
    assert sample2.props["name"] == 'new name for sample2'
    assert sample2.parents == [sample1.identifier]

    trans = openbis_instance.new_transaction()
    # get samples with attributes and change name
    samples = openbis_instance.get_samples(space=space.code, attrs=["parents", "children"])
    for sample in samples:
        sample.props["name"] = "default name"
        trans.add(sample)
    trans.commit()

    # Verify that name has been changed and parent/child relationship remains
    sample1 = openbis_instance.get_sample(
        sample_ident=sample1.identifier,
        space=space.code,
        props="*"
    )
    sample2 = openbis_instance.get_sample(
        sample_ident=sample2.identifier,
        space=space.code,
        props="*"
    )
    assert sample1.props["name"] == 'default name'
    assert sample1.children == [sample2.identifier]
    assert sample2.props["name"] == 'default name'
    assert sample2.parents == [sample1.identifier]

    sample3 = openbis_instance.new_sample(
        type='TEST_YEAST',
        space=space.code,
        experiment=experiment.identifier,
        parents=[],
        children=[],
        props={"name": "sample3"}
    )
    sample3.save()

    trans = openbis_instance.new_transaction()
    # get sample1 without attributes and add sample3 as a parent
    samples = openbis_instance.get_samples(space=space.code, identifier=sample1.identifier)
    for sample in samples:
        sample.add_parents([sample3.identifier])
        trans.add(sample)
    # get sample2 without attributes and remove sample1 as a parent
    samples = openbis_instance.get_samples(space=space.code, identifier=sample2.identifier)
    for sample in samples:
        sample.del_parents([sample1.identifier])
        trans.add(sample)
    trans.commit()

    # Verify changes
    sample1 = openbis_instance.get_sample(
        sample_ident=sample1.identifier,
        space=space.code,
        props="*"
    )
    sample2 = openbis_instance.get_sample(
        sample_ident=sample2.identifier,
        space=space.code,
        props="*"
    )
    sample3 = openbis_instance.get_sample(
        sample_ident=sample3.identifier,
        space=space.code,
        props="*"
    )
    assert sample1.children == []
    assert sample1.parents == [sample3.identifier]
    assert sample2.parents == []
    assert sample3.children == [sample1.identifier]


def test_failed_second_login_raises_exception(openbis_instance):
    """
        Logins to openBIS using wrong username/password, PyBIS should raise exception
    """
    assert openbis_instance.is_session_active() is True

    try:
        openbis_instance.login('non_existing_username_for_test', 'abcdef')
        # Login should fail at this point
        assert False
    except ValueError as e:
        assert str(e) == "login to openBIS failed"


def test_set_token_accepts_personal_access_token_object(openbis_instance):
    """
        Verifies that set_token method accepts both permId and PersonalAccessToken object
    """
    assert openbis_instance.is_session_active() is True

    pat = openbis_instance.get_or_create_personal_access_token(sessionName="Project A")

    openbis_instance.set_token(pat, save_token=True)
    openbis_instance.set_token(pat.permId, save_token=True)





def test_openbis():
    def get_instance():
        base_url = "http://localhost:8888/openbis"
        # base_url = "http://local.openbis.ch:8080/openbis"
        # base_url = "https://alaskowski:8443/openbis"
        # base_url = "https://openbis-sis-ci-sprint.ethz.ch/"
        openbis_instance = Openbis(
            url=base_url,
            verify_certificates=False,
            allow_http_but_do_not_use_this_in_production_and_only_within_safe_networks=True
        )
        # openbis_instance.set_token('$pat-test-250123130509616x7A0A9FE0C2CD7D2C9D5F54AFB163828C')
        token = openbis_instance.login('admin', 'changeit')
        # print(token)
        return openbis_instance

    o = get_instance()

    # samples = o.get_samples(['20250120153353271-42'])
    # samples = o.get_samples(collection='/IMAGING/NANONIS/SXM_COLLECTION')
    # samples = o.get_samples(collection='/IMAGING/NANONIS/SXM_COLLECTION', attrs=['parents', 'children'])
    # children = samples.get_children()
    # child = children[1]

    # samples = o.get_samples(attrs=['children', 'parents'])
    # s = samples[8]
    # c = s.children
    # cc = s.get_children()
    # c = s.children
    # samples2 = o.get_samples(collection='/IMAGING/NANONIS/SXM_COLLECTION')
    # samples2 = o.get_samples(collection='/DEFAULT/DEFAULT/DEFAULT')
    # samples2 = samples2[0]

    # children = samples2.get_children()
    # print(samples2)

    # s1 = o.new_sample('IMAGING_SAMPLE', collection='/IMAGING/NANONIS/SXM_COLLECTION')
    # s1.props['DEFAULT_OBJECT_VIEW'.lower()] = 'IMAGING_GALLERY_VIEW'
    # s1.props['name'] = 'LH021-SPM-meas-sub'
    # s1.save()

    # sample = o.get_sample('20240820130458427-43')
    #
    # dataset = o.new_dataset(
    #     type="RAW_DATA",
    #     sample='/DEFAULT/DEFAULT/DEFAULT',
    #     files=['/home/alaskowski/Downloads/test'],
    #     props={"$name": "file comma test"},
    # )
    # dataset.save()

    # dataset = o.get_dataset('20241018081822706-77')
    #
    # ds_new = o.new_dataset(type="RAW_DATA",
    #                        sample='/DEFAULT/DEFAULT/DEFAULT',
    #                        files=['/home/alaskowski/Downloads/test'],
    #                        props={"$name": "some good name"},
    #                        )
    # ds_new.save()
    # dataset = ds_new

    # dataset = o.get_dataset('20241127142823523-96')
    # # print(dataset.file_list)
    # dataset.download(destination='/home/alaskowski/Downloads/test/test2', create_default_folders = False, wait_until_finished = False)
    # vocab = o.get_vocabulary('MY_VOCAB')




    import json
    # sample = o.get_sample('/ELN_SETTINGS/GENERAL_ELN_SETTINGS')
    # j = json.loads(sample.props['$eln_settings'])
    # j['instanceSettings']['minBarcodeLength'] = '13'
    # sample.props['$eln_settings'] = json.dumps(j)
    # sample.save()



    # conf = {
    #     'code': "test",
    #     'label': 'label',
    #     'description': 'desc',
    #     'dataType': 'SAMPLE',
    #     'multiValue': True
    # }
    #
    # prop = o.new_property_type(**conf)
    # prop.save()
    # print(prop)

    # ic = ImagingControl(o)
    #
    # from pybis import ImagingDataSetPreview
    #
    # # preview_config = ImagingDataSetPreview("png")
    #
    # # ic.make_preview('20241129123539459-38', 0, preview_config)
    #
    # # ic.export_image('20241129123539459-38', 0, '/home/alaskowski/pendrive/PREMISE')
    # #
    # ic.export_previews(permIds, [0 for _ in permIds], [0 for _ in permIds],
    #                        '/home/alaskowski/pendrive/PREMISE')


    # sample = o.new_sample('EXPERIMENTAL_STEP', collection='/DEFAULT/DEFAULT/DEFAULT')
    # # sample = o.get_sample('20250109094351184-53')
    # sample = o.get_sample('20250128125335843-49')
    # spreadsheet = o.new_spreadsheet(columns=10, rows=15)
    # spreadsheet = sample.props['experimental_step.spreadsheet']
    # cell = spreadsheet.cell(2,2)
    #
    # column = spreadsheet.column(1)
    # df = column.df
    # spreadsheet.data[0][0] = '1'
    # spreadsheet.data[1][0] = '2'
    # spreadsheet.data[2][0] = '3'
    # spreadsheet['A4'] = '=SUM(A1:A3)'
    # spreadsheet.values[0][4] = '3'
    # spreadsheet['OPENBIS1', 1] = 'test'
    # spreadsheet.width[0] = 150
    #
    # spreadsheet.add_column('OPENBIS')
    # spreadsheet.width[10] = 150
    # spreadsheet.add_row()
    #
    #
    # sample.props['experimental_step.spreadsheet'] = spreadsheet
    # sample.save()

    # experiment = o.get_experiment('20250117123121422-31')
    #
    # # sample = o.get_samples('20250120153353271-42', attrs=['children'])
    # objects = o.get_objects(attrs=['parents'], withParents=['/IMAGING/NANONIS/IMG4', '/IMAGING/NANONIS/IMG3'])

    # parents = o.get_datasets(experiment='20250117123121422-31')

    # ds_new = o.new_dataset(
    #     type="RAW_DATA",
    #     experiment=experiment,
    #     object=sample[0].children[0],
    #     files=['/home/alaskowski/Downloads/test'],
    #     props={"name": 'new_file_name'},
    # )
    # ds_new.set_parents(parents[0])
    # ds_new.save()


    print("DONE")




